# @metcoder95/https-pem

> This is a fork of the original [https-pem](https://github.com/watson/https-pem) package, modified to work with Node.js 24 and above.

Self-signed PEM key and certificate ready for use in your HTTPS server.

A dead simple way to get an HTTPS server running in development with no
need to generate the self signed PEM key and certificate.

[![CI](https://github.com/metcoder95/https-pem/actions/workflows/ci.yml/badge.svg)](https://github.com/metcoder95/https-pem/actions/workflows/ci.yml)
[![js-standard-style](https://img.shields.io/badge/code%20style-standard-brightgreen.svg?style=flat)](https://github.com/feross/standard)

## Installation

```
npm install @metcoder95/https-pem
```

**Warning:** Upon installation a private key and a self signed
certificate will be generated inside `./node_modules/https-pem`. The
certificate is valid for 365 days and no attempt have been made to make
this secure in any way. I suggest only using this for testing and
development where you just need an easy and quick way to run an HTTPS
server with Node.js.

> **Note**: for Node.js 24 and above, the key size has been increased to 2048 bits due to changes in OpenSSL. For earlier versions, the default key size is used.

## Example Usage

```js
const https = require('node:https');
const pem = require('@metcoder95/https-pem');

const server = https.createServer(pem, function (req, res) {
  res.end('This is servered over HTTPS');
});

server.listen(443, function () {
  console.log('The server is running on https://localhost');
});
```

### Connecting

When connecting to an HTTPS server from Node.js that uses a self-signed
certificate, `https.request` will normally emit an `error` and refuse to
complete the request. To get around that simply set the
`rejectUnauthorized` option to `false`:

```js
const opts = { rejectUnauthorized: false };

const req = https.request(opts, function (res) {
  // ...
});

req.end();
```

If using `curl` to connect to a Node.js HTTPS server using a
self-signed certificate, use the `-k` option:

```
curl -k https://localhost:443
```

If you want to use the `fetch` API with the self-signed certificate, you can do so by using the `undici` package:

```js
const https = require('node:https');
const { Agent } = require('undici');
const pem = require('@metcoder95/https-pem');

const agent = new Agent({
  connect: {
    rejectUnauthorized: false,
  },
});

const server = https.createServer(pem, function (req, res) {
  res.end('This is served over HTTPS with fetch');
});

server.listen(443, function () {
  console.log('The server is running on https://localhost');
  fetch('https://localhost', { dispatcher: agent })
    .then((res) => res.text())
    .then((body) => {
      console.log('Response from server:', body);
    })
    .catch((err) => {
      console.error('Error fetching from server:', err);
    });
});
```

## API

The `https-pem` module simply exposes an object with two properties:
`key` and `cert`.

### `pem.generate()`

Generates a new self-signed certificate and key pair.

```js
const pem = require('@metcoder95/https-pem');
pem.generate();
```

#### Parameters

- `HttpPEMGeneratorOptions`: An object that can be passed to the `selfsigned.generate` method. This allows you to customize the generation of the key and certificate, such as setting the key size or adding custom attributes.
  - `attr`: (optional) An array of attributes to include in the certificate. See the [selfsigned](https://github.com/jfromaniello/selfsigned#attributes) documentation for more information.
  - `opts`: (optional) An object with options for the `selfsigned.generate` method. See the [selfsigned](https://github.com/jfromaniello/selfsigned#attributes) documentation for available options.
- `done`: (optional) A callback function that will be called with the generated key and certificate. If not provided, the function will return a promise that resolves with the generated key and certificate.

#### Returns

- HttpsPEMGenerateResult: An object containing the generated key and certificate.
  - `key`: The generated private key in PEM format.
  - `cert`: The generated self-signed certificate in PEM format.

### `pem.key`

The autogenerated private key (RSA).

### `pem.cert`

The autogenerated self-signed certificate.

## License

MIT
