// SPDX-License-Identifier: Apache-2.0
// Copyright (C) 2022-2025 Advanced Micro Devices, Inc. All rights reserved

#ifndef AIE_DEBUG_METADATA_H
#define AIE_DEBUG_METADATA_H

#include <boost/property_tree/ptree.hpp>
#include <memory>
#include <vector>
#include <set>

#include "core/common/device.h"
#include "core/common/message.h"
#include "core/include/xrt/xrt_hw_context.h"
#include "xdp/config.h"
#include "xdp/profile/database/static_info/aie_constructs.h"
#include "xdp/profile/database/static_info/aie_util.h"
#include "xdp/profile/database/static_info/filetypes/base_filetype_impl.h"
#include "xdp/profile/plugin/aie_profile/aie_profile_defs.h"
#include "xdp/profile/plugin/aie_base/generations/aie_generations.h"
#include "xdp/profile/plugin/aie_base/generations/aie1_registers.h"
#include "xdp/profile/plugin/aie_base/generations/aie2_registers.h"
#include "xdp/profile/plugin/aie_base/generations/aie2ps_registers.h"
#include "xdp/profile/plugin/vp_base/vp_base_plugin.h"

extern "C" {
#include <xaiengine.h>
#include <xaiengine/xaiegbl_params.h>
}

namespace xdp {

// Forward declarations
class BaseReadableTile;
class UsedRegisters;

class AieDebugMetadata {
  public:
    AieDebugMetadata(uint64_t deviceID, void* handle);

    void parseMetrics();

    module_type getModuleType(int mod) {return moduleTypes[mod];}
    uint64_t getDeviceID() {return deviceID;}
    void* getHandle() {return handle;}

    std::map<tile_type, std::string> getConfigMetrics(const int module) {
      return configMetrics[module];
    }
    std::vector<std::pair<tile_type, std::string>> getConfigMetricsVec(const int module) {
      return {configMetrics[module].begin(), configMetrics[module].end()};
    }

    std::map<module_type, std::vector<uint64_t>>& getRegisterValues() {
      return parsedRegValues;
    }
    
    bool aieMetadataEmpty() const {return (metadataReader == nullptr);}
    xdp::aie::driver_config getAIEConfigMetadata() {return metadataReader->getDriverConfig();}

    uint8_t getAIETileRowOffset() const {
      return (metadataReader == nullptr) ? 0 : metadataReader->getAIETileRowOffset();
    }
    int getHardwareGen() const {
      return (metadataReader == nullptr) ? 0 : metadataReader->getHardwareGeneration();
    }

    int getNumModules() {return NUM_MODULES;}
    xrt::hw_context getHwContext() {return hwContext;}
    void setHwContext(xrt::hw_context c) {
      hwContext = std::move(c);
    }
 
    std::string lookupRegisterName(uint64_t regVal);
    uint64_t lookupRegisterAddr(std::string regName);

  private:
    std::vector<uint64_t> stringToRegList(std::string stringEntry, module_type t);
    std::vector<std::string> getSettingsVector(std::string settingsString);

  private:
    // Currently supporting Core, Memory, Interface Tiles, and Memory Tiles
    static constexpr int NUM_MODULES = 4;
    const std::vector<std::string> moduleNames =
      {"aie", "aie_memory", "interface_tile", "memory_tile"};
    const module_type moduleTypes[NUM_MODULES] =
      {module_type::core, module_type::dma, module_type::shim, module_type::mem_tile};

    void* handle;
    uint64_t deviceID;
    xrt::hw_context hwContext;
    std::vector<std::map<tile_type, std::string>> configMetrics;
    std::map<module_type, std::vector<uint64_t>> parsedRegValues;
    const aie::BaseFiletypeImpl* metadataReader = nullptr;

    // List of AIE HW generation-specific registers
    std::unique_ptr<UsedRegisters> usedRegisters;
};

/*****************************************************************
The BaseReadableTile is created to simplify the retrieving of value at
each tile. This class encapsulates all the data (row, col, list of registers
to read) pertaining to a particuar tile, for easy extraction of tile by tile data.
****************************************************************** */
class BaseReadableTile {
  public:
    virtual void readValues(XAie_DevInst* aieDevInst)=0;

    void setTileOffset(uint64_t offset) {tileOffset = offset;}
    void addOffsetName(uint64_t rel, std::string name) {
      relativeOffsets.push_back(rel);
      registerNames.push_back(name);
    }

    void printValues(uint32_t deviceID, VPDatabase* db) {
      int i = 0;
      for (auto& offset : relativeOffsets) {
        db->getDynamicInfo().addAIEDebugSample(deviceID, col, row, values[i],
                                               offset, registerNames[i]);
        i++;
      }
    }

  public:
    uint8_t col;
    uint8_t row;
    uint64_t tileOffset;
    std::vector<uint32_t> values;
    std::vector<uint64_t> relativeOffsets;
    std::vector<std::string> registerNames;
};

/*************************************************************************************
The class UsedRegisters is what gives us AIE hw generation specific data. The base class
has virtual functions which populate the correct registers and their addresses according
to the AIE hw generation in the derived classes. Thus we can dynamically populate the
correct registers and their addresses at runtime.
**************************************************************************************/
class UsedRegisters {
  public:
    UsedRegisters() {
      populateRegNameToValueMap();
      populateRegValueToNameMap();
    }
    virtual ~UsedRegisters() {
      core_addresses.clear();
      memory_addresses.clear();
      interface_addresses.clear();
      memory_tile_addresses.clear();
      regNameToValues.clear();
      regValueToName.clear();
    }

    std::set<uint64_t> getCoreAddresses() {
      return core_addresses;
    }
    std::set<uint64_t> getMemoryAddresses() {
      return memory_addresses;
    }
    std::set<uint64_t> getInterfaceAddresses() {
      return interface_addresses;
    }
    std::set<uint64_t> getMemoryTileAddresses() {
      return memory_tile_addresses;
    }

    std::string getRegisterName(uint64_t regVal) {
      auto itr = regValueToName.find(regVal);
      return (itr != regValueToName.end()) ? itr->second : "";
    }
    uint64_t getRegisterAddr(std::string regName) {
      auto itr = regNameToValues.find(regName);
      return (itr != regNameToValues.end()) ? itr->second : 0;
    }

    virtual void populateProfileRegisters() {};
    virtual void populateTraceRegisters() {};
    virtual void populateRegNameToValueMap() {};
    virtual void populateRegValueToNameMap() {};
    
    void populateAllRegisters() {
      populateProfileRegisters();
      populateTraceRegisters();
    }

  protected:
    std::set<uint64_t> core_addresses;
    std::set<uint64_t> memory_addresses;
    std::set<uint64_t> interface_addresses;
    std::set<uint64_t> memory_tile_addresses;
    std::map<std::string, uint64_t> regNameToValues;
    std::map<uint64_t, std::string> regValueToName;
};

/*************************************************************************************
 AIE1 Registers
 *************************************************************************************/
class AIE1UsedRegisters : public UsedRegisters {
public:
  AIE1UsedRegisters() {
    populateRegNameToValueMap();
    populateRegValueToNameMap();
  }
  ~AIE1UsedRegisters() = default;

  void populateProfileRegisters() {
    // Core modules
    core_addresses.emplace(aie1::cm_performance_control0);
    core_addresses.emplace(aie1::cm_performance_control1);
    core_addresses.emplace(aie1::cm_performance_control2);
    core_addresses.emplace(aie1::cm_performance_counter0);
    core_addresses.emplace(aie1::cm_performance_counter1);
    core_addresses.emplace(aie1::cm_performance_counter2);
    core_addresses.emplace(aie1::cm_performance_counter3);
    core_addresses.emplace(aie1::cm_performance_counter0_event_value);
    core_addresses.emplace(aie1::cm_performance_counter1_event_value);
    core_addresses.emplace(aie1::cm_performance_counter2_event_value);
    core_addresses.emplace(aie1::cm_performance_counter3_event_value);
    core_addresses.emplace(aie1::cm_ecc_scrubbing_event);

    // Memory modules
    memory_addresses.emplace(aie1::mm_performance_control0);
    memory_addresses.emplace(aie1::mm_performance_control1);
    memory_addresses.emplace(aie1::mm_performance_counter0);
    memory_addresses.emplace(aie1::mm_performance_counter1);
    memory_addresses.emplace(aie1::mm_performance_counter0_event_value);
    memory_addresses.emplace(aie1::mm_performance_counter1_event_value);

    // Interface tiles
    interface_addresses.emplace(aie1::shim_performance_control0);
    interface_addresses.emplace(aie1::shim_performance_control1);
    interface_addresses.emplace(aie1::shim_performance_counter0);
    interface_addresses.emplace(aie1::shim_performance_counter1);
    interface_addresses.emplace(aie1::shim_performance_counter0_event_value);
    interface_addresses.emplace(aie1::shim_performance_counter1_event_value);

    // Memory tiles
    // NOTE, not available on AIE1
  }
  
  void populateTraceRegisters() {
    // Core modules
    core_addresses.emplace(aie1::cm_core_status);
    core_addresses.emplace(aie1::cm_trace_control0);
    core_addresses.emplace(aie1::cm_trace_control1);
    core_addresses.emplace(aie1::cm_trace_status);
    core_addresses.emplace(aie1::cm_trace_event0);
    core_addresses.emplace(aie1::cm_trace_event1);
    core_addresses.emplace(aie1::cm_event_status0);
    core_addresses.emplace(aie1::cm_event_status1);
    core_addresses.emplace(aie1::cm_event_status2);
    core_addresses.emplace(aie1::cm_event_status3);
    core_addresses.emplace(aie1::cm_event_broadcast0);
    core_addresses.emplace(aie1::cm_event_broadcast1);
    core_addresses.emplace(aie1::cm_event_broadcast2);
    core_addresses.emplace(aie1::cm_event_broadcast3);
    core_addresses.emplace(aie1::cm_event_broadcast4);
    core_addresses.emplace(aie1::cm_event_broadcast5);
    core_addresses.emplace(aie1::cm_event_broadcast6);
    core_addresses.emplace(aie1::cm_event_broadcast7);
    core_addresses.emplace(aie1::cm_event_broadcast8);
    core_addresses.emplace(aie1::cm_event_broadcast9);
    core_addresses.emplace(aie1::cm_event_broadcast10);
    core_addresses.emplace(aie1::cm_event_broadcast11);
    core_addresses.emplace(aie1::cm_event_broadcast12);
    core_addresses.emplace(aie1::cm_event_broadcast13);
    core_addresses.emplace(aie1::cm_event_broadcast14);
    core_addresses.emplace(aie1::cm_event_broadcast15);
    core_addresses.emplace(aie1::cm_event_broadcast_block_south_value);
    core_addresses.emplace(aie1::cm_event_broadcast_block_west_value);
    core_addresses.emplace(aie1::cm_event_broadcast_block_north_value);
    core_addresses.emplace(aie1::cm_event_broadcast_block_east_value);
    core_addresses.emplace(aie1::cm_timer_trig_event_low_value);
    core_addresses.emplace(aie1::cm_timer_trig_event_high_value);
    core_addresses.emplace(aie1::cm_timer_low);
    core_addresses.emplace(aie1::cm_timer_high);
    core_addresses.emplace(aie1::cm_stream_switch_event_port_selection_0);
    core_addresses.emplace(aie1::cm_stream_switch_event_port_selection_1);

    // Memory modules
    memory_addresses.emplace(aie1::mm_trace_control0);
    memory_addresses.emplace(aie1::mm_trace_control1);
    memory_addresses.emplace(aie1::mm_trace_status);
    memory_addresses.emplace(aie1::mm_trace_event0);
    memory_addresses.emplace(aie1::mm_trace_event1);
    memory_addresses.emplace(aie1::mm_event_status0);
    memory_addresses.emplace(aie1::mm_event_status1);
    memory_addresses.emplace(aie1::mm_event_status2);
    memory_addresses.emplace(aie1::mm_event_status3);
    memory_addresses.emplace(aie1::mm_event_broadcast0);
    memory_addresses.emplace(aie1::mm_event_broadcast1);
    memory_addresses.emplace(aie1::mm_event_broadcast2);
    memory_addresses.emplace(aie1::mm_event_broadcast3);
    memory_addresses.emplace(aie1::mm_event_broadcast4);
    memory_addresses.emplace(aie1::mm_event_broadcast5);
    memory_addresses.emplace(aie1::mm_event_broadcast6);
    memory_addresses.emplace(aie1::mm_event_broadcast7);
    memory_addresses.emplace(aie1::mm_event_broadcast8);
    memory_addresses.emplace(aie1::mm_event_broadcast9);
    memory_addresses.emplace(aie1::mm_event_broadcast10);
    memory_addresses.emplace(aie1::mm_event_broadcast11);
    memory_addresses.emplace(aie1::mm_event_broadcast12);
    memory_addresses.emplace(aie1::mm_event_broadcast13);
    memory_addresses.emplace(aie1::mm_event_broadcast14);
    memory_addresses.emplace(aie1::mm_event_broadcast15);

    // Interface tiles
    interface_addresses.emplace(aie1::shim_trace_control0);
    interface_addresses.emplace(aie1::shim_trace_control1);
    interface_addresses.emplace(aie1::shim_trace_status);
    interface_addresses.emplace(aie1::shim_trace_event0);
    interface_addresses.emplace(aie1::shim_trace_event1);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_0);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_1);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_2);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_3);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_4);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_5);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_6);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_7);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_8);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_9);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_10);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_11);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_12);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_13);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_14);
    interface_addresses.emplace(aie1::shim_event_broadcast_a_15);
    interface_addresses.emplace(aie1::shim_event_status0);
    interface_addresses.emplace(aie1::shim_event_status1);
    interface_addresses.emplace(aie1::shim_event_status2);
    interface_addresses.emplace(aie1::shim_event_status3);
    interface_addresses.emplace(aie1::shim_stream_switch_event_port_selection_0);
    interface_addresses.emplace(aie1::shim_stream_switch_event_port_selection_1);

    // Memory tiles
    // NOTE, not available on AIE1
  }

  void populateRegNameToValueMap() {
    regNameToValues = {
      {"cm_program_counter", aie1::cm_program_counter},
      {"cm_md0", aie1::cm_md0},
      {"cm_md1", aie1::cm_md1},
      {"cm_mc0", aie1::cm_mc0},
      {"cm_mc1", aie1::cm_mc1},
      {"cm_performance_control0", aie1::cm_performance_control0},
      {"cm_performance_control1", aie1::cm_performance_control1},
      {"cm_performance_control2", aie1::cm_performance_control2},
      {"cm_performance_counter0", aie1::cm_performance_counter0},
      {"cm_performance_counter1", aie1::cm_performance_counter1},
      {"cm_performance_counter2", aie1::cm_performance_counter2},
      {"cm_performance_counter3", aie1::cm_performance_counter3},
      {"cm_performance_counter0_event_value", aie1::cm_performance_counter0_event_value},
      {"cm_performance_counter1_event_value", aie1::cm_performance_counter1_event_value},
      {"cm_performance_counter2_event_value", aie1::cm_performance_counter2_event_value},
      {"cm_performance_counter3_event_value", aie1::cm_performance_counter3_event_value},
      {"cm_core_control", aie1::cm_core_control},
      {"cm_core_status", aie1::cm_core_status},
      {"cm_enable_events", aie1::cm_enable_events},
      {"cm_reset_event", aie1::cm_reset_event},
      {"cm_debug_control0", aie1::cm_debug_control0},
      {"cm_debug_control1", aie1::cm_debug_control1},
      {"cm_debug_control2", aie1::cm_debug_control2},
      {"cm_debug_status", aie1::cm_debug_status},
      {"cm_pc_event0", aie1::cm_pc_event0},
      {"cm_pc_event1", aie1::cm_pc_event1},
      {"cm_pc_event2", aie1::cm_pc_event2},
      {"cm_pc_event3", aie1::cm_pc_event3},
      {"cm_error_halt_control", aie1::cm_error_halt_control},
      {"cm_error_halt_event", aie1::cm_error_halt_event},
      {"cm_ecc_scrubbing_event", aie1::cm_ecc_scrubbing_event},
      {"cm_timer_control", aie1::cm_timer_control},
      {"cm_event_generate", aie1::cm_event_generate},
      {"cm_event_broadcast0", aie1::cm_event_broadcast0},
      {"cm_event_broadcast1", aie1::cm_event_broadcast1},
      {"cm_event_broadcast2", aie1::cm_event_broadcast2},
      {"cm_event_broadcast3", aie1::cm_event_broadcast3},
      {"cm_event_broadcast4", aie1::cm_event_broadcast4},
      {"cm_event_broadcast5", aie1::cm_event_broadcast5},
      {"cm_event_broadcast6", aie1::cm_event_broadcast6},
      {"cm_event_broadcast7", aie1::cm_event_broadcast7},
      {"cm_event_broadcast8", aie1::cm_event_broadcast8},
      {"cm_event_broadcast9", aie1::cm_event_broadcast9},
      {"cm_event_broadcast10", aie1::cm_event_broadcast10},
      {"cm_event_broadcast11", aie1::cm_event_broadcast11},
      {"cm_event_broadcast12", aie1::cm_event_broadcast12},
      {"cm_event_broadcast13", aie1::cm_event_broadcast13},
      {"cm_event_broadcast14", aie1::cm_event_broadcast14},
      {"cm_event_broadcast15", aie1::cm_event_broadcast15},
      {"cm_event_broadcast_block_south_set", aie1::cm_event_broadcast_block_south_set},
      {"cm_event_broadcast_block_south_clr", aie1::cm_event_broadcast_block_south_clr},
      {"cm_event_broadcast_block_south_value", aie1::cm_event_broadcast_block_south_value},
      {"cm_event_broadcast_block_west_set", aie1::cm_event_broadcast_block_west_set},
      {"cm_event_broadcast_block_west_clr", aie1::cm_event_broadcast_block_west_clr},
      {"cm_event_broadcast_block_west_value", aie1::cm_event_broadcast_block_west_value},
      {"cm_event_broadcast_block_north_set", aie1::cm_event_broadcast_block_north_set},
      {"cm_event_broadcast_block_north_clr", aie1::cm_event_broadcast_block_north_clr},
      {"cm_event_broadcast_block_north_value", aie1::cm_event_broadcast_block_north_value},
      {"cm_event_broadcast_block_east_set", aie1::cm_event_broadcast_block_east_set},
      {"cm_event_broadcast_block_east_clr", aie1::cm_event_broadcast_block_east_clr},
      {"cm_event_broadcast_block_east_value", aie1::cm_event_broadcast_block_east_value},
      {"cm_trace_control0", aie1::cm_trace_control0},
      {"cm_trace_control1", aie1::cm_trace_control1},
      {"cm_trace_status", aie1::cm_trace_status},
      {"cm_trace_event0", aie1::cm_trace_event0},
      {"cm_trace_event1", aie1::cm_trace_event1},
      {"cm_timer_trig_event_low_value", aie1::cm_timer_trig_event_low_value},
      {"cm_timer_trig_event_high_value", aie1::cm_timer_trig_event_high_value},
      {"cm_timer_low", aie1::cm_timer_low},
      {"cm_timer_high", aie1::cm_timer_high},
      {"cm_event_status0", aie1::cm_event_status0},
      {"cm_event_status1", aie1::cm_event_status1},
      {"cm_event_status2", aie1::cm_event_status2},
      {"cm_event_status3", aie1::cm_event_status3},
      {"cm_combo_event_inputs", aie1::cm_combo_event_inputs},
      {"cm_combo_event_control", aie1::cm_combo_event_control},
      {"cm_event_group_0_enable", aie1::cm_event_group_0_enable},
      {"cm_event_group_pc_enable", aie1::cm_event_group_pc_enable},
      {"cm_event_group_core_stall_enable", aie1::cm_event_group_core_stall_enable},
      {"cm_event_group_core_program_flow_enable", aie1::cm_event_group_core_program_flow_enable},
      {"cm_event_group_errors0_enable", aie1::cm_event_group_errors0_enable},
      {"cm_event_group_errors1_enable", aie1::cm_event_group_errors1_enable},
      {"cm_event_group_stream_switch_enable", aie1::cm_event_group_stream_switch_enable},
      {"cm_event_group_broadcast_enable", aie1::cm_event_group_broadcast_enable},
      {"cm_event_group_user_event_enable", aie1::cm_event_group_user_event_enable},
      {"cm_tile_control", aie1::cm_tile_control},
      {"cm_tile_control_packet_handler_status", aie1::cm_tile_control_packet_handler_status},
      {"cm_tile_clock_control", aie1::cm_tile_clock_control},
      {"cm_stream_switch_master_config_aie_core0", aie1::cm_stream_switch_master_config_aie_core0},
      {"cm_stream_switch_master_config_aie_core1", aie1::cm_stream_switch_master_config_aie_core1},
      {"cm_stream_switch_master_config_dma0", aie1::cm_stream_switch_master_config_dma0},
      {"cm_stream_switch_master_config_dma1", aie1::cm_stream_switch_master_config_dma1},
      {"cm_stream_switch_master_config_tile_ctrl", aie1::cm_stream_switch_master_config_tile_ctrl},
      {"cm_stream_switch_master_config_fifo0", aie1::cm_stream_switch_master_config_fifo0},
      {"cm_stream_switch_master_config_fifo1", aie1::cm_stream_switch_master_config_fifo1},
      {"cm_stream_switch_master_config_south0", aie1::cm_stream_switch_master_config_south0},
      {"cm_stream_switch_master_config_south1", aie1::cm_stream_switch_master_config_south1},
      {"cm_stream_switch_master_config_south2", aie1::cm_stream_switch_master_config_south2},
      {"cm_stream_switch_master_config_south3", aie1::cm_stream_switch_master_config_south3},
      {"cm_stream_switch_master_config_west0", aie1::cm_stream_switch_master_config_west0},
      {"cm_stream_switch_master_config_west1", aie1::cm_stream_switch_master_config_west1},
      {"cm_stream_switch_master_config_west2", aie1::cm_stream_switch_master_config_west2},
      {"cm_stream_switch_master_config_west3", aie1::cm_stream_switch_master_config_west3},
      {"cm_stream_switch_master_config_north0", aie1::cm_stream_switch_master_config_north0},
      {"cm_stream_switch_master_config_north1", aie1::cm_stream_switch_master_config_north1},
      {"cm_stream_switch_master_config_north2", aie1::cm_stream_switch_master_config_north2},
      {"cm_stream_switch_master_config_north3", aie1::cm_stream_switch_master_config_north3},
      {"cm_stream_switch_master_config_north4", aie1::cm_stream_switch_master_config_north4},
      {"cm_stream_switch_master_config_north5", aie1::cm_stream_switch_master_config_north5},
      {"cm_stream_switch_master_config_east0", aie1::cm_stream_switch_master_config_east0},
      {"cm_stream_switch_master_config_east1", aie1::cm_stream_switch_master_config_east1},
      {"cm_stream_switch_master_config_east2", aie1::cm_stream_switch_master_config_east2},
      {"cm_stream_switch_master_config_east3", aie1::cm_stream_switch_master_config_east3},
      {"cm_stream_switch_slave_config_aie_core0", aie1::cm_stream_switch_slave_config_aie_core0},
      {"cm_stream_switch_slave_config_aie_core1", aie1::cm_stream_switch_slave_config_aie_core1},
      {"cm_stream_switch_slave_config_dma_0", aie1::cm_stream_switch_slave_config_dma_0},
      {"cm_stream_switch_slave_config_dma_1", aie1::cm_stream_switch_slave_config_dma_1},
      {"cm_stream_switch_slave_config_tile_ctrl", aie1::cm_stream_switch_slave_config_tile_ctrl},
      {"cm_stream_switch_slave_config_fifo_0", aie1::cm_stream_switch_slave_config_fifo_0},
      {"cm_stream_switch_slave_config_fifo_1", aie1::cm_stream_switch_slave_config_fifo_1},
      {"cm_stream_switch_slave_config_south_0", aie1::cm_stream_switch_slave_config_south_0},
      {"cm_stream_switch_slave_config_south_1", aie1::cm_stream_switch_slave_config_south_1},
      {"cm_stream_switch_slave_config_south_2", aie1::cm_stream_switch_slave_config_south_2},
      {"cm_stream_switch_slave_config_south_3", aie1::cm_stream_switch_slave_config_south_3},
      {"cm_stream_switch_slave_config_south_4", aie1::cm_stream_switch_slave_config_south_4},
      {"cm_stream_switch_slave_config_south_5", aie1::cm_stream_switch_slave_config_south_5},
      {"cm_stream_switch_slave_config_west_0", aie1::cm_stream_switch_slave_config_west_0},
      {"cm_stream_switch_slave_config_west_1", aie1::cm_stream_switch_slave_config_west_1},
      {"cm_stream_switch_slave_config_west_2", aie1::cm_stream_switch_slave_config_west_2},
      {"cm_stream_switch_slave_config_west_3", aie1::cm_stream_switch_slave_config_west_3},
      {"cm_stream_switch_slave_config_north_0", aie1::cm_stream_switch_slave_config_north_0},
      {"cm_stream_switch_slave_config_north_1", aie1::cm_stream_switch_slave_config_north_1},
      {"cm_stream_switch_slave_config_north_2", aie1::cm_stream_switch_slave_config_north_2},
      {"cm_stream_switch_slave_config_north_3", aie1::cm_stream_switch_slave_config_north_3},
      {"cm_stream_switch_slave_config_east_0", aie1::cm_stream_switch_slave_config_east_0},
      {"cm_stream_switch_slave_config_east_1", aie1::cm_stream_switch_slave_config_east_1},
      {"cm_stream_switch_slave_config_east_2", aie1::cm_stream_switch_slave_config_east_2},
      {"cm_stream_switch_slave_config_east_3", aie1::cm_stream_switch_slave_config_east_3},
      {"cm_stream_switch_slave_config_aie_trace", aie1::cm_stream_switch_slave_config_aie_trace},
      {"cm_stream_switch_slave_config_mem_trace", aie1::cm_stream_switch_slave_config_mem_trace},
      {"cm_stream_switch_slave_aie_core0_slot0", aie1::cm_stream_switch_slave_aie_core0_slot0},
      {"cm_stream_switch_slave_aie_core0_slot1", aie1::cm_stream_switch_slave_aie_core0_slot1},
      {"cm_stream_switch_slave_aie_core0_slot2", aie1::cm_stream_switch_slave_aie_core0_slot2},
      {"cm_stream_switch_slave_aie_core0_slot3", aie1::cm_stream_switch_slave_aie_core0_slot3},
      {"cm_stream_switch_slave_aie_core1_slot0", aie1::cm_stream_switch_slave_aie_core1_slot0},
      {"cm_stream_switch_slave_aie_core1_slot1", aie1::cm_stream_switch_slave_aie_core1_slot1},
      {"cm_stream_switch_slave_aie_core1_slot2", aie1::cm_stream_switch_slave_aie_core1_slot2},
      {"cm_stream_switch_slave_aie_core1_slot3", aie1::cm_stream_switch_slave_aie_core1_slot3},
      {"cm_stream_switch_slave_dma_0_slot0", aie1::cm_stream_switch_slave_dma_0_slot0},
      {"cm_stream_switch_slave_dma_0_slot1", aie1::cm_stream_switch_slave_dma_0_slot1},
      {"cm_stream_switch_slave_dma_0_slot2", aie1::cm_stream_switch_slave_dma_0_slot2},
      {"cm_stream_switch_slave_dma_0_slot3", aie1::cm_stream_switch_slave_dma_0_slot3},
      {"cm_stream_switch_slave_dma_1_slot0", aie1::cm_stream_switch_slave_dma_1_slot0},
      {"cm_stream_switch_slave_dma_1_slot1", aie1::cm_stream_switch_slave_dma_1_slot1},
      {"cm_stream_switch_slave_dma_1_slot2", aie1::cm_stream_switch_slave_dma_1_slot2},
      {"cm_stream_switch_slave_dma_1_slot3", aie1::cm_stream_switch_slave_dma_1_slot3},
      {"cm_stream_switch_slave_tile_ctrl_slot0", aie1::cm_stream_switch_slave_tile_ctrl_slot0},
      {"cm_stream_switch_slave_tile_ctrl_slot1", aie1::cm_stream_switch_slave_tile_ctrl_slot1},
      {"cm_stream_switch_slave_tile_ctrl_slot2", aie1::cm_stream_switch_slave_tile_ctrl_slot2},
      {"cm_stream_switch_slave_tile_ctrl_slot3", aie1::cm_stream_switch_slave_tile_ctrl_slot3},
      {"cm_stream_switch_slave_fifo_0_slot0", aie1::cm_stream_switch_slave_fifo_0_slot0},
      {"cm_stream_switch_slave_fifo_0_slot1", aie1::cm_stream_switch_slave_fifo_0_slot1},
      {"cm_stream_switch_slave_fifo_0_slot2", aie1::cm_stream_switch_slave_fifo_0_slot2},
      {"cm_stream_switch_slave_fifo_0_slot3", aie1::cm_stream_switch_slave_fifo_0_slot3},
      {"cm_stream_switch_slave_fifo_1_slot0", aie1::cm_stream_switch_slave_fifo_1_slot0},
      {"cm_stream_switch_slave_fifo_1_slot1", aie1::cm_stream_switch_slave_fifo_1_slot1},
      {"cm_stream_switch_slave_fifo_1_slot2", aie1::cm_stream_switch_slave_fifo_1_slot2},
      {"cm_stream_switch_slave_fifo_1_slot3", aie1::cm_stream_switch_slave_fifo_1_slot3},
      {"cm_stream_switch_slave_south_0_slot0", aie1::cm_stream_switch_slave_south_0_slot0},
      {"cm_stream_switch_slave_south_0_slot1", aie1::cm_stream_switch_slave_south_0_slot1},
      {"cm_stream_switch_slave_south_0_slot2", aie1::cm_stream_switch_slave_south_0_slot2},
      {"cm_stream_switch_slave_south_0_slot3", aie1::cm_stream_switch_slave_south_0_slot3},
      {"cm_stream_switch_slave_south_1_slot0", aie1::cm_stream_switch_slave_south_1_slot0},
      {"cm_stream_switch_slave_south_1_slot1", aie1::cm_stream_switch_slave_south_1_slot1},
      {"cm_stream_switch_slave_south_1_slot2", aie1::cm_stream_switch_slave_south_1_slot2},
      {"cm_stream_switch_slave_south_1_slot3", aie1::cm_stream_switch_slave_south_1_slot3},
      {"cm_stream_switch_slave_south_2_slot0", aie1::cm_stream_switch_slave_south_2_slot0},
      {"cm_stream_switch_slave_south_2_slot1", aie1::cm_stream_switch_slave_south_2_slot1},
      {"cm_stream_switch_slave_south_2_slot2", aie1::cm_stream_switch_slave_south_2_slot2},
      {"cm_stream_switch_slave_south_2_slot3", aie1::cm_stream_switch_slave_south_2_slot3},
      {"cm_stream_switch_slave_south_3_slot0", aie1::cm_stream_switch_slave_south_3_slot0},
      {"cm_stream_switch_slave_south_3_slot1", aie1::cm_stream_switch_slave_south_3_slot1},
      {"cm_stream_switch_slave_south_3_slot2", aie1::cm_stream_switch_slave_south_3_slot2},
      {"cm_stream_switch_slave_south_3_slot3", aie1::cm_stream_switch_slave_south_3_slot3},
      {"cm_stream_switch_slave_south_4_slot0", aie1::cm_stream_switch_slave_south_4_slot0},
      {"cm_stream_switch_slave_south_4_slot1", aie1::cm_stream_switch_slave_south_4_slot1},
      {"cm_stream_switch_slave_south_4_slot2", aie1::cm_stream_switch_slave_south_4_slot2},
      {"cm_stream_switch_slave_south_4_slot3", aie1::cm_stream_switch_slave_south_4_slot3},
      {"cm_stream_switch_slave_south_5_slot0", aie1::cm_stream_switch_slave_south_5_slot0},
      {"cm_stream_switch_slave_south_5_slot1", aie1::cm_stream_switch_slave_south_5_slot1},
      {"cm_stream_switch_slave_south_5_slot2", aie1::cm_stream_switch_slave_south_5_slot2},
      {"cm_stream_switch_slave_south_5_slot3", aie1::cm_stream_switch_slave_south_5_slot3},
      {"cm_stream_switch_slave_west_0_slot0", aie1::cm_stream_switch_slave_west_0_slot0},
      {"cm_stream_switch_slave_west_0_slot1", aie1::cm_stream_switch_slave_west_0_slot1},
      {"cm_stream_switch_slave_west_0_slot2", aie1::cm_stream_switch_slave_west_0_slot2},
      {"cm_stream_switch_slave_west_0_slot3", aie1::cm_stream_switch_slave_west_0_slot3},
      {"cm_stream_switch_slave_west_1_slot0", aie1::cm_stream_switch_slave_west_1_slot0},
      {"cm_stream_switch_slave_west_1_slot1", aie1::cm_stream_switch_slave_west_1_slot1},
      {"cm_stream_switch_slave_west_1_slot2", aie1::cm_stream_switch_slave_west_1_slot2},
      {"cm_stream_switch_slave_west_1_slot3", aie1::cm_stream_switch_slave_west_1_slot3},
      {"cm_stream_switch_slave_west_2_slot0", aie1::cm_stream_switch_slave_west_2_slot0},
      {"cm_stream_switch_slave_west_2_slot1", aie1::cm_stream_switch_slave_west_2_slot1},
      {"cm_stream_switch_slave_west_2_slot2", aie1::cm_stream_switch_slave_west_2_slot2},
      {"cm_stream_switch_slave_west_2_slot3", aie1::cm_stream_switch_slave_west_2_slot3},
      {"cm_stream_switch_slave_west_3_slot0", aie1::cm_stream_switch_slave_west_3_slot0},
      {"cm_stream_switch_slave_west_3_slot1", aie1::cm_stream_switch_slave_west_3_slot1},
      {"cm_stream_switch_slave_west_3_slot2", aie1::cm_stream_switch_slave_west_3_slot2},
      {"cm_stream_switch_slave_west_3_slot3", aie1::cm_stream_switch_slave_west_3_slot3},
      {"cm_stream_switch_slave_north_0_slot0", aie1::cm_stream_switch_slave_north_0_slot0},
      {"cm_stream_switch_slave_north_0_slot1", aie1::cm_stream_switch_slave_north_0_slot1},
      {"cm_stream_switch_slave_north_0_slot2", aie1::cm_stream_switch_slave_north_0_slot2},
      {"cm_stream_switch_slave_north_0_slot3", aie1::cm_stream_switch_slave_north_0_slot3},
      {"cm_stream_switch_slave_north_1_slot0", aie1::cm_stream_switch_slave_north_1_slot0},
      {"cm_stream_switch_slave_north_1_slot1", aie1::cm_stream_switch_slave_north_1_slot1},
      {"cm_stream_switch_slave_north_1_slot2", aie1::cm_stream_switch_slave_north_1_slot2},
      {"cm_stream_switch_slave_north_1_slot3", aie1::cm_stream_switch_slave_north_1_slot3},
      {"cm_stream_switch_slave_north_2_slot0", aie1::cm_stream_switch_slave_north_2_slot0},
      {"cm_stream_switch_slave_north_2_slot1", aie1::cm_stream_switch_slave_north_2_slot1},
      {"cm_stream_switch_slave_north_2_slot2", aie1::cm_stream_switch_slave_north_2_slot2},
      {"cm_stream_switch_slave_north_2_slot3", aie1::cm_stream_switch_slave_north_2_slot3},
      {"cm_stream_switch_slave_north_3_slot0", aie1::cm_stream_switch_slave_north_3_slot0},
      {"cm_stream_switch_slave_north_3_slot1", aie1::cm_stream_switch_slave_north_3_slot1},
      {"cm_stream_switch_slave_north_3_slot2", aie1::cm_stream_switch_slave_north_3_slot2},
      {"cm_stream_switch_slave_north_3_slot3", aie1::cm_stream_switch_slave_north_3_slot3},
      {"cm_stream_switch_slave_east_0_slot0", aie1::cm_stream_switch_slave_east_0_slot0},
      {"cm_stream_switch_slave_east_0_slot1", aie1::cm_stream_switch_slave_east_0_slot1},
      {"cm_stream_switch_slave_east_0_slot2", aie1::cm_stream_switch_slave_east_0_slot2},
      {"cm_stream_switch_slave_east_0_slot3", aie1::cm_stream_switch_slave_east_0_slot3},
      {"cm_stream_switch_slave_east_1_slot0", aie1::cm_stream_switch_slave_east_1_slot0},
      {"cm_stream_switch_slave_east_1_slot1", aie1::cm_stream_switch_slave_east_1_slot1},
      {"cm_stream_switch_slave_east_1_slot2", aie1::cm_stream_switch_slave_east_1_slot2},
      {"cm_stream_switch_slave_east_1_slot3", aie1::cm_stream_switch_slave_east_1_slot3},
      {"cm_stream_switch_slave_east_2_slot0", aie1::cm_stream_switch_slave_east_2_slot0},
      {"cm_stream_switch_slave_east_2_slot1", aie1::cm_stream_switch_slave_east_2_slot1},
      {"cm_stream_switch_slave_east_2_slot2", aie1::cm_stream_switch_slave_east_2_slot2},
      {"cm_stream_switch_slave_east_2_slot3", aie1::cm_stream_switch_slave_east_2_slot3},
      {"cm_stream_switch_slave_east_3_slot0", aie1::cm_stream_switch_slave_east_3_slot0},
      {"cm_stream_switch_slave_east_3_slot1", aie1::cm_stream_switch_slave_east_3_slot1},
      {"cm_stream_switch_slave_east_3_slot2", aie1::cm_stream_switch_slave_east_3_slot2},
      {"cm_stream_switch_slave_east_3_slot3", aie1::cm_stream_switch_slave_east_3_slot3},
      {"cm_stream_switch_slave_aie_trace_slot0", aie1::cm_stream_switch_slave_aie_trace_slot0},
      {"cm_stream_switch_slave_aie_trace_slot1", aie1::cm_stream_switch_slave_aie_trace_slot1},
      {"cm_stream_switch_slave_aie_trace_slot2", aie1::cm_stream_switch_slave_aie_trace_slot2},
      {"cm_stream_switch_slave_aie_trace_slot3", aie1::cm_stream_switch_slave_aie_trace_slot3},
      {"cm_stream_switch_slave_mem_trace_slot0", aie1::cm_stream_switch_slave_mem_trace_slot0},
      {"cm_stream_switch_slave_mem_trace_slot1", aie1::cm_stream_switch_slave_mem_trace_slot1},
      {"cm_stream_switch_slave_mem_trace_slot2", aie1::cm_stream_switch_slave_mem_trace_slot2},
      {"cm_stream_switch_slave_mem_trace_slot3", aie1::cm_stream_switch_slave_mem_trace_slot3},
      {"cm_stream_switch_event_port_selection_0", aie1::cm_stream_switch_event_port_selection_0},
      {"cm_stream_switch_event_port_selection_1", aie1::cm_stream_switch_event_port_selection_1},
      {"mm_performance_control0", aie1::mm_performance_control0},
      {"mm_performance_control1", aie1::mm_performance_control1},
      {"mm_performance_counter0", aie1::mm_performance_counter0},
      {"mm_performance_counter1", aie1::mm_performance_counter1},
      {"mm_performance_counter0_event_value", aie1::mm_performance_counter0_event_value},
      {"mm_performance_counter1_event_value", aie1::mm_performance_counter1_event_value},
      {"mm_checkbit_error_generation", aie1::mm_checkbit_error_generation},
      {"mm_ecc_scrubbing_event", aie1::mm_ecc_scrubbing_event},
      {"mm_ecc_failing_address", aie1::mm_ecc_failing_address},
      {"mm_parity_failing_address", aie1::mm_parity_failing_address},
      {"mm_reset_control", aie1::mm_reset_control},
      {"mm_timer_control", aie1::mm_timer_control},
      {"mm_event_generate", aie1::mm_event_generate},
      {"mm_event_broadcast0", aie1::mm_event_broadcast0},
      {"mm_event_broadcast1", aie1::mm_event_broadcast1},
      {"mm_event_broadcast2", aie1::mm_event_broadcast2},
      {"mm_event_broadcast3", aie1::mm_event_broadcast3},
      {"mm_event_broadcast4", aie1::mm_event_broadcast4},
      {"mm_event_broadcast5", aie1::mm_event_broadcast5},
      {"mm_event_broadcast6", aie1::mm_event_broadcast6},
      {"mm_event_broadcast7", aie1::mm_event_broadcast7},
      {"mm_event_broadcast8", aie1::mm_event_broadcast8},
      {"mm_event_broadcast9", aie1::mm_event_broadcast9},
      {"mm_event_broadcast10", aie1::mm_event_broadcast10},
      {"mm_event_broadcast11", aie1::mm_event_broadcast11},
      {"mm_event_broadcast12", aie1::mm_event_broadcast12},
      {"mm_event_broadcast13", aie1::mm_event_broadcast13},
      {"mm_event_broadcast14", aie1::mm_event_broadcast14},
      {"mm_event_broadcast15", aie1::mm_event_broadcast15},
      {"mm_event_broadcast_block_south_set", aie1::mm_event_broadcast_block_south_set},
      {"mm_event_broadcast_block_south_clr", aie1::mm_event_broadcast_block_south_clr},
      {"mm_event_broadcast_block_south_value", aie1::mm_event_broadcast_block_south_value},
      {"mm_event_broadcast_block_west_set", aie1::mm_event_broadcast_block_west_set},
      {"mm_event_broadcast_block_west_clr", aie1::mm_event_broadcast_block_west_clr},
      {"mm_event_broadcast_block_west_value", aie1::mm_event_broadcast_block_west_value},
      {"mm_event_broadcast_block_north_set", aie1::mm_event_broadcast_block_north_set},
      {"mm_event_broadcast_block_north_clr", aie1::mm_event_broadcast_block_north_clr},
      {"mm_event_broadcast_block_north_value", aie1::mm_event_broadcast_block_north_value},
      {"mm_event_broadcast_block_east_set", aie1::mm_event_broadcast_block_east_set},
      {"mm_event_broadcast_block_east_clr", aie1::mm_event_broadcast_block_east_clr},
      {"mm_event_broadcast_block_east_value", aie1::mm_event_broadcast_block_east_value},
      {"mm_trace_control0", aie1::mm_trace_control0},
      {"mm_trace_control1", aie1::mm_trace_control1},
      {"mm_trace_status", aie1::mm_trace_status},
      {"mm_trace_event0", aie1::mm_trace_event0},
      {"mm_trace_event1", aie1::mm_trace_event1},
      {"mm_timer_trig_event_low_value", aie1::mm_timer_trig_event_low_value},
      {"mm_timer_trig_event_high_value", aie1::mm_timer_trig_event_high_value},
      {"mm_timer_low", aie1::mm_timer_low},
      {"mm_timer_high", aie1::mm_timer_high},
      {"mm_watchpoint0", aie1::mm_watchpoint0},
      {"mm_watchpoint1", aie1::mm_watchpoint1},
      {"mm_event_status0", aie1::mm_event_status0},
      {"mm_event_status1", aie1::mm_event_status1},
      {"mm_event_status2", aie1::mm_event_status2},
      {"mm_event_status3", aie1::mm_event_status3},
      {"mm_reserved0", aie1::mm_reserved0},
      {"mm_reserved1", aie1::mm_reserved1},
      {"mm_reserved2", aie1::mm_reserved2},
      {"mm_reserved3", aie1::mm_reserved3},
      {"mm_combo_event_inputs", aie1::mm_combo_event_inputs},
      {"mm_combo_event_control", aie1::mm_combo_event_control},
      {"mm_event_group_0_enable", aie1::mm_event_group_0_enable},
      {"mm_event_group_watchpoint_enable", aie1::mm_event_group_watchpoint_enable},
      {"mm_event_group_dma_enable", aie1::mm_event_group_dma_enable},
      {"mm_event_group_lock_enable", aie1::mm_event_group_lock_enable},
      {"mm_event_group_memory_conflict_enable", aie1::mm_event_group_memory_conflict_enable},
      {"mm_event_group_error_enable", aie1::mm_event_group_error_enable},
      {"mm_event_group_broadcast_enable", aie1::mm_event_group_broadcast_enable},
      {"mm_event_group_user_event_enable", aie1::mm_event_group_user_event_enable},
      {"mm_spare_reg", aie1::mm_spare_reg},
      {"mm_dma_bd0_addr_a", aie1::mm_dma_bd0_addr_a},
      {"mm_dma_bd0_addr_b", aie1::mm_dma_bd0_addr_b},
      {"mm_dma_bd0_2d_x", aie1::mm_dma_bd0_2d_x},
      {"mm_dma_bd0_2d_y", aie1::mm_dma_bd0_2d_y},
      {"mm_dma_bd0_packet", aie1::mm_dma_bd0_packet},
      {"mm_dma_bd0_interleaved_state", aie1::mm_dma_bd0_interleaved_state},
      {"mm_dma_bd0_control", aie1::mm_dma_bd0_control},
      {"mm_dma_bd1_addr_a", aie1::mm_dma_bd1_addr_a},
      {"mm_dma_bd1_addr_b", aie1::mm_dma_bd1_addr_b},
      {"mm_dma_bd1_2d_x", aie1::mm_dma_bd1_2d_x},
      {"mm_dma_bd1_2d_y", aie1::mm_dma_bd1_2d_y},
      {"mm_dma_bd1_packet", aie1::mm_dma_bd1_packet},
      {"mm_dma_bd1_interleaved_state", aie1::mm_dma_bd1_interleaved_state},
      {"mm_dma_bd1_control", aie1::mm_dma_bd1_control},
      {"mm_dma_bd2_addr_a", aie1::mm_dma_bd2_addr_a},
      {"mm_dma_bd2_addr_b", aie1::mm_dma_bd2_addr_b},
      {"mm_dma_bd2_2d_x", aie1::mm_dma_bd2_2d_x},
      {"mm_dma_bd2_2d_y", aie1::mm_dma_bd2_2d_y},
      {"mm_dma_bd2_packet", aie1::mm_dma_bd2_packet},
      {"mm_dma_bd2_interleaved_state", aie1::mm_dma_bd2_interleaved_state},
      {"mm_dma_bd2_control", aie1::mm_dma_bd2_control},
      {"mm_dma_bd3_addr_a", aie1::mm_dma_bd3_addr_a},
      {"mm_dma_bd3_addr_b", aie1::mm_dma_bd3_addr_b},
      {"mm_dma_bd3_2d_x", aie1::mm_dma_bd3_2d_x},
      {"mm_dma_bd3_2d_y", aie1::mm_dma_bd3_2d_y},
      {"mm_dma_bd3_packet", aie1::mm_dma_bd3_packet},
      {"mm_dma_bd3_interleaved_state", aie1::mm_dma_bd3_interleaved_state},
      {"mm_dma_bd3_control", aie1::mm_dma_bd3_control},
      {"mm_dma_bd4_addr_a", aie1::mm_dma_bd4_addr_a},
      {"mm_dma_bd4_addr_b", aie1::mm_dma_bd4_addr_b},
      {"mm_dma_bd4_2d_x", aie1::mm_dma_bd4_2d_x},
      {"mm_dma_bd4_2d_y", aie1::mm_dma_bd4_2d_y},
      {"mm_dma_bd4_packet", aie1::mm_dma_bd4_packet},
      {"mm_dma_bd4_interleaved_state", aie1::mm_dma_bd4_interleaved_state},
      {"mm_dma_bd4_control", aie1::mm_dma_bd4_control},
      {"mm_dma_bd5_addr_a", aie1::mm_dma_bd5_addr_a},
      {"mm_dma_bd5_addr_b", aie1::mm_dma_bd5_addr_b},
      {"mm_dma_bd5_2d_x", aie1::mm_dma_bd5_2d_x},
      {"mm_dma_bd5_2d_y", aie1::mm_dma_bd5_2d_y},
      {"mm_dma_bd5_packet", aie1::mm_dma_bd5_packet},
      {"mm_dma_bd5_interleaved_state", aie1::mm_dma_bd5_interleaved_state},
      {"mm_dma_bd5_control", aie1::mm_dma_bd5_control},
      {"mm_dma_bd6_addr_a", aie1::mm_dma_bd6_addr_a},
      {"mm_dma_bd6_addr_b", aie1::mm_dma_bd6_addr_b},
      {"mm_dma_bd6_2d_x", aie1::mm_dma_bd6_2d_x},
      {"mm_dma_bd6_2d_y", aie1::mm_dma_bd6_2d_y},
      {"mm_dma_bd6_packet", aie1::mm_dma_bd6_packet},
      {"mm_dma_bd6_interleaved_state", aie1::mm_dma_bd6_interleaved_state},
      {"mm_dma_bd6_control", aie1::mm_dma_bd6_control},
      {"mm_dma_bd7_addr_a", aie1::mm_dma_bd7_addr_a},
      {"mm_dma_bd7_addr_b", aie1::mm_dma_bd7_addr_b},
      {"mm_dma_bd7_2d_x", aie1::mm_dma_bd7_2d_x},
      {"mm_dma_bd7_2d_y", aie1::mm_dma_bd7_2d_y},
      {"mm_dma_bd7_packet", aie1::mm_dma_bd7_packet},
      {"mm_dma_bd7_interleaved_state", aie1::mm_dma_bd7_interleaved_state},
      {"mm_dma_bd7_control", aie1::mm_dma_bd7_control},
      {"mm_dma_bd8_addr_a", aie1::mm_dma_bd8_addr_a},
      {"mm_dma_bd8_addr_b", aie1::mm_dma_bd8_addr_b},
      {"mm_dma_bd8_2d_x", aie1::mm_dma_bd8_2d_x},
      {"mm_dma_bd8_2d_y", aie1::mm_dma_bd8_2d_y},
      {"mm_dma_bd8_packet", aie1::mm_dma_bd8_packet},
      {"mm_dma_bd8_interleaved_state", aie1::mm_dma_bd8_interleaved_state},
      {"mm_dma_bd8_control", aie1::mm_dma_bd8_control},
      {"mm_dma_bd9_addr_a", aie1::mm_dma_bd9_addr_a},
      {"mm_dma_bd9_addr_b", aie1::mm_dma_bd9_addr_b},
      {"mm_dma_bd9_2d_x", aie1::mm_dma_bd9_2d_x},
      {"mm_dma_bd9_2d_y", aie1::mm_dma_bd9_2d_y},
      {"mm_dma_bd9_packet", aie1::mm_dma_bd9_packet},
      {"mm_dma_bd9_interleaved_state", aie1::mm_dma_bd9_interleaved_state},
      {"mm_dma_bd9_control", aie1::mm_dma_bd9_control},
      {"mm_dma_bd10_addr_a", aie1::mm_dma_bd10_addr_a},
      {"mm_dma_bd10_addr_b", aie1::mm_dma_bd10_addr_b},
      {"mm_dma_bd10_2d_x", aie1::mm_dma_bd10_2d_x},
      {"mm_dma_bd10_2d_y", aie1::mm_dma_bd10_2d_y},
      {"mm_dma_bd10_packet", aie1::mm_dma_bd10_packet},
      {"mm_dma_bd10_interleaved_state", aie1::mm_dma_bd10_interleaved_state},
      {"mm_dma_bd10_control", aie1::mm_dma_bd10_control},
      {"mm_dma_bd11_addr_a", aie1::mm_dma_bd11_addr_a},
      {"mm_dma_bd11_addr_b", aie1::mm_dma_bd11_addr_b},
      {"mm_dma_bd11_2d_x", aie1::mm_dma_bd11_2d_x},
      {"mm_dma_bd11_2d_y", aie1::mm_dma_bd11_2d_y},
      {"mm_dma_bd11_packet", aie1::mm_dma_bd11_packet},
      {"mm_dma_bd11_interleaved_state", aie1::mm_dma_bd11_interleaved_state},
      {"mm_dma_bd11_control", aie1::mm_dma_bd11_control},
      {"mm_dma_bd12_addr_a", aie1::mm_dma_bd12_addr_a},
      {"mm_dma_bd12_addr_b", aie1::mm_dma_bd12_addr_b},
      {"mm_dma_bd12_2d_x", aie1::mm_dma_bd12_2d_x},
      {"mm_dma_bd12_2d_y", aie1::mm_dma_bd12_2d_y},
      {"mm_dma_bd12_packet", aie1::mm_dma_bd12_packet},
      {"mm_dma_bd12_interleaved_state", aie1::mm_dma_bd12_interleaved_state},
      {"mm_dma_bd12_control", aie1::mm_dma_bd12_control},
      {"mm_dma_bd13_addr_a", aie1::mm_dma_bd13_addr_a},
      {"mm_dma_bd13_addr_b", aie1::mm_dma_bd13_addr_b},
      {"mm_dma_bd13_2d_x", aie1::mm_dma_bd13_2d_x},
      {"mm_dma_bd13_2d_y", aie1::mm_dma_bd13_2d_y},
      {"mm_dma_bd13_packet", aie1::mm_dma_bd13_packet},
      {"mm_dma_bd13_interleaved_state", aie1::mm_dma_bd13_interleaved_state},
      {"mm_dma_bd13_control", aie1::mm_dma_bd13_control},
      {"mm_dma_bd14_addr_a", aie1::mm_dma_bd14_addr_a},
      {"mm_dma_bd14_addr_b", aie1::mm_dma_bd14_addr_b},
      {"mm_dma_bd14_2d_x", aie1::mm_dma_bd14_2d_x},
      {"mm_dma_bd14_2d_y", aie1::mm_dma_bd14_2d_y},
      {"mm_dma_bd14_packet", aie1::mm_dma_bd14_packet},
      {"mm_dma_bd14_interleaved_state", aie1::mm_dma_bd14_interleaved_state},
      {"mm_dma_bd14_control", aie1::mm_dma_bd14_control},
      {"mm_dma_bd15_addr_a", aie1::mm_dma_bd15_addr_a},
      {"mm_dma_bd15_addr_b", aie1::mm_dma_bd15_addr_b},
      {"mm_dma_bd15_2d_x", aie1::mm_dma_bd15_2d_x},
      {"mm_dma_bd15_2d_y", aie1::mm_dma_bd15_2d_y},
      {"mm_dma_bd15_packet", aie1::mm_dma_bd15_packet},
      {"mm_dma_bd15_interleaved_state", aie1::mm_dma_bd15_interleaved_state},
      {"mm_dma_bd15_control", aie1::mm_dma_bd15_control},
      {"mm_dma_s2mm_0_ctrl", aie1::mm_dma_s2mm_0_ctrl},
      {"mm_dma_s2mm_0_start_queue", aie1::mm_dma_s2mm_0_start_queue},
      {"mm_dma_s2mm_1_ctrl", aie1::mm_dma_s2mm_1_ctrl},
      {"mm_dma_s2mm_1_start_queue", aie1::mm_dma_s2mm_1_start_queue},
      {"mm_dma_mm2s_0_ctrl", aie1::mm_dma_mm2s_0_ctrl},
      {"mm_dma_mm2s_0_start_queue", aie1::mm_dma_mm2s_0_start_queue},
      {"mm_dma_mm2s_1_ctrl", aie1::mm_dma_mm2s_1_ctrl},
      {"mm_dma_mm2s_1_start_queue", aie1::mm_dma_mm2s_1_start_queue},
      {"mm_dma_s2mm_status", aie1::mm_dma_s2mm_status},
      {"mm_dma_mm2s_status", aie1::mm_dma_mm2s_status},
      {"mm_dma_fifo_counter", aie1::mm_dma_fifo_counter},
      {"mm_lock0_release_nv", aie1::mm_lock0_release_nv},
      {"mm_lock0_release_v0", aie1::mm_lock0_release_v0},
      {"mm_lock0_release_v1", aie1::mm_lock0_release_v1},
      {"mm_lock0_acquire_nv", aie1::mm_lock0_acquire_nv},
      {"mm_lock0_acquire_v0", aie1::mm_lock0_acquire_v0},
      {"mm_lock0_acquire_v1", aie1::mm_lock0_acquire_v1},
      {"mm_lock1_release_nv", aie1::mm_lock1_release_nv},
      {"mm_lock1_release_v0", aie1::mm_lock1_release_v0},
      {"mm_lock1_release_v1", aie1::mm_lock1_release_v1},
      {"mm_lock1_acquire_nv", aie1::mm_lock1_acquire_nv},
      {"mm_lock1_acquire_v0", aie1::mm_lock1_acquire_v0},
      {"mm_lock1_acquire_v1", aie1::mm_lock1_acquire_v1},
      {"mm_lock2_release_nv", aie1::mm_lock2_release_nv},
      {"mm_lock2_release_v0", aie1::mm_lock2_release_v0},
      {"mm_lock2_release_v1", aie1::mm_lock2_release_v1},
      {"mm_lock2_acquire_nv", aie1::mm_lock2_acquire_nv},
      {"mm_lock2_acquire_v0", aie1::mm_lock2_acquire_v0},
      {"mm_lock2_acquire_v1", aie1::mm_lock2_acquire_v1},
      {"mm_lock3_release_nv", aie1::mm_lock3_release_nv},
      {"mm_lock3_release_v0", aie1::mm_lock3_release_v0},
      {"mm_lock3_release_v1", aie1::mm_lock3_release_v1},
      {"mm_lock3_acquire_nv", aie1::mm_lock3_acquire_nv},
      {"mm_lock3_acquire_v0", aie1::mm_lock3_acquire_v0},
      {"mm_lock3_acquire_v1", aie1::mm_lock3_acquire_v1},
      {"mm_lock4_release_nv", aie1::mm_lock4_release_nv},
      {"mm_lock4_release_v0", aie1::mm_lock4_release_v0},
      {"mm_lock4_release_v1", aie1::mm_lock4_release_v1},
      {"mm_lock4_acquire_nv", aie1::mm_lock4_acquire_nv},
      {"mm_lock4_acquire_v0", aie1::mm_lock4_acquire_v0},
      {"mm_lock4_acquire_v1", aie1::mm_lock4_acquire_v1},
      {"mm_lock5_release_nv", aie1::mm_lock5_release_nv},
      {"mm_lock5_release_v0", aie1::mm_lock5_release_v0},
      {"mm_lock5_release_v1", aie1::mm_lock5_release_v1},
      {"mm_lock5_acquire_nv", aie1::mm_lock5_acquire_nv},
      {"mm_lock5_acquire_v0", aie1::mm_lock5_acquire_v0},
      {"mm_lock5_acquire_v1", aie1::mm_lock5_acquire_v1},
      {"mm_lock6_release_nv", aie1::mm_lock6_release_nv},
      {"mm_lock6_release_v0", aie1::mm_lock6_release_v0},
      {"mm_lock6_release_v1", aie1::mm_lock6_release_v1},
      {"mm_lock6_acquire_nv", aie1::mm_lock6_acquire_nv},
      {"mm_lock6_acquire_v0", aie1::mm_lock6_acquire_v0},
      {"mm_lock6_acquire_v1", aie1::mm_lock6_acquire_v1},
      {"mm_lock7_release_nv", aie1::mm_lock7_release_nv},
      {"mm_lock7_release_v0", aie1::mm_lock7_release_v0},
      {"mm_lock7_release_v1", aie1::mm_lock7_release_v1},
      {"mm_lock7_acquire_nv", aie1::mm_lock7_acquire_nv},
      {"mm_lock7_acquire_v0", aie1::mm_lock7_acquire_v0},
      {"mm_lock7_acquire_v1", aie1::mm_lock7_acquire_v1},
      {"mm_lock8_release_nv", aie1::mm_lock8_release_nv},
      {"mm_lock8_release_v0", aie1::mm_lock8_release_v0},
      {"mm_lock8_release_v1", aie1::mm_lock8_release_v1},
      {"mm_lock8_acquire_nv", aie1::mm_lock8_acquire_nv},
      {"mm_lock8_acquire_v0", aie1::mm_lock8_acquire_v0},
      {"mm_lock8_acquire_v1", aie1::mm_lock8_acquire_v1},
      {"mm_lock9_release_nv", aie1::mm_lock9_release_nv},
      {"mm_lock9_release_v0", aie1::mm_lock9_release_v0},
      {"mm_lock9_release_v1", aie1::mm_lock9_release_v1},
      {"mm_lock9_acquire_nv", aie1::mm_lock9_acquire_nv},
      {"mm_lock9_acquire_v0", aie1::mm_lock9_acquire_v0},
      {"mm_lock9_acquire_v1", aie1::mm_lock9_acquire_v1},
      {"mm_lock10_release_nv", aie1::mm_lock10_release_nv},
      {"mm_lock10_release_v0", aie1::mm_lock10_release_v0},
      {"mm_lock10_release_v1", aie1::mm_lock10_release_v1},
      {"mm_lock10_acquire_nv", aie1::mm_lock10_acquire_nv},
      {"mm_lock10_acquire_v0", aie1::mm_lock10_acquire_v0},
      {"mm_lock10_acquire_v1", aie1::mm_lock10_acquire_v1},
      {"mm_lock11_release_nv", aie1::mm_lock11_release_nv},
      {"mm_lock11_release_v0", aie1::mm_lock11_release_v0},
      {"mm_lock11_release_v1", aie1::mm_lock11_release_v1},
      {"mm_lock11_acquire_nv", aie1::mm_lock11_acquire_nv},
      {"mm_lock11_acquire_v0", aie1::mm_lock11_acquire_v0},
      {"mm_lock11_acquire_v1", aie1::mm_lock11_acquire_v1},
      {"mm_lock12_release_nv", aie1::mm_lock12_release_nv},
      {"mm_lock12_release_v0", aie1::mm_lock12_release_v0},
      {"mm_lock12_release_v1", aie1::mm_lock12_release_v1},
      {"mm_lock12_acquire_nv", aie1::mm_lock12_acquire_nv},
      {"mm_lock12_acquire_v0", aie1::mm_lock12_acquire_v0},
      {"mm_lock12_acquire_v1", aie1::mm_lock12_acquire_v1},
      {"mm_lock13_release_nv", aie1::mm_lock13_release_nv},
      {"mm_lock13_release_v0", aie1::mm_lock13_release_v0},
      {"mm_lock13_release_v1", aie1::mm_lock13_release_v1},
      {"mm_lock13_acquire_nv", aie1::mm_lock13_acquire_nv},
      {"mm_lock13_acquire_v0", aie1::mm_lock13_acquire_v0},
      {"mm_lock13_acquire_v1", aie1::mm_lock13_acquire_v1},
      {"mm_lock14_release_nv", aie1::mm_lock14_release_nv},
      {"mm_lock14_release_v0", aie1::mm_lock14_release_v0},
      {"mm_lock14_release_v1", aie1::mm_lock14_release_v1},
      {"mm_lock14_acquire_nv", aie1::mm_lock14_acquire_nv},
      {"mm_lock14_acquire_v0", aie1::mm_lock14_acquire_v0},
      {"mm_lock14_acquire_v1", aie1::mm_lock14_acquire_v1},
      {"mm_lock15_release_nv", aie1::mm_lock15_release_nv},
      {"mm_lock15_release_v0", aie1::mm_lock15_release_v0},
      {"mm_lock15_release_v1", aie1::mm_lock15_release_v1},
      {"mm_lock15_acquire_nv", aie1::mm_lock15_acquire_nv},
      {"mm_lock15_acquire_v0", aie1::mm_lock15_acquire_v0},
      {"mm_lock15_acquire_v1", aie1::mm_lock15_acquire_v1},
      {"mm_all_lock_state_value", aie1::mm_all_lock_state_value},
      {"mm_lock_event_value_control_0", aie1::mm_lock_event_value_control_0},
      {"mm_lock_event_value_control_1", aie1::mm_lock_event_value_control_1},
      {"shim_dma_bd_step_size", aie1::shim_dma_bd_step_size},
      {"shim_dma_s2mm_step_size", aie1::shim_dma_s2mm_step_size},
      {"shim_all_lock_state_value", aie1::shim_all_lock_state_value},
      {"shim_dma_bd0_addr_low", aie1::shim_dma_bd0_addr_low},
      {"shim_dma_bd0_buffer_length", aie1::shim_dma_bd0_buffer_length},
      {"shim_dma_bd0_control", aie1::shim_dma_bd0_control},
      {"shim_dma_bd0_axi_config", aie1::shim_dma_bd0_axi_config},
      {"shim_dma_bd0_packet", aie1::shim_dma_bd0_packet},
      {"shim_dma_s2mm_0_ctrl", aie1::shim_dma_s2mm_0_ctrl},
      {"shim_dma_s2mm_0_start_queue", aie1::shim_dma_s2mm_0_start_queue},
      {"shim_performance_control0", aie1::shim_performance_control0},
      {"shim_performance_control1", aie1::shim_performance_control1},
      {"shim_performance_counter0", aie1::shim_performance_counter0},
      {"shim_performance_counter1", aie1::shim_performance_counter1},
      {"shim_performance_counter0_event_value", aie1::shim_performance_counter0_event_value},
      {"shim_performance_counter1_event_value", aie1::shim_performance_counter1_event_value},
      {"shim_event_generate", aie1::shim_event_generate},
      {"shim_event_broadcast_a_0", aie1::shim_event_broadcast_a_0},
      {"shim_event_broadcast_a_1", aie1::shim_event_broadcast_a_1},
      {"shim_event_broadcast_a_2", aie1::shim_event_broadcast_a_2},
      {"shim_event_broadcast_a_3", aie1::shim_event_broadcast_a_3},
      {"shim_event_broadcast_a_4", aie1::shim_event_broadcast_a_4},
      {"shim_event_broadcast_a_5", aie1::shim_event_broadcast_a_5},
      {"shim_event_broadcast_a_6", aie1::shim_event_broadcast_a_6},
      {"shim_event_broadcast_a_7", aie1::shim_event_broadcast_a_7},
      {"shim_event_broadcast_a_8", aie1::shim_event_broadcast_a_8},
      {"shim_event_broadcast_a_9", aie1::shim_event_broadcast_a_9},
      {"shim_event_broadcast_a_10", aie1::shim_event_broadcast_a_10},
      {"shim_event_broadcast_a_11", aie1::shim_event_broadcast_a_11},
      {"shim_event_broadcast_a_12", aie1::shim_event_broadcast_a_12},
      {"shim_event_broadcast_a_13", aie1::shim_event_broadcast_a_13},
      {"shim_event_broadcast_a_14", aie1::shim_event_broadcast_a_14},
      {"shim_event_broadcast_a_15", aie1::shim_event_broadcast_a_15},
      {"shim_event_broadcast_a_block_south_set", aie1::shim_event_broadcast_a_block_south_set},
      {"shim_event_broadcast_a_block_south_clr", aie1::shim_event_broadcast_a_block_south_clr},
      {"shim_event_broadcast_a_block_south_value", aie1::shim_event_broadcast_a_block_south_value},
      {"shim_event_broadcast_a_block_west_set", aie1::shim_event_broadcast_a_block_west_set},
      {"shim_event_broadcast_a_block_west_clr", aie1::shim_event_broadcast_a_block_west_clr},
      {"shim_event_broadcast_a_block_west_value", aie1::shim_event_broadcast_a_block_west_value},
      {"shim_event_broadcast_a_block_north_set", aie1::shim_event_broadcast_a_block_north_set},
      {"shim_event_broadcast_a_block_north_clr", aie1::shim_event_broadcast_a_block_north_clr},
      {"shim_event_broadcast_a_block_north_value", aie1::shim_event_broadcast_a_block_north_value},
      {"shim_event_broadcast_a_block_east_set", aie1::shim_event_broadcast_a_block_east_set},
      {"shim_event_broadcast_a_block_east_clr", aie1::shim_event_broadcast_a_block_east_clr},
      {"shim_event_broadcast_a_block_east_value", aie1::shim_event_broadcast_a_block_east_value},
      {"shim_trace_control0", aie1::shim_trace_control0},
      {"shim_trace_control1", aie1::shim_trace_control1},
      {"shim_trace_status", aie1::shim_trace_status},
      {"shim_trace_event0", aie1::shim_trace_event0},
      {"shim_trace_event1", aie1::shim_trace_event1},
      {"shim_timer_trig_event_low_value", aie1::shim_timer_trig_event_low_value},
      {"shim_timer_trig_event_high_value", aie1::shim_timer_trig_event_high_value},
      {"shim_timer_low", aie1::shim_timer_low},
      {"shim_timer_high", aie1::shim_timer_high},
      {"shim_event_status0", aie1::shim_event_status0},
      {"shim_event_status1", aie1::shim_event_status1},
      {"shim_event_status2", aie1::shim_event_status2},
      {"shim_event_status3", aie1::shim_event_status3},
      {"shim_event_group_dma_enable", aie1::shim_event_group_dma_enable},
      {"shim_stream_switch_event_port_selection_0", aie1::shim_stream_switch_event_port_selection_0},
      {"shim_stream_switch_event_port_selection_1", aie1::shim_stream_switch_event_port_selection_1},
      {"mem_performance_counter0", aie1::mem_performance_counter0}
    };
  }

  void populateRegValueToNameMap() {
    regValueToName=  {
      {0x0003f054, "cm_stream_switch_master_config_east0"},
      {0x0003f058, "cm_stream_switch_master_config_east1"},
      {0x0003f05c, "cm_stream_switch_master_config_east2"},
      {0x0003f060, "cm_stream_switch_master_config_east3"},
      {0x0001ef20, "mm_lock_event_value_control_0"},
      {0x0001d110, "mm_dma_bd8_packet"},
      {0x00034208, "cm_event_status2"},
      {0x0003420c, "cm_event_status3"},
      {0x00034200, "cm_event_status0"},
      {0x00034204, "cm_event_status1"},
      {0x00034048, "cm_event_broadcast14"},
      {0x0003404c, "cm_event_broadcast15"},
      {0x00034040, "cm_event_broadcast12"},
      {0x00034044, "cm_event_broadcast13"},
      {0x00034038, "cm_event_broadcast10"},
      {0x0003403c, "cm_event_broadcast11"},
      {0x00034080, "cm_event_broadcast_block_east_set"},
      {0x0001d174, "mm_dma_bd11_interleaved_state"},
      {0x0001d1b4, "mm_dma_bd13_interleaved_state"},
      {0x0001d1d4, "mm_dma_bd14_interleaved_state"},
      {0x0001d1cc, "mm_dma_bd14_2d_y"},
      {0x0001d1c8, "mm_dma_bd14_2d_x"},
      {0x0001e070, "mm_lock0_acquire_v1"},
      {0x0001e060, "mm_lock0_acquire_v0"},
      {0x0001e4b0, "mm_lock9_release_v1"},
      {0x0001e4a0, "mm_lock9_release_v0"},
      {0x000340f4, "shim_timer_trig_event_high_value"},
      {0x0001d150, "mm_dma_bd10_packet"},
      {0x00034068, "shim_event_broadcast_a_block_west_value"},
      {0x00034504, "shim_event_group_dma_enable"},
      {0x0001d060, "mm_dma_bd3_addr_a"},
      {0x0001d064, "mm_dma_bd3_addr_b"},
      {0x0003450c, "cm_event_group_core_program_flow_enable"},
      {0x00014074, "mm_event_broadcast_block_north_clr"},
      {0x00014050, "mm_event_broadcast_block_south_set"},
      {0x0001e100, "mm_lock2_release_nv"},
      {0x0001e2b0, "mm_lock5_release_v1"},
      {0x0001e2a0, "mm_lock5_release_v0"},
      {0x0003f154, "cm_stream_switch_slave_config_east_0"},
      {0x0001d1c0, "mm_dma_bd14_addr_a"},
      {0x00014008, "mm_event_generate"},
      {0x0003f2f4, "cm_stream_switch_slave_west_2_slot1"},
      {0x0003f2f0, "cm_stream_switch_slave_west_2_slot0"},
      {0x0003f2fc, "cm_stream_switch_slave_west_2_slot3"},
      {0x0003f2f8, "cm_stream_switch_slave_west_2_slot2"},
      {0x00034014, "shim_event_broadcast_a_1"},
      {0x0001e320, "mm_lock6_release_v0"},
      {0x00034500, "cm_event_group_0_enable"},
      {0x0001d00c, "shim_dma_bd0_axi_config"},
      {0x00016000, "mm_spare_reg"},
      {0x0003f150, "cm_stream_switch_slave_config_north_3"},
      {0x0003f14c, "cm_stream_switch_slave_config_north_2"},
      {0x0003f148, "cm_stream_switch_slave_config_north_1"},
      {0x0003f144, "cm_stream_switch_slave_config_north_0"},
      {0x0001de14, "mm_dma_mm2s_0_start_queue"},
      {0x0001e420, "mm_lock8_release_v0"},
      {0x0001e430, "mm_lock8_release_v1"},
      {0x000140d8, "mm_trace_status"},
      {0x0001d190, "mm_dma_bd12_packet"},
      {0x0003f204, "cm_stream_switch_slave_aie_core0_slot1"},
      {0x0003f200, "cm_stream_switch_slave_aie_core0_slot0"},
      {0x0003f20c, "cm_stream_switch_slave_aie_core0_slot3"},
      {0x0003f208, "cm_stream_switch_slave_aie_core0_slot2"},
      {0x0001d08c, "mm_dma_bd4_2d_y"},
      {0x0001d088, "mm_dma_bd4_2d_x"},
      {0x0001451c, "mm_event_group_user_event_enable"},
      {0x00014000, "mm_timer_control"},
      {0x00031000, "shim_performance_control0"},
      {0x0003f388, "cm_stream_switch_slave_east_3_slot2"},
      {0x0001e560, "mm_lock10_acquire_v0"},
      {0x00034024, "cm_event_broadcast5"},
      {0x0003f380, "cm_stream_switch_slave_east_3_slot0"},
      {0x0003f384, "cm_stream_switch_slave_east_3_slot1"},
      {0x0001d168, "mm_dma_bd11_2d_x"},
      {0x0001d16c, "mm_dma_bd11_2d_y"},
      {0x00031080, "shim_performance_counter0_event_value"},
      {0x0001d114, "mm_dma_bd8_interleaved_state"},
      {0x0001de0c, "mm_dma_s2mm_1_start_queue"},
      {0x0001d0d0, "mm_dma_bd6_packet"},
      {0x00034508, "cm_event_group_core_stall_enable"},
      {0x0001d158, "mm_dma_bd10_control"},
      {0x00034510, "cm_event_group_errors0_enable"},
      {0x00034058, "shim_event_broadcast_a_block_south_value"},
      {0x0001e770, "mm_lock14_acquire_v1"},
      {0x0001e760, "mm_lock14_acquire_v0"},
      {0x00032030, "cm_error_halt_control"},
      {0x0001e6b0, "mm_lock13_release_v1"},
      {0x0001e6c0, "mm_lock13_acquire_nv"},
      {0x0001e0b0, "mm_lock1_release_v1"},
      {0x0001e0a0, "mm_lock1_release_v0"},
      {0x0001d004, "shim_dma_bd0_buffer_length"},
      {0x0003f110, "cm_stream_switch_slave_config_tile_ctrl"},
      {0x0001d0b0, "mm_dma_bd5_packet"},
      {0x00014404, "mm_combo_event_control"},
      {0x0003f334, "cm_stream_switch_slave_north_2_slot1"},
      {0x0003f338, "cm_stream_switch_slave_north_2_slot2"},
      {0x0003f33c, "cm_stream_switch_slave_north_2_slot3"},
      {0x0001d134, "mm_dma_bd9_interleaved_state"},
      {0x00031020, "shim_performance_counter0"},
      {0x00031024, "shim_performance_counter1"},
      {0x00014514, "mm_event_group_error_enable"},
      {0x00014508, "mm_event_group_dma_enable"},
      {0x000340f4, "cm_timer_trig_event_high_value"},
      {0x0001d140, "shim_dma_s2mm_0_ctrl"},
      {0x00031008, "shim_performance_control1"},
      {0x0001e780, "mm_lock15_release_nv"},
      {0x0003ff00, "cm_stream_switch_event_port_selection_0"},
      {0x0001d038, "mm_dma_bd1_control"},
      {0x0001e680, "mm_lock13_release_nv"},
      {0x00034050, "cm_event_broadcast_block_south_set"},
      {0x0001e330, "mm_lock6_release_v1"},
      {0x0001d080, "mm_dma_bd4_addr_a"},
      {0x0001d084, "mm_dma_bd4_addr_b"},
      {0x000340f0, "cm_timer_trig_event_low_value"},
      {0x00014058, "mm_event_broadcast_block_south_value"},
      {0x0003f354, "cm_stream_switch_slave_east_0_slot1"},
      {0x0003f350, "cm_stream_switch_slave_east_0_slot0"},
      {0x0003f35c, "cm_stream_switch_slave_east_0_slot3"},
      {0x0003f358, "cm_stream_switch_slave_east_0_slot2"},
      {0x0003f018, "cm_stream_switch_master_config_fifo1"},
      {0x0003f014, "cm_stream_switch_master_config_fifo0"},
      {0x0003f298, "cm_stream_switch_slave_south_2_slot2"},
      {0x0003f29c, "cm_stream_switch_slave_south_2_slot3"},
      {0x0003f290, "cm_stream_switch_slave_south_2_slot0"},
      {0x0003f294, "cm_stream_switch_slave_south_2_slot1"},
      {0x00014200, "mm_event_status0"},
      {0x00014204, "mm_event_status1"},
      {0x00014208, "mm_event_status2"},
      {0x0001420c, "mm_event_status3"},
      {0x00034088, "cm_event_broadcast_block_east_value"},
      {0x00011084, "mm_performance_counter1_event_value"},
      {0x00034058, "cm_event_broadcast_block_south_value"},
      {0x0003202c, "cm_pc_event3"},
      {0x0001d0d4, "mm_dma_bd6_interleaved_state"},
      {0x0003ff04, "cm_stream_switch_event_port_selection_1"},
      {0x0003f360, "cm_stream_switch_slave_east_1_slot0"},
      {0x0003f364, "cm_stream_switch_slave_east_1_slot1"},
      {0x0003f368, "cm_stream_switch_slave_east_1_slot2"},
      {0x0003f36c, "cm_stream_switch_slave_east_1_slot3"},
      {0x0001d070, "mm_dma_bd3_packet"},
      {0x0001d14c, "mm_dma_bd10_2d_y"},
      {0x0001d148, "mm_dma_bd10_2d_x"},
      {0x0001d1a4, "mm_dma_bd13_addr_b"},
      {0x0001d1a0, "mm_dma_bd13_addr_a"},
      {0x0001ef24, "mm_lock_event_value_control_1"},
      {0x0001d068, "mm_dma_bd3_2d_x"},
      {0x0001d06c, "mm_dma_bd3_2d_y"},
      {0x0001e340, "mm_lock6_acquire_nv"},
      {0x0001d058, "mm_dma_bd2_control"},
      {0x00031088, "cm_performance_counter2_event_value"},
      {0x000340f0, "shim_timer_trig_event_low_value"},
      {0x000140f4, "mm_timer_trig_event_high_value"},
      {0x0001d0d8, "mm_dma_bd6_control"},
      {0x00032034, "cm_error_halt_event"},
      {0x00034074, "shim_event_broadcast_a_block_north_clr"},
      {0x00014400, "mm_combo_event_inputs"},
      {0x0001e5b0, "mm_lock11_release_v1"},
      {0x0001e5a0, "mm_lock11_release_v0"},
      {0x0, "mem_performance_counter0"},
      {0x0003f100, "cm_stream_switch_slave_config_aie_core0"},
      {0x0003f040, "cm_stream_switch_master_config_north1"},
      {0x0003f03c, "cm_stream_switch_master_config_north0"},
      {0x0003f048, "cm_stream_switch_master_config_north3"},
      {0x0003f044, "cm_stream_switch_master_config_north2"},
      {0x0003f050, "cm_stream_switch_master_config_north5"},
      {0x0003f04c, "cm_stream_switch_master_config_north4"},
      {0x0001d1c4, "mm_dma_bd14_addr_b"},
      {0x8, "shim_dma_s2mm_step_size"},
      {0x0003f37c, "cm_stream_switch_slave_east_2_slot3"},
      {0x0003f378, "cm_stream_switch_slave_east_2_slot2"},
      {0x0003f374, "cm_stream_switch_slave_east_2_slot1"},
      {0x0003f370, "cm_stream_switch_slave_east_2_slot0"},
      {0x00014070, "mm_event_broadcast_block_north_set"},
      {0x0001de10, "mm_dma_mm2s_0_ctrl"},
      {0x0001d180, "mm_dma_bd12_addr_a"},
      {0x0001d0f0, "mm_dma_bd7_packet"},
      {0x0001e280, "mm_lock5_release_nv"},
      {0x0001e170, "mm_lock2_acquire_v1"},
      {0x0001e160, "mm_lock2_acquire_v0"},
      {0x0001e4c0, "mm_lock9_acquire_nv"},
      {0x0003f284, "cm_stream_switch_slave_south_1_slot1"},
      {0x0003f280, "cm_stream_switch_slave_south_1_slot0"},
      {0x0003f28c, "cm_stream_switch_slave_south_1_slot3"},
      {0x0003f288, "cm_stream_switch_slave_south_1_slot2"},
      {0x000140e0, "mm_trace_event0"},
      {0x000140e4, "mm_trace_event1"},
      {0x0003f21c, "cm_stream_switch_slave_aie_core1_slot3"},
      {0x0001df00, "mm_dma_s2mm_status"},
      {0x00034078, "cm_event_broadcast_block_north_value"},
      {0x00014060, "mm_event_broadcast_block_west_set"},
      {0x00034400, "cm_combo_event_inputs"},
      {0x0001e180, "mm_lock3_release_nv"},
      {0x0001450c, "mm_event_group_lock_enable"},
      {0x0003f164, "cm_stream_switch_slave_config_aie_trace"},
      {0x0001e7f0, "mm_lock15_acquire_v1"},
      {0x0001d078, "mm_dma_bd3_control"},
      {0x0003f240, "cm_stream_switch_slave_tile_ctrl_slot0"},
      {0x0003f244, "cm_stream_switch_slave_tile_ctrl_slot1"},
      {0x0003f248, "cm_stream_switch_slave_tile_ctrl_slot2"},
      {0x0003f24c, "cm_stream_switch_slave_tile_ctrl_slot3"},
      {0x00034070, "cm_event_broadcast_block_north_set"},
      {0x0003f158, "cm_stream_switch_slave_config_east_1"},
      {0x0003f15c, "cm_stream_switch_slave_config_east_2"},
      {0x0003f160, "cm_stream_switch_slave_config_east_3"},
      {0x00034008, "shim_event_generate"},
      {0x0001e400, "mm_lock8_release_nv"},
      {0x0001e3c0, "mm_lock7_acquire_nv"},
      {0x0001d18c, "mm_dma_bd12_2d_y"},
      {0x0001d188, "mm_dma_bd12_2d_x"},
      {0x0001d1f0, "mm_dma_bd15_packet"},
      {0x0001e040, "mm_lock0_acquire_nv"},
      {0x0001e540, "mm_lock10_acquire_nv"},
      {0x000140fc, "mm_timer_high"},
      {0x0003f3a4, "cm_stream_switch_slave_mem_trace_slot1"},
      {0x0003f12c, "cm_stream_switch_slave_config_south_4"},
      {0x0003f2c4, "cm_stream_switch_slave_south_5_slot1"},
      {0x0003f2c0, "cm_stream_switch_slave_south_5_slot0"},
      {0x0003f2cc, "cm_stream_switch_slave_south_5_slot3"},
      {0x0003f2c8, "cm_stream_switch_slave_south_5_slot2"},
      {0x0001d04c, "mm_dma_bd2_2d_y"},
      {0x000340d4, "cm_trace_control1"},
      {0x000340d0, "cm_trace_control0"},
      {0x0001d048, "mm_dma_bd2_2d_x"},
      {0x0003f210, "cm_stream_switch_slave_aie_core1_slot0"},
      {0x0003f214, "cm_stream_switch_slave_aie_core1_slot1"},
      {0x0003f218, "cm_stream_switch_slave_aie_core1_slot2"},
      {0x00014500, "mm_event_group_0_enable"},
      {0x0003f394, "cm_stream_switch_slave_aie_trace_slot1"},
      {0x0003f390, "cm_stream_switch_slave_aie_trace_slot0"},
      {0x0003f39c, "cm_stream_switch_slave_aie_trace_slot3"},
      {0x0003f398, "cm_stream_switch_slave_aie_trace_slot2"},
      {0x0001d10c, "mm_dma_bd8_2d_y"},
      {0x0001d108, "mm_dma_bd8_2d_x"},
      {0x0001d144, "shim_dma_s2mm_0_start_queue"},
      {0x0001e6e0, "mm_lock13_acquire_v0"},
      {0x0001e6f0, "mm_lock13_acquire_v1"},
      {0x0003f010, "cm_stream_switch_master_config_tile_ctrl"},
      {0x0001e7b0, "mm_lock15_release_v1"},
      {0x0001e7a0, "mm_lock15_release_v0"},
      {0x0001d024, "mm_dma_bd1_addr_b"},
      {0x0001d020, "mm_dma_bd1_addr_a"},
      {0x0001e360, "mm_lock6_acquire_v0"},
      {0x0001d028, "mm_dma_bd1_2d_x"},
      {0x0001d02c, "mm_dma_bd1_2d_y"},
      {0x0001d124, "mm_dma_bd9_addr_b"},
      {0x0001d120, "mm_dma_bd9_addr_a"},
      {0x0001e240, "mm_lock4_acquire_nv"},
      {0x00031024, "cm_performance_counter1"},
      {0x00031020, "cm_performance_counter0"},
      {0x0003102c, "cm_performance_counter3"},
      {0x00031028, "cm_performance_counter2"},
      {0x000340e0, "cm_trace_event0"},
      {0x000340e4, "cm_trace_event1"},
      {0x0001e080, "mm_lock1_release_nv"},
      {0x0001e1a0, "mm_lock3_release_v0"},
      {0x0003f2dc, "cm_stream_switch_slave_west_0_slot3"},
      {0x0003f2d8, "cm_stream_switch_slave_west_0_slot2"},
      {0x0003f2d4, "cm_stream_switch_slave_west_0_slot1"},
      {0x0003f2d0, "cm_stream_switch_slave_west_0_slot0"},
      {0x0001e300, "mm_lock6_release_nv"},
      {0x00034060, "cm_event_broadcast_block_west_set"},
      {0x00034080, "shim_event_broadcast_a_block_east_set"},
      {0x0001d094, "mm_dma_bd4_interleaved_state"},
      {0x0001e7e0, "mm_lock15_acquire_v0"},
      {0x00034084, "cm_event_broadcast_block_east_clr"},
      {0x0001e370, "mm_lock6_acquire_v1"},
      {0x00034064, "shim_event_broadcast_a_block_west_clr"},
      {0x0001e0e0, "mm_lock1_acquire_v0"},
      {0x0001e0f0, "mm_lock1_acquire_v1"},
      {0x0003f318, "cm_stream_switch_slave_north_0_slot2"},
      {0x0003f31c, "cm_stream_switch_slave_north_0_slot3"},
      {0x0003f310, "cm_stream_switch_slave_north_0_slot0"},
      {0x0003f314, "cm_stream_switch_slave_north_0_slot1"},
      {0x0003f2e8, "cm_stream_switch_slave_west_1_slot2"},
      {0x00011024, "mm_performance_counter1"},
      {0x00011020, "mm_performance_counter0"},
      {0x0003f2ec, "cm_stream_switch_slave_west_1_slot3"},
      {0x0003f2e0, "cm_stream_switch_slave_west_1_slot0"},
      {0x0001d000, "shim_dma_bd0_addr_low"},
      {0x0003f348, "cm_stream_switch_slave_north_3_slot2"},
      {0x0001d118, "mm_dma_bd8_control"},
      {0x00014068, "mm_event_broadcast_block_west_value"},
      {0x0001d100, "mm_dma_bd8_addr_a"},
      {0x0001d104, "mm_dma_bd8_addr_b"},
      {0x0001d0cc, "mm_dma_bd6_2d_y"},
      {0x0001d010, "mm_dma_bd0_packet"},
      {0x0001d0c8, "mm_dma_bd6_2d_x"},
      {0x0001e000, "mm_lock0_release_nv"},
      {0x0001d0ec, "mm_dma_bd7_2d_y"},
      {0x00014088, "mm_event_broadcast_block_east_value"},
      {0x0001d0b8, "mm_dma_bd5_control"},
      {0x00034000, "cm_timer_control"},
      {0x00032000, "cm_core_control"},
      {0x0001d130, "mm_dma_bd9_packet"},
      {0x0001e2e0, "mm_lock5_acquire_v0"},
      {0x0001e4e0, "mm_lock9_acquire_v0"},
      {0x0001e4f0, "mm_lock9_acquire_v1"},
      {0x00014054, "mm_event_broadcast_block_south_clr"},
      {0x00034084, "shim_event_broadcast_a_block_east_clr"},
      {0x0001e5f0, "mm_lock11_acquire_v1"},
      {0x00031084, "cm_performance_counter1_event_value"},
      {0x0001d1b0, "mm_dma_bd13_packet"},
      {0x0001df20, "mm_dma_fifo_counter"},
      {0x0001e7c0, "mm_lock15_acquire_nv"},
      {0x00036034, "cm_tile_control_packet_handler_status"},
      {0x00034030, "cm_event_broadcast8"},
      {0x00034034, "cm_event_broadcast9"},
      {0x0001e580, "mm_lock11_release_nv"},
      {0x00034020, "cm_event_broadcast4"},
      {0x0003f38c, "cm_stream_switch_slave_east_3_slot3"},
      {0x00034028, "cm_event_broadcast6"},
      {0x0003402c, "cm_event_broadcast7"},
      {0x00034010, "cm_event_broadcast0"},
      {0x00034014, "cm_event_broadcast1"},
      {0x00034018, "cm_event_broadcast2"},
      {0x0003401c, "cm_event_broadcast3"},
      {0x0001d0f4, "mm_dma_bd7_interleaved_state"},
      {0x0001e2c0, "mm_lock5_acquire_nv"},
      {0x0003f008, "cm_stream_switch_master_config_dma0"},
      {0x0003f00c, "cm_stream_switch_master_config_dma1"},
      {0x0001d1d0, "mm_dma_bd14_packet"},
      {0x0001d194, "mm_dma_bd12_interleaved_state"},
      {0x00034504, "cm_event_group_pc_enable"},
      {0x00034054, "cm_event_broadcast_block_south_clr"},
      {0x0003f034, "cm_stream_switch_master_config_west2"},
      {0x0003f038, "cm_stream_switch_master_config_west3"},
      {0x0003f02c, "cm_stream_switch_master_config_west0"},
      {0x0003f030, "cm_stream_switch_master_config_west1"},
      {0x0003f264, "cm_stream_switch_slave_fifo_1_slot1"},
      {0x0003f260, "cm_stream_switch_slave_fifo_1_slot0"},
      {0x0003f26c, "cm_stream_switch_slave_fifo_1_slot3"},
      {0x0003f268, "cm_stream_switch_slave_fifo_1_slot2"},
      {0x0001d184, "mm_dma_bd12_addr_b"},
      {0x0003420c, "shim_event_status3"},
      {0x00034208, "shim_event_status2"},
      {0x00034204, "shim_event_status1"},
      {0x00034200, "shim_event_status0"},
      {0x0001e140, "mm_lock2_acquire_nv"},
      {0x0001d1a8, "mm_dma_bd13_2d_x"},
      {0x0001d1ac, "mm_dma_bd13_2d_y"},
      {0x0003f130, "cm_stream_switch_slave_config_south_5"},
      {0x0003f3a0, "cm_stream_switch_slave_mem_trace_slot0"},
      {0x0003f3ac, "cm_stream_switch_slave_mem_trace_slot3"},
      {0x0003f3a8, "cm_stream_switch_slave_mem_trace_slot2"},
      {0x0003f120, "cm_stream_switch_slave_config_south_1"},
      {0x0003f11c, "cm_stream_switch_slave_config_south_0"},
      {0x0003f128, "cm_stream_switch_slave_config_south_3"},
      {0x0003f124, "cm_stream_switch_slave_config_south_2"},
      {0x0001d1f4, "mm_dma_bd15_interleaved_state"},
      {0x0001d160, "mm_dma_bd11_addr_a"},
      {0x0001e1b0, "mm_lock3_release_v1"},
      {0x00012110, "mm_ecc_scrubbing_event"},
      {0x0003f028, "cm_stream_switch_master_config_south3"},
      {0x0003f024, "cm_stream_switch_master_config_south2"},
      {0x0003f020, "cm_stream_switch_master_config_south1"},
      {0x0003f01c, "cm_stream_switch_master_config_south0"},
      {0x0001d074, "mm_dma_bd3_interleaved_state"},
      {0x00031084, "shim_performance_counter1_event_value"},
      {0x000340e4, "shim_trace_event1"},
      {0x0001d138, "mm_dma_bd9_control"},
      {0x000340e0, "shim_trace_event0"},
      {0x00013000, "mm_reset_control"},
      {0x00014100, "mm_watchpoint0"},
      {0x00014104, "mm_watchpoint1"},
      {0x0001421c, "mm_reserved3"},
      {0x00014218, "mm_reserved2"},
      {0x00014214, "mm_reserved1"},
      {0x00014210, "mm_reserved0"},
      {0x0001e270, "mm_lock4_acquire_v1"},
      {0x0001e260, "mm_lock4_acquire_v0"},
      {0x0001d144, "mm_dma_bd10_addr_b"},
      {0x0001d140, "mm_dma_bd10_addr_a"},
      {0x0001d0b4, "mm_dma_bd5_interleaved_state"},
      {0x00014038, "mm_event_broadcast10"},
      {0x0001403c, "mm_event_broadcast11"},
      {0x00014040, "mm_event_broadcast12"},
      {0x00014044, "mm_event_broadcast13"},
      {0x00014048, "mm_event_broadcast14"},
      {0x0001404c, "mm_event_broadcast15"},
      {0x0001d00c, "mm_dma_bd0_2d_y"},
      {0x0001d008, "mm_dma_bd0_2d_x"},
      {0x0003f168, "cm_stream_switch_slave_config_mem_trace"},
      {0x00034514, "cm_event_group_errors1_enable"},
      {0x0001e1c0, "mm_lock3_acquire_nv"},
      {0x0001d044, "mm_dma_bd2_addr_b"},
      {0x0001df10, "mm_dma_mm2s_status"},
      {0x0001d040, "mm_dma_bd2_addr_a"},
      {0x00031008, "cm_performance_control2"},
      {0x0001d054, "mm_dma_bd2_interleaved_state"},
      {0x0003f2ac, "cm_stream_switch_slave_south_3_slot3"},
      {0x0003f2a8, "cm_stream_switch_slave_south_3_slot2"},
      {0x0003f2a4, "cm_stream_switch_slave_south_3_slot1"},
      {0x0003f2a0, "cm_stream_switch_slave_south_3_slot0"},
      {0x0001e5c0, "mm_lock11_acquire_nv"},
      {0x0001e460, "mm_lock8_acquire_v0"},
      {0x000140d4, "mm_trace_control1"},
      {0x000140d0, "mm_trace_control0"},
      {0x0001e0c0, "mm_lock1_acquire_nv"},
      {0x0001e3e0, "mm_lock7_acquire_v0"},
      {0x0001e3b0, "mm_lock7_release_v1"},
      {0x0001e3a0, "mm_lock7_release_v0"},
      {0x00034050, "shim_event_broadcast_a_block_south_set"},
      {0x00034078, "shim_event_broadcast_a_block_north_value"},
      {0x00034064, "cm_event_broadcast_block_west_clr"},
      {0x00034054, "shim_event_broadcast_a_block_south_clr"},
      {0x0001e3f0, "mm_lock7_acquire_v1"},
      {0x0001d0e0, "mm_dma_bd7_addr_a"},
      {0x0001d0a4, "mm_dma_bd5_addr_b"},
      {0x0001d0a0, "mm_dma_bd5_addr_a"},
      {0x0001d0e4, "mm_dma_bd7_addr_b"},
      {0x000140f0, "mm_timer_trig_event_low_value"},
      {0x0001d1b8, "mm_dma_bd13_control"},
      {0x0003451c, "cm_event_group_broadcast_enable"},
      {0x00014084, "mm_event_broadcast_block_east_clr"},
      {0x00014064, "mm_event_broadcast_block_west_clr"},
      {0x0001e520, "mm_lock10_release_v0"},
      {0x0001e530, "mm_lock10_release_v1"},
      {0x0001e440, "mm_lock8_acquire_nv"},
      {0x0001d090, "mm_dma_bd4_packet"},
      {0x0001d000, "mm_dma_bd0_addr_a"},
      {0x0001d004, "mm_dma_bd0_addr_b"},
      {0x0003404c, "shim_event_broadcast_a_15"},
      {0x0003108c, "cm_performance_counter3_event_value"},
      {0x00011080, "mm_performance_counter0_event_value"},
      {0x0001ef00, "mm_all_lock_state_value"},
      {0x0003f108, "cm_stream_switch_slave_config_dma_0"},
      {0x0003f10c, "cm_stream_switch_slave_config_dma_1"},
      {0x00014510, "mm_event_group_memory_conflict_enable"},
      {0x00034068, "cm_event_broadcast_block_west_value"},
      {0x00034034, "shim_event_broadcast_a_9"},
      {0x00034030, "shim_event_broadcast_a_8"},
      {0x0003402c, "shim_event_broadcast_a_7"},
      {0x00034028, "shim_event_broadcast_a_6"},
      {0x00034024, "shim_event_broadcast_a_5"},
      {0x00034020, "shim_event_broadcast_a_4"},
      {0x0003401c, "shim_event_broadcast_a_3"},
      {0x00034018, "shim_event_broadcast_a_2"},
      {0x0001d198, "mm_dma_bd12_control"},
      {0x00034010, "shim_event_broadcast_a_0"},
      {0x0001d170, "mm_dma_bd11_packet"},
      {0x0001d008, "shim_dma_bd0_control"},
      {0x00036040, "cm_tile_clock_control"},
      {0x0001e220, "mm_lock4_release_v0"},
      {0x0001e230, "mm_lock4_release_v1"},
      {0x0001d014, "mm_dma_bd0_interleaved_state"},
      {0x0003f104, "cm_stream_switch_slave_config_aie_core1"},
      {0x0001d0c0, "mm_dma_bd6_addr_a"},
      {0x000340d8, "shim_trace_status"},
      {0x000340d8, "cm_trace_status"},
      {0x00012120, "mm_ecc_failing_address"},
      {0x0003f250, "cm_stream_switch_slave_fifo_0_slot0"},
      {0x0003f254, "cm_stream_switch_slave_fifo_0_slot1"},
      {0x0003f258, "cm_stream_switch_slave_fifo_0_slot2"},
      {0x0003f25c, "cm_stream_switch_slave_fifo_0_slot3"},
      {0x0003200c, "cm_reset_event"},
      {0x0001e700, "mm_lock14_release_nv"},
      {0x000140f8, "mm_timer_low"},
      {0x0001e020, "mm_lock0_release_v0"},
      {0x0001e030, "mm_lock0_release_v1"},
      {0x00014030, "mm_event_broadcast8"},
      {0x00014034, "mm_event_broadcast9"},
      {0x00014028, "mm_event_broadcast6"},
      {0x0001402c, "mm_event_broadcast7"},
      {0x00014020, "mm_event_broadcast4"},
      {0x00014024, "mm_event_broadcast5"},
      {0x00014018, "mm_event_broadcast2"},
      {0x0001401c, "mm_event_broadcast3"},
      {0x00014010, "mm_event_broadcast0"},
      {0x00014014, "mm_event_broadcast1"},
      {0x0001e470, "mm_lock8_acquire_v1"},
      {0x00012124, "mm_parity_failing_address"},
      {0x0003f2b0, "cm_stream_switch_slave_south_4_slot0"},
      {0x0003f2b4, "cm_stream_switch_slave_south_4_slot1"},
      {0x0003f2b8, "cm_stream_switch_slave_south_4_slot2"},
      {0x0003f2bc, "cm_stream_switch_slave_south_4_slot3"},
      {0x0003f238, "cm_stream_switch_slave_dma_1_slot2"},
      {0x0003f23c, "cm_stream_switch_slave_dma_1_slot3"},
      {0x0003f230, "cm_stream_switch_slave_dma_1_slot0"},
      {0x0003f234, "cm_stream_switch_slave_dma_1_slot1"},
      {0x0001d034, "mm_dma_bd1_interleaved_state"},
      {0x0001d164, "mm_dma_bd11_addr_b"},
      {0x00034008, "cm_event_generate"},
      {0x0001de18, "mm_dma_mm2s_1_ctrl"},
      {0x0001d178, "mm_dma_bd11_control"},
      {0x00031000, "cm_performance_control0"},
      {0x00031004, "cm_performance_control1"},
      {0x00030470, "cm_mc1"},
      {0x00030460, "cm_mc0"},
      {0x0001d0a8, "mm_dma_bd5_2d_x"},
      {0x0001d0ac, "mm_dma_bd5_2d_y"},
      {0x00034044, "shim_event_broadcast_a_13"},
      {0x00034040, "shim_event_broadcast_a_12"},
      {0x0003403c, "shim_event_broadcast_a_11"},
      {0x00034038, "shim_event_broadcast_a_10"},
      {0x0001e480, "mm_lock9_release_nv"},
      {0x0001de00, "mm_dma_s2mm_0_ctrl"},
      {0x00034048, "shim_event_broadcast_a_14"},
      {0x00012000, "mm_checkbit_error_generation"},
      {0x0001d1f8, "mm_dma_bd15_control"},
      {0x00034404, "cm_combo_event_control"},
      {0x0001e640, "mm_lock12_acquire_nv"},
      {0x00011000, "mm_performance_control0"},
      {0x00011008, "mm_performance_control1"},
      {0x00032018, "cm_debug_control2"},
      {0x00032014, "cm_debug_control1"},
      {0x00032010, "cm_debug_control0"},
      {0x0003f13c, "cm_stream_switch_slave_config_west_2"},
      {0x0003f140, "cm_stream_switch_slave_config_west_3"},
      {0x0003f134, "cm_stream_switch_slave_config_west_0"},
      {0x0003f138, "cm_stream_switch_slave_config_west_1"},
      {0x00014518, "mm_event_group_broadcast_enable"},
      {0x00034070, "shim_event_broadcast_a_block_north_set"},
      {0x0001e600, "mm_lock12_release_nv"},
      {0x00030440, "cm_md0"},
      {0x00030450, "cm_md1"},
      {0x0001e2f0, "mm_lock5_acquire_v1"},
      {0x0003ff04, "shim_stream_switch_event_port_selection_1"},
      {0x0001e1e0, "mm_lock3_acquire_v0"},
      {0x0001d050, "mm_dma_bd2_packet"},
      {0x0001e1f0, "mm_lock3_acquire_v1"},
      {0x0001d0c4, "mm_dma_bd6_addr_b"},
      {0x000340f8, "shim_timer_low"},
      {0x0001e200, "mm_lock4_release_nv"},
      {0x0003f32c, "cm_stream_switch_slave_north_1_slot3"},
      {0x0003f328, "cm_stream_switch_slave_north_1_slot2"},
      {0x0003f324, "cm_stream_switch_slave_north_1_slot1"},
      {0x0003f320, "cm_stream_switch_slave_north_1_slot0"},
      {0x00034088, "shim_event_broadcast_a_block_east_value"},
      {0x00014078, "mm_event_broadcast_block_north_value"},
      {0x00014080, "mm_event_broadcast_block_east_set"},
      {0x00034518, "cm_event_group_stream_switch_enable"},
      {0x0003f300, "cm_stream_switch_slave_west_3_slot0"},
      {0x0003f304, "cm_stream_switch_slave_west_3_slot1"},
      {0x0003f308, "cm_stream_switch_slave_west_3_slot2"},
      {0x0003f30c, "cm_stream_switch_slave_west_3_slot3"},
      {0x0003201c, "cm_debug_status"},
      {0x0001e6a0, "mm_lock13_release_v0"},
      {0x0001d154, "mm_dma_bd10_interleaved_state"},
      {0x000340fc, "cm_timer_high"},
      {0x00032028, "cm_pc_event2"},
      {0x00032024, "cm_pc_event1"},
      {0x00032020, "cm_pc_event0"},
      {0x0003f000, "cm_stream_switch_master_config_aie_core0"},
      {0x0003f004, "cm_stream_switch_master_config_aie_core1"},
      {0x0001e120, "mm_lock2_release_v0"},
      {0x0001e130, "mm_lock2_release_v1"},
      {0x0001d018, "mm_dma_bd0_control"},
      {0x0001d030, "mm_dma_bd1_packet"},
      {0x14, "shim_dma_bd_step_size"},
      {0x0001e740, "mm_lock14_acquire_nv"},
      {0x0001e380, "mm_lock7_release_nv"},
      {0x0003ff00, "shim_stream_switch_event_port_selection_0"},
      {0x0001de1c, "mm_dma_mm2s_1_start_queue"},
      {0x00034060, "shim_event_broadcast_a_block_west_set"},
      {0x00032110, "cm_ecc_scrubbing_event"},
      {0x0001e720, "mm_lock14_release_v0"},
      {0x0001e730, "mm_lock14_release_v1"},
      {0x0003f344, "cm_stream_switch_slave_north_3_slot1"},
      {0x0003f340, "cm_stream_switch_slave_north_3_slot0"},
      {0x0003f34c, "cm_stream_switch_slave_north_3_slot3"},
      {0x0003f2e4, "cm_stream_switch_slave_west_1_slot1"},
      {0x0001e500, "mm_lock10_release_nv"},
      {0x0003f118, "cm_stream_switch_slave_config_fifo_1"},
      {0x0003f114, "cm_stream_switch_slave_config_fifo_0"},
      {0x000340d0, "shim_trace_control0"},
      {0x000340d4, "shim_trace_control1"},
      {0x0001e5e0, "mm_lock11_acquire_v0"},
      {0x0001d1d8, "mm_dma_bd14_control"},
      {0x00036030, "cm_tile_control"},
      {0x00031080, "cm_performance_counter0_event_value"},
      {0x00034074, "cm_event_broadcast_block_north_clr"},
      {0x00032008, "cm_enable_events"},
      {0x00030280, "cm_program_counter"},
      {0x0001d1e0, "mm_dma_bd15_addr_a"},
      {0x0001d1e4, "mm_dma_bd15_addr_b"},
      {0x0001d128, "mm_dma_bd9_2d_x"},
      {0x0001d12c, "mm_dma_bd9_2d_y"},
      {0x000340f8, "cm_timer_low"},
      {0x0003f270, "cm_stream_switch_slave_south_0_slot0"},
      {0x0003f274, "cm_stream_switch_slave_south_0_slot1"},
      {0x0003f278, "cm_stream_switch_slave_south_0_slot2"},
      {0x0003f27c, "cm_stream_switch_slave_south_0_slot3"},
      {0x0001e670, "mm_lock12_acquire_v1"},
      {0x0001e660, "mm_lock12_acquire_v0"},
      {0x00014504, "mm_event_group_watchpoint_enable"},
      {0x0001e570, "mm_lock10_acquire_v1"},
      {0x0003f330, "cm_stream_switch_slave_north_2_slot0"},
      {0x0001d098, "mm_dma_bd4_control"},
      {0x00032004, "cm_core_status"},
      {0x00014f00, "shim_all_lock_state_value"},
      {0x0001d0e8, "mm_dma_bd7_2d_x"},
      {0x0001de08, "mm_dma_s2mm_1_ctrl"},
      {0x0001d0f8, "mm_dma_bd7_control"},
      {0x0001d010, "shim_dma_bd0_packet"},
      {0x0001d1e8, "mm_dma_bd15_2d_x"},
      {0x0001d1ec, "mm_dma_bd15_2d_y"},
      {0x0001de04, "mm_dma_s2mm_0_start_queue"},
      {0x0001e620, "mm_lock12_release_v0"},
      {0x0001e630, "mm_lock12_release_v1"},
      {0x000340fc, "shim_timer_high"},
      {0x00034520, "cm_event_group_user_event_enable"},
      {0x0003f22c, "cm_stream_switch_slave_dma_0_slot3"},
      {0x0003f228, "cm_stream_switch_slave_dma_0_slot2"},
      {0x0003f224, "cm_stream_switch_slave_dma_0_slot1"},
      {0x0003f220, "cm_stream_switch_slave_dma_0_slot0"}
    };
  }
};

/*************************************************************************************
 AIE2 Registers
 *************************************************************************************/
class AIE2UsedRegisters : public UsedRegisters {
public:
  AIE2UsedRegisters() {
    populateRegNameToValueMap();
    populateRegValueToNameMap();
  }
  ~AIE2UsedRegisters() = default;

  void populateProfileRegisters() {
    // Core modules
    core_addresses.emplace(aie2::cm_performance_control0);
    core_addresses.emplace(aie2::cm_performance_control1);
    core_addresses.emplace(aie2::cm_performance_control2);
    core_addresses.emplace(aie2::cm_performance_counter0);
    core_addresses.emplace(aie2::cm_performance_counter1);
    core_addresses.emplace(aie2::cm_performance_counter2);
    core_addresses.emplace(aie2::cm_performance_counter3);
    core_addresses.emplace(aie2::cm_performance_counter0_event_value);
    core_addresses.emplace(aie2::cm_performance_counter1_event_value);
    core_addresses.emplace(aie2::cm_performance_counter2_event_value);
    core_addresses.emplace(aie2::cm_performance_counter3_event_value);
    core_addresses.emplace(aie2::cm_ecc_scrubbing_event);

    // Memory modules
    memory_addresses.emplace(aie2::mm_performance_control0);
    memory_addresses.emplace(aie2::mm_performance_control1);
    memory_addresses.emplace(aie2::mm_performance_counter0);
    memory_addresses.emplace(aie2::mm_performance_counter1);
    memory_addresses.emplace(aie2::mm_performance_counter0_event_value);
    memory_addresses.emplace(aie2::mm_performance_counter1_event_value);

    // Interface tiles
    interface_addresses.emplace(aie2::shim_performance_control0);
    interface_addresses.emplace(aie2::shim_performance_control1);
    interface_addresses.emplace(aie2::shim_performance_counter0);
    interface_addresses.emplace(aie2::shim_performance_counter1);
    interface_addresses.emplace(aie2::shim_performance_counter0_event_value);
    interface_addresses.emplace(aie2::shim_performance_counter1_event_value);

    // Memory tiles
    memory_tile_addresses.emplace(aie2::mem_performance_control0);
    memory_tile_addresses.emplace(aie2::mem_performance_control1);
    memory_tile_addresses.emplace(aie2::mem_performance_control2);
    memory_tile_addresses.emplace(aie2::mem_performance_counter0);
    memory_tile_addresses.emplace(aie2::mem_performance_counter1);
    memory_tile_addresses.emplace(aie2::mem_performance_counter2);
    memory_tile_addresses.emplace(aie2::mem_performance_counter3);
    memory_tile_addresses.emplace(aie2::mem_performance_counter0_event_value);
    memory_tile_addresses.emplace(aie2::mem_performance_counter1_event_value);
    memory_tile_addresses.emplace(aie2::mem_performance_counter2_event_value);
    memory_tile_addresses.emplace(aie2::mem_performance_counter3_event_value);
  }

  void populateTraceRegisters() {
    // Core modules
    core_addresses.emplace(aie2::cm_core_status);
    core_addresses.emplace(aie2::cm_trace_control0);
    core_addresses.emplace(aie2::cm_trace_control1);
    core_addresses.emplace(aie2::cm_trace_status);
    core_addresses.emplace(aie2::cm_trace_event0);
    core_addresses.emplace(aie2::cm_trace_event1);
    core_addresses.emplace(aie2::cm_event_status0);
    core_addresses.emplace(aie2::cm_event_status1);
    core_addresses.emplace(aie2::cm_event_status2);
    core_addresses.emplace(aie2::cm_event_status3);
    core_addresses.emplace(aie2::cm_event_broadcast0);
    core_addresses.emplace(aie2::cm_event_broadcast1);
    core_addresses.emplace(aie2::cm_event_broadcast2);
    core_addresses.emplace(aie2::cm_event_broadcast3);
    core_addresses.emplace(aie2::cm_event_broadcast4);
    core_addresses.emplace(aie2::cm_event_broadcast5);
    core_addresses.emplace(aie2::cm_event_broadcast6);
    core_addresses.emplace(aie2::cm_event_broadcast7);
    core_addresses.emplace(aie2::cm_event_broadcast8);
    core_addresses.emplace(aie2::cm_event_broadcast9);
    core_addresses.emplace(aie2::cm_event_broadcast10);
    core_addresses.emplace(aie2::cm_event_broadcast11);
    core_addresses.emplace(aie2::cm_event_broadcast12);
    core_addresses.emplace(aie2::cm_event_broadcast13);
    core_addresses.emplace(aie2::cm_event_broadcast14);
    core_addresses.emplace(aie2::cm_event_broadcast15);
    core_addresses.emplace(aie2::cm_event_broadcast_block_south_value);
    core_addresses.emplace(aie2::cm_event_broadcast_block_west_value);
    core_addresses.emplace(aie2::cm_event_broadcast_block_north_value);
    core_addresses.emplace(aie2::cm_event_broadcast_block_east_value);
    core_addresses.emplace(aie2::cm_timer_trig_event_low_value);
    core_addresses.emplace(aie2::cm_timer_trig_event_high_value);
    core_addresses.emplace(aie2::cm_timer_low);
    core_addresses.emplace(aie2::cm_timer_high);
    core_addresses.emplace(aie2::cm_edge_detection_event_control);
    core_addresses.emplace(aie2::cm_stream_switch_event_port_selection_0);
    core_addresses.emplace(aie2::cm_stream_switch_event_port_selection_1);
    
    // Memory modules
    memory_addresses.emplace(aie2::mm_trace_control0);
    memory_addresses.emplace(aie2::mm_trace_control1);
    memory_addresses.emplace(aie2::mm_trace_status);
    memory_addresses.emplace(aie2::mm_trace_event0);
    memory_addresses.emplace(aie2::mm_trace_event1);
    memory_addresses.emplace(aie2::mm_event_status0);
    memory_addresses.emplace(aie2::mm_event_status1);
    memory_addresses.emplace(aie2::mm_event_status2);
    memory_addresses.emplace(aie2::mm_event_status3);
    memory_addresses.emplace(aie2::mm_event_broadcast0);
    memory_addresses.emplace(aie2::mm_event_broadcast1);
    memory_addresses.emplace(aie2::mm_event_broadcast2);
    memory_addresses.emplace(aie2::mm_event_broadcast3);
    memory_addresses.emplace(aie2::mm_event_broadcast4);
    memory_addresses.emplace(aie2::mm_event_broadcast5);
    memory_addresses.emplace(aie2::mm_event_broadcast6);
    memory_addresses.emplace(aie2::mm_event_broadcast7);
    memory_addresses.emplace(aie2::mm_event_broadcast8);
    memory_addresses.emplace(aie2::mm_event_broadcast9);
    memory_addresses.emplace(aie2::mm_event_broadcast10);
    memory_addresses.emplace(aie2::mm_event_broadcast11);
    memory_addresses.emplace(aie2::mm_event_broadcast12);
    memory_addresses.emplace(aie2::mm_event_broadcast13);
    memory_addresses.emplace(aie2::mm_event_broadcast14);
    memory_addresses.emplace(aie2::mm_event_broadcast15);

    // Interface tiles
    interface_addresses.emplace(aie2::shim_trace_control0);
    interface_addresses.emplace(aie2::shim_trace_control1);
    interface_addresses.emplace(aie2::shim_trace_status);
    interface_addresses.emplace(aie2::shim_trace_event0);
    interface_addresses.emplace(aie2::shim_trace_event1);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_0);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_1);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_2);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_3);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_4);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_5);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_6);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_7);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_8);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_9);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_10);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_11);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_12);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_13);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_14);
    interface_addresses.emplace(aie2::shim_event_broadcast_a_15);
    interface_addresses.emplace(aie2::shim_event_status0);
    interface_addresses.emplace(aie2::shim_event_status1);
    interface_addresses.emplace(aie2::shim_event_status2);
    interface_addresses.emplace(aie2::shim_event_status3);
    interface_addresses.emplace(aie2::shim_stream_switch_event_port_selection_0);
    interface_addresses.emplace(aie2::shim_stream_switch_event_port_selection_1);

    // Memory tiles
    memory_tile_addresses.emplace(aie2::mem_trace_control0);
    memory_tile_addresses.emplace(aie2::mem_trace_control1);
    memory_tile_addresses.emplace(aie2::mem_trace_status);
    memory_tile_addresses.emplace(aie2::mem_trace_event0);
    memory_tile_addresses.emplace(aie2::mem_trace_event1);
    memory_tile_addresses.emplace(aie2::mem_dma_event_channel_selection);
    memory_tile_addresses.emplace(aie2::mem_edge_detection_event_control);
    memory_tile_addresses.emplace(aie2::mem_stream_switch_event_port_selection_0);
    memory_tile_addresses.emplace(aie2::mem_stream_switch_event_port_selection_1);
    memory_tile_addresses.emplace(aie2::mem_event_broadcast0);
    memory_tile_addresses.emplace(aie2::mem_event_status0);
    memory_tile_addresses.emplace(aie2::mem_event_status1);
    memory_tile_addresses.emplace(aie2::mem_event_status2);
    memory_tile_addresses.emplace(aie2::mem_event_status3);
    memory_tile_addresses.emplace(aie2::mem_event_status4);
    memory_tile_addresses.emplace(aie2::mem_event_status5);
  }

  void populateRegNameToValueMap() {
    regNameToValues = {
      {"cm_program_counter", aie2::cm_program_counter},
      {"cm_performance_control0", aie2::cm_performance_control0},
      {"cm_performance_control1", aie2::cm_performance_control1},
      {"cm_performance_control2", aie2::cm_performance_control2},
      {"cm_performance_counter0", aie2::cm_performance_counter0},
      {"cm_performance_counter1", aie2::cm_performance_counter1},
      {"cm_performance_counter2", aie2::cm_performance_counter2},
      {"cm_performance_counter3", aie2::cm_performance_counter3},
      {"cm_performance_counter0_event_value", aie2::cm_performance_counter0_event_value},
      {"cm_performance_counter1_event_value", aie2::cm_performance_counter1_event_value},
      {"cm_performance_counter2_event_value", aie2::cm_performance_counter2_event_value},
      {"cm_performance_counter3_event_value", aie2::cm_performance_counter3_event_value},
      {"cm_core_control", aie2::cm_core_control},
      {"cm_core_status", aie2::cm_core_status},
      {"cm_enable_events", aie2::cm_enable_events},
      {"cm_reset_event", aie2::cm_reset_event},
      {"cm_debug_control0", aie2::cm_debug_control0},
      {"cm_debug_control1", aie2::cm_debug_control1},
      {"cm_debug_control2", aie2::cm_debug_control2},
      {"cm_debug_status", aie2::cm_debug_status},
      {"cm_pc_event0", aie2::cm_pc_event0},
      {"cm_pc_event1", aie2::cm_pc_event1},
      {"cm_pc_event2", aie2::cm_pc_event2},
      {"cm_pc_event3", aie2::cm_pc_event3},
      {"cm_error_halt_control", aie2::cm_error_halt_control},
      {"cm_error_halt_event", aie2::cm_error_halt_event},
      {"cm_core_processor_bus", aie2::cm_core_processor_bus},
      {"cm_ecc_scrubbing_event", aie2::cm_ecc_scrubbing_event},
      {"cm_timer_control", aie2::cm_timer_control},
      {"cm_event_generate", aie2::cm_event_generate},
      {"cm_event_broadcast0", aie2::cm_event_broadcast0},
      {"cm_event_broadcast1", aie2::cm_event_broadcast1},
      {"cm_event_broadcast2", aie2::cm_event_broadcast2},
      {"cm_event_broadcast3", aie2::cm_event_broadcast3},
      {"cm_event_broadcast4", aie2::cm_event_broadcast4},
      {"cm_event_broadcast5", aie2::cm_event_broadcast5},
      {"cm_event_broadcast6", aie2::cm_event_broadcast6},
      {"cm_event_broadcast7", aie2::cm_event_broadcast7},
      {"cm_event_broadcast8", aie2::cm_event_broadcast8},
      {"cm_event_broadcast9", aie2::cm_event_broadcast9},
      {"cm_event_broadcast10", aie2::cm_event_broadcast10},
      {"cm_event_broadcast11", aie2::cm_event_broadcast11},
      {"cm_event_broadcast12", aie2::cm_event_broadcast12},
      {"cm_event_broadcast13", aie2::cm_event_broadcast13},
      {"cm_event_broadcast14", aie2::cm_event_broadcast14},
      {"cm_event_broadcast15", aie2::cm_event_broadcast15},
      {"cm_event_broadcast_block_south_set", aie2::cm_event_broadcast_block_south_set},
      {"cm_event_broadcast_block_south_clr", aie2::cm_event_broadcast_block_south_clr},
      {"cm_event_broadcast_block_south_value", aie2::cm_event_broadcast_block_south_value},
      {"cm_event_broadcast_block_west_set", aie2::cm_event_broadcast_block_west_set},
      {"cm_event_broadcast_block_west_clr", aie2::cm_event_broadcast_block_west_clr},
      {"cm_event_broadcast_block_west_value", aie2::cm_event_broadcast_block_west_value},
      {"cm_event_broadcast_block_north_set", aie2::cm_event_broadcast_block_north_set},
      {"cm_event_broadcast_block_north_clr", aie2::cm_event_broadcast_block_north_clr},
      {"cm_event_broadcast_block_north_value", aie2::cm_event_broadcast_block_north_value},
      {"cm_event_broadcast_block_east_set", aie2::cm_event_broadcast_block_east_set},
      {"cm_event_broadcast_block_east_clr", aie2::cm_event_broadcast_block_east_clr},
      {"cm_event_broadcast_block_east_value", aie2::cm_event_broadcast_block_east_value},
      {"cm_trace_control0", aie2::cm_trace_control0},
      {"cm_trace_control1", aie2::cm_trace_control1},
      {"cm_trace_status", aie2::cm_trace_status},
      {"cm_trace_event0", aie2::cm_trace_event0},
      {"cm_trace_event1", aie2::cm_trace_event1},
      {"cm_timer_trig_event_low_value", aie2::cm_timer_trig_event_low_value},
      {"cm_timer_trig_event_high_value", aie2::cm_timer_trig_event_high_value},
      {"cm_timer_low", aie2::cm_timer_low},
      {"cm_timer_high", aie2::cm_timer_high},
      {"cm_event_status0", aie2::cm_event_status0},
      {"cm_event_status1", aie2::cm_event_status1},
      {"cm_event_status2", aie2::cm_event_status2},
      {"cm_event_status3", aie2::cm_event_status3},
      {"cm_combo_event_inputs", aie2::cm_combo_event_inputs},
      {"cm_combo_event_control", aie2::cm_combo_event_control},
      {"cm_edge_detection_event_control", aie2::cm_edge_detection_event_control},
      {"cm_event_group_0_enable", aie2::cm_event_group_0_enable},
      {"cm_event_group_pc_enable", aie2::cm_event_group_pc_enable},
      {"cm_event_group_core_stall_enable", aie2::cm_event_group_core_stall_enable},
      {"cm_event_group_core_program_flow_enable", aie2::cm_event_group_core_program_flow_enable},
      {"cm_event_group_errors0_enable", aie2::cm_event_group_errors0_enable},
      {"cm_event_group_errors1_enable", aie2::cm_event_group_errors1_enable},
      {"cm_event_group_stream_switch_enable", aie2::cm_event_group_stream_switch_enable},
      {"cm_event_group_broadcast_enable", aie2::cm_event_group_broadcast_enable},
      {"cm_event_group_user_event_enable", aie2::cm_event_group_user_event_enable},
      {"cm_tile_control", aie2::cm_tile_control},
      {"cm_accumulator_control", aie2::cm_accumulator_control},
      {"cm_memory_control", aie2::cm_memory_control},
      {"cm_stream_switch_master_config_aie_core0", aie2::cm_stream_switch_master_config_aie_core0},
      {"cm_stream_switch_master_config_dma0", aie2::cm_stream_switch_master_config_dma0},
      {"cm_stream_switch_master_config_dma1", aie2::cm_stream_switch_master_config_dma1},
      {"cm_stream_switch_master_config_tile_ctrl", aie2::cm_stream_switch_master_config_tile_ctrl},
      {"cm_stream_switch_master_config_fifo0", aie2::cm_stream_switch_master_config_fifo0},
      {"cm_stream_switch_master_config_south0", aie2::cm_stream_switch_master_config_south0},
      {"cm_stream_switch_master_config_south1", aie2::cm_stream_switch_master_config_south1},
      {"cm_stream_switch_master_config_south2", aie2::cm_stream_switch_master_config_south2},
      {"cm_stream_switch_master_config_south3", aie2::cm_stream_switch_master_config_south3},
      {"cm_stream_switch_master_config_west0", aie2::cm_stream_switch_master_config_west0},
      {"cm_stream_switch_master_config_west1", aie2::cm_stream_switch_master_config_west1},
      {"cm_stream_switch_master_config_west2", aie2::cm_stream_switch_master_config_west2},
      {"cm_stream_switch_master_config_west3", aie2::cm_stream_switch_master_config_west3},
      {"cm_stream_switch_master_config_north0", aie2::cm_stream_switch_master_config_north0},
      {"cm_stream_switch_master_config_north1", aie2::cm_stream_switch_master_config_north1},
      {"cm_stream_switch_master_config_north2", aie2::cm_stream_switch_master_config_north2},
      {"cm_stream_switch_master_config_north3", aie2::cm_stream_switch_master_config_north3},
      {"cm_stream_switch_master_config_north4", aie2::cm_stream_switch_master_config_north4},
      {"cm_stream_switch_master_config_north5", aie2::cm_stream_switch_master_config_north5},
      {"cm_stream_switch_master_config_east0", aie2::cm_stream_switch_master_config_east0},
      {"cm_stream_switch_master_config_east1", aie2::cm_stream_switch_master_config_east1},
      {"cm_stream_switch_master_config_east2", aie2::cm_stream_switch_master_config_east2},
      {"cm_stream_switch_master_config_east3", aie2::cm_stream_switch_master_config_east3},
      {"cm_stream_switch_slave_config_aie_core0", aie2::cm_stream_switch_slave_config_aie_core0},
      {"cm_stream_switch_slave_config_dma_0", aie2::cm_stream_switch_slave_config_dma_0},
      {"cm_stream_switch_slave_config_dma_1", aie2::cm_stream_switch_slave_config_dma_1},
      {"cm_stream_switch_slave_config_tile_ctrl", aie2::cm_stream_switch_slave_config_tile_ctrl},
      {"cm_stream_switch_slave_config_fifo_0", aie2::cm_stream_switch_slave_config_fifo_0},
      {"cm_stream_switch_slave_config_south_0", aie2::cm_stream_switch_slave_config_south_0},
      {"cm_stream_switch_slave_config_south_1", aie2::cm_stream_switch_slave_config_south_1},
      {"cm_stream_switch_slave_config_south_2", aie2::cm_stream_switch_slave_config_south_2},
      {"cm_stream_switch_slave_config_south_3", aie2::cm_stream_switch_slave_config_south_3},
      {"cm_stream_switch_slave_config_south_4", aie2::cm_stream_switch_slave_config_south_4},
      {"cm_stream_switch_slave_config_south_5", aie2::cm_stream_switch_slave_config_south_5},
      {"cm_stream_switch_slave_config_west_0", aie2::cm_stream_switch_slave_config_west_0},
      {"cm_stream_switch_slave_config_west_1", aie2::cm_stream_switch_slave_config_west_1},
      {"cm_stream_switch_slave_config_west_2", aie2::cm_stream_switch_slave_config_west_2},
      {"cm_stream_switch_slave_config_west_3", aie2::cm_stream_switch_slave_config_west_3},
      {"cm_stream_switch_slave_config_north_0", aie2::cm_stream_switch_slave_config_north_0},
      {"cm_stream_switch_slave_config_north_1", aie2::cm_stream_switch_slave_config_north_1},
      {"cm_stream_switch_slave_config_north_2", aie2::cm_stream_switch_slave_config_north_2},
      {"cm_stream_switch_slave_config_north_3", aie2::cm_stream_switch_slave_config_north_3},
      {"cm_stream_switch_slave_config_east_0", aie2::cm_stream_switch_slave_config_east_0},
      {"cm_stream_switch_slave_config_east_1", aie2::cm_stream_switch_slave_config_east_1},
      {"cm_stream_switch_slave_config_east_2", aie2::cm_stream_switch_slave_config_east_2},
      {"cm_stream_switch_slave_config_east_3", aie2::cm_stream_switch_slave_config_east_3},
      {"cm_stream_switch_slave_config_aie_trace", aie2::cm_stream_switch_slave_config_aie_trace},
      {"cm_stream_switch_slave_config_mem_trace", aie2::cm_stream_switch_slave_config_mem_trace},
      {"cm_stream_switch_slave_aie_core0_slot0", aie2::cm_stream_switch_slave_aie_core0_slot0},
      {"cm_stream_switch_slave_aie_core0_slot1", aie2::cm_stream_switch_slave_aie_core0_slot1},
      {"cm_stream_switch_slave_aie_core0_slot2", aie2::cm_stream_switch_slave_aie_core0_slot2},
      {"cm_stream_switch_slave_aie_core0_slot3", aie2::cm_stream_switch_slave_aie_core0_slot3},
      {"cm_stream_switch_slave_dma_0_slot0", aie2::cm_stream_switch_slave_dma_0_slot0},
      {"cm_stream_switch_slave_dma_0_slot1", aie2::cm_stream_switch_slave_dma_0_slot1},
      {"cm_stream_switch_slave_dma_0_slot2", aie2::cm_stream_switch_slave_dma_0_slot2},
      {"cm_stream_switch_slave_dma_0_slot3", aie2::cm_stream_switch_slave_dma_0_slot3},
      {"cm_stream_switch_slave_dma_1_slot0", aie2::cm_stream_switch_slave_dma_1_slot0},
      {"cm_stream_switch_slave_dma_1_slot1", aie2::cm_stream_switch_slave_dma_1_slot1},
      {"cm_stream_switch_slave_dma_1_slot2", aie2::cm_stream_switch_slave_dma_1_slot2},
      {"cm_stream_switch_slave_dma_1_slot3", aie2::cm_stream_switch_slave_dma_1_slot3},
      {"cm_stream_switch_slave_tile_ctrl_slot0", aie2::cm_stream_switch_slave_tile_ctrl_slot0},
      {"cm_stream_switch_slave_tile_ctrl_slot1", aie2::cm_stream_switch_slave_tile_ctrl_slot1},
      {"cm_stream_switch_slave_tile_ctrl_slot2", aie2::cm_stream_switch_slave_tile_ctrl_slot2},
      {"cm_stream_switch_slave_tile_ctrl_slot3", aie2::cm_stream_switch_slave_tile_ctrl_slot3},
      {"cm_stream_switch_slave_fifo_0_slot0", aie2::cm_stream_switch_slave_fifo_0_slot0},
      {"cm_stream_switch_slave_fifo_0_slot1", aie2::cm_stream_switch_slave_fifo_0_slot1},
      {"cm_stream_switch_slave_fifo_0_slot2", aie2::cm_stream_switch_slave_fifo_0_slot2},
      {"cm_stream_switch_slave_fifo_0_slot3", aie2::cm_stream_switch_slave_fifo_0_slot3},
      {"cm_stream_switch_slave_south_0_slot0", aie2::cm_stream_switch_slave_south_0_slot0},
      {"cm_stream_switch_slave_south_0_slot1", aie2::cm_stream_switch_slave_south_0_slot1},
      {"cm_stream_switch_slave_south_0_slot2", aie2::cm_stream_switch_slave_south_0_slot2},
      {"cm_stream_switch_slave_south_0_slot3", aie2::cm_stream_switch_slave_south_0_slot3},
      {"cm_stream_switch_slave_south_1_slot0", aie2::cm_stream_switch_slave_south_1_slot0},
      {"cm_stream_switch_slave_south_1_slot1", aie2::cm_stream_switch_slave_south_1_slot1},
      {"cm_stream_switch_slave_south_1_slot2", aie2::cm_stream_switch_slave_south_1_slot2},
      {"cm_stream_switch_slave_south_1_slot3", aie2::cm_stream_switch_slave_south_1_slot3},
      {"cm_stream_switch_slave_south_2_slot0", aie2::cm_stream_switch_slave_south_2_slot0},
      {"cm_stream_switch_slave_south_2_slot1", aie2::cm_stream_switch_slave_south_2_slot1},
      {"cm_stream_switch_slave_south_2_slot2", aie2::cm_stream_switch_slave_south_2_slot2},
      {"cm_stream_switch_slave_south_2_slot3", aie2::cm_stream_switch_slave_south_2_slot3},
      {"cm_stream_switch_slave_south_3_slot0", aie2::cm_stream_switch_slave_south_3_slot0},
      {"cm_stream_switch_slave_south_3_slot1", aie2::cm_stream_switch_slave_south_3_slot1},
      {"cm_stream_switch_slave_south_3_slot2", aie2::cm_stream_switch_slave_south_3_slot2},
      {"cm_stream_switch_slave_south_3_slot3", aie2::cm_stream_switch_slave_south_3_slot3},
      {"cm_stream_switch_slave_south_4_slot0", aie2::cm_stream_switch_slave_south_4_slot0},
      {"cm_stream_switch_slave_south_4_slot1", aie2::cm_stream_switch_slave_south_4_slot1},
      {"cm_stream_switch_slave_south_4_slot2", aie2::cm_stream_switch_slave_south_4_slot2},
      {"cm_stream_switch_slave_south_4_slot3", aie2::cm_stream_switch_slave_south_4_slot3},
      {"cm_stream_switch_slave_south_5_slot0", aie2::cm_stream_switch_slave_south_5_slot0},
      {"cm_stream_switch_slave_south_5_slot1", aie2::cm_stream_switch_slave_south_5_slot1},
      {"cm_stream_switch_slave_south_5_slot2", aie2::cm_stream_switch_slave_south_5_slot2},
      {"cm_stream_switch_slave_south_5_slot3", aie2::cm_stream_switch_slave_south_5_slot3},
      {"cm_stream_switch_slave_west_0_slot0", aie2::cm_stream_switch_slave_west_0_slot0},
      {"cm_stream_switch_slave_west_0_slot1", aie2::cm_stream_switch_slave_west_0_slot1},
      {"cm_stream_switch_slave_west_0_slot2", aie2::cm_stream_switch_slave_west_0_slot2},
      {"cm_stream_switch_slave_west_0_slot3", aie2::cm_stream_switch_slave_west_0_slot3},
      {"cm_stream_switch_slave_west_1_slot0", aie2::cm_stream_switch_slave_west_1_slot0},
      {"cm_stream_switch_slave_west_1_slot1", aie2::cm_stream_switch_slave_west_1_slot1},
      {"cm_stream_switch_slave_west_1_slot2", aie2::cm_stream_switch_slave_west_1_slot2},
      {"cm_stream_switch_slave_west_1_slot3", aie2::cm_stream_switch_slave_west_1_slot3},
      {"cm_stream_switch_slave_west_2_slot0", aie2::cm_stream_switch_slave_west_2_slot0},
      {"cm_stream_switch_slave_west_2_slot1", aie2::cm_stream_switch_slave_west_2_slot1},
      {"cm_stream_switch_slave_west_2_slot2", aie2::cm_stream_switch_slave_west_2_slot2},
      {"cm_stream_switch_slave_west_2_slot3", aie2::cm_stream_switch_slave_west_2_slot3},
      {"cm_stream_switch_slave_west_3_slot0", aie2::cm_stream_switch_slave_west_3_slot0},
      {"cm_stream_switch_slave_west_3_slot1", aie2::cm_stream_switch_slave_west_3_slot1},
      {"cm_stream_switch_slave_west_3_slot2", aie2::cm_stream_switch_slave_west_3_slot2},
      {"cm_stream_switch_slave_west_3_slot3", aie2::cm_stream_switch_slave_west_3_slot3},
      {"cm_stream_switch_slave_north_0_slot0", aie2::cm_stream_switch_slave_north_0_slot0},
      {"cm_stream_switch_slave_north_0_slot1", aie2::cm_stream_switch_slave_north_0_slot1},
      {"cm_stream_switch_slave_north_0_slot2", aie2::cm_stream_switch_slave_north_0_slot2},
      {"cm_stream_switch_slave_north_0_slot3", aie2::cm_stream_switch_slave_north_0_slot3},
      {"cm_stream_switch_slave_north_1_slot0", aie2::cm_stream_switch_slave_north_1_slot0},
      {"cm_stream_switch_slave_north_1_slot1", aie2::cm_stream_switch_slave_north_1_slot1},
      {"cm_stream_switch_slave_north_1_slot2", aie2::cm_stream_switch_slave_north_1_slot2},
      {"cm_stream_switch_slave_north_1_slot3", aie2::cm_stream_switch_slave_north_1_slot3},
      {"cm_stream_switch_slave_north_2_slot0", aie2::cm_stream_switch_slave_north_2_slot0},
      {"cm_stream_switch_slave_north_2_slot1", aie2::cm_stream_switch_slave_north_2_slot1},
      {"cm_stream_switch_slave_north_2_slot2", aie2::cm_stream_switch_slave_north_2_slot2},
      {"cm_stream_switch_slave_north_2_slot3", aie2::cm_stream_switch_slave_north_2_slot3},
      {"cm_stream_switch_slave_north_3_slot0", aie2::cm_stream_switch_slave_north_3_slot0},
      {"cm_stream_switch_slave_north_3_slot1", aie2::cm_stream_switch_slave_north_3_slot1},
      {"cm_stream_switch_slave_north_3_slot2", aie2::cm_stream_switch_slave_north_3_slot2},
      {"cm_stream_switch_slave_north_3_slot3", aie2::cm_stream_switch_slave_north_3_slot3},
      {"cm_stream_switch_slave_east_0_slot0", aie2::cm_stream_switch_slave_east_0_slot0},
      {"cm_stream_switch_slave_east_0_slot1", aie2::cm_stream_switch_slave_east_0_slot1},
      {"cm_stream_switch_slave_east_0_slot2", aie2::cm_stream_switch_slave_east_0_slot2},
      {"cm_stream_switch_slave_east_0_slot3", aie2::cm_stream_switch_slave_east_0_slot3},
      {"cm_stream_switch_slave_east_1_slot0", aie2::cm_stream_switch_slave_east_1_slot0},
      {"cm_stream_switch_slave_east_1_slot1", aie2::cm_stream_switch_slave_east_1_slot1},
      {"cm_stream_switch_slave_east_1_slot2", aie2::cm_stream_switch_slave_east_1_slot2},
      {"cm_stream_switch_slave_east_1_slot3", aie2::cm_stream_switch_slave_east_1_slot3},
      {"cm_stream_switch_slave_east_2_slot0", aie2::cm_stream_switch_slave_east_2_slot0},
      {"cm_stream_switch_slave_east_2_slot1", aie2::cm_stream_switch_slave_east_2_slot1},
      {"cm_stream_switch_slave_east_2_slot2", aie2::cm_stream_switch_slave_east_2_slot2},
      {"cm_stream_switch_slave_east_2_slot3", aie2::cm_stream_switch_slave_east_2_slot3},
      {"cm_stream_switch_slave_east_3_slot0", aie2::cm_stream_switch_slave_east_3_slot0},
      {"cm_stream_switch_slave_east_3_slot1", aie2::cm_stream_switch_slave_east_3_slot1},
      {"cm_stream_switch_slave_east_3_slot2", aie2::cm_stream_switch_slave_east_3_slot2},
      {"cm_stream_switch_slave_east_3_slot3", aie2::cm_stream_switch_slave_east_3_slot3},
      {"cm_stream_switch_slave_aie_trace_slot0", aie2::cm_stream_switch_slave_aie_trace_slot0},
      {"cm_stream_switch_slave_aie_trace_slot1", aie2::cm_stream_switch_slave_aie_trace_slot1},
      {"cm_stream_switch_slave_aie_trace_slot2", aie2::cm_stream_switch_slave_aie_trace_slot2},
      {"cm_stream_switch_slave_aie_trace_slot3", aie2::cm_stream_switch_slave_aie_trace_slot3},
      {"cm_stream_switch_slave_mem_trace_slot0", aie2::cm_stream_switch_slave_mem_trace_slot0},
      {"cm_stream_switch_slave_mem_trace_slot1", aie2::cm_stream_switch_slave_mem_trace_slot1},
      {"cm_stream_switch_slave_mem_trace_slot2", aie2::cm_stream_switch_slave_mem_trace_slot2},
      {"cm_stream_switch_slave_mem_trace_slot3", aie2::cm_stream_switch_slave_mem_trace_slot3},
      {"cm_stream_switch_deterministic_merge_arb0_slave0_1", aie2::cm_stream_switch_deterministic_merge_arb0_slave0_1},
      {"cm_stream_switch_deterministic_merge_arb0_slave2_3", aie2::cm_stream_switch_deterministic_merge_arb0_slave2_3},
      {"cm_stream_switch_deterministic_merge_arb0_ctrl", aie2::cm_stream_switch_deterministic_merge_arb0_ctrl},
      {"cm_stream_switch_deterministic_merge_arb1_slave0_1", aie2::cm_stream_switch_deterministic_merge_arb1_slave0_1},
      {"cm_stream_switch_deterministic_merge_arb1_slave2_3", aie2::cm_stream_switch_deterministic_merge_arb1_slave2_3},
      {"cm_stream_switch_deterministic_merge_arb1_ctrl", aie2::cm_stream_switch_deterministic_merge_arb1_ctrl},
      {"cm_stream_switch_event_port_selection_0", aie2::cm_stream_switch_event_port_selection_0},
      {"cm_stream_switch_event_port_selection_1", aie2::cm_stream_switch_event_port_selection_1},
      {"cm_stream_switch_parity_status", aie2::cm_stream_switch_parity_status},
      {"cm_stream_switch_parity_injection", aie2::cm_stream_switch_parity_injection},
      {"cm_tile_control_packet_handler_status", aie2::cm_tile_control_packet_handler_status},
      {"cm_stream_switch_adaptive_clock_gate_status", aie2::cm_stream_switch_adaptive_clock_gate_status},
      {"cm_stream_switch_adaptive_clock_gate_abort_period", aie2::cm_stream_switch_adaptive_clock_gate_abort_period},
      {"cm_module_clock_control", aie2::cm_module_clock_control},
      {"cm_module_reset_control", aie2::cm_module_reset_control},
      {"mm_performance_control0", aie2::mm_performance_control0},
      {"mm_performance_control1", aie2::mm_performance_control1},
      {"mm_performance_counter0", aie2::mm_performance_counter0},
      {"mm_performance_counter1", aie2::mm_performance_counter1},
      {"mm_performance_counter0_event_value", aie2::mm_performance_counter0_event_value},
      {"mm_performance_counter1_event_value", aie2::mm_performance_counter1_event_value},
      {"mm_checkbit_error_generation", aie2::mm_checkbit_error_generation},
      {"mm_ecc_scrubbing_event", aie2::mm_ecc_scrubbing_event},
      {"mm_ecc_failing_address", aie2::mm_ecc_failing_address},
      {"mm_parity_failing_address", aie2::mm_parity_failing_address},
      {"mm_timer_control", aie2::mm_timer_control},
      {"mm_event_generate", aie2::mm_event_generate},
      {"mm_event_broadcast0", aie2::mm_event_broadcast0},
      {"mm_event_broadcast1", aie2::mm_event_broadcast1},
      {"mm_event_broadcast2", aie2::mm_event_broadcast2},
      {"mm_event_broadcast3", aie2::mm_event_broadcast3},
      {"mm_event_broadcast4", aie2::mm_event_broadcast4},
      {"mm_event_broadcast5", aie2::mm_event_broadcast5},
      {"mm_event_broadcast6", aie2::mm_event_broadcast6},
      {"mm_event_broadcast7", aie2::mm_event_broadcast7},
      {"mm_event_broadcast8", aie2::mm_event_broadcast8},
      {"mm_event_broadcast9", aie2::mm_event_broadcast9},
      {"mm_event_broadcast10", aie2::mm_event_broadcast10},
      {"mm_event_broadcast11", aie2::mm_event_broadcast11},
      {"mm_event_broadcast12", aie2::mm_event_broadcast12},
      {"mm_event_broadcast13", aie2::mm_event_broadcast13},
      {"mm_event_broadcast14", aie2::mm_event_broadcast14},
      {"mm_event_broadcast15", aie2::mm_event_broadcast15},
      {"mm_event_broadcast_block_south_set", aie2::mm_event_broadcast_block_south_set},
      {"mm_event_broadcast_block_south_clr", aie2::mm_event_broadcast_block_south_clr},
      {"mm_event_broadcast_block_south_value", aie2::mm_event_broadcast_block_south_value},
      {"mm_event_broadcast_block_west_set", aie2::mm_event_broadcast_block_west_set},
      {"mm_event_broadcast_block_west_clr", aie2::mm_event_broadcast_block_west_clr},
      {"mm_event_broadcast_block_west_value", aie2::mm_event_broadcast_block_west_value},
      {"mm_event_broadcast_block_north_set", aie2::mm_event_broadcast_block_north_set},
      {"mm_event_broadcast_block_north_clr", aie2::mm_event_broadcast_block_north_clr},
      {"mm_event_broadcast_block_north_value", aie2::mm_event_broadcast_block_north_value},
      {"mm_event_broadcast_block_east_set", aie2::mm_event_broadcast_block_east_set},
      {"mm_event_broadcast_block_east_clr", aie2::mm_event_broadcast_block_east_clr},
      {"mm_event_broadcast_block_east_value", aie2::mm_event_broadcast_block_east_value},
      {"mm_trace_control0", aie2::mm_trace_control0},
      {"mm_trace_control1", aie2::mm_trace_control1},
      {"mm_trace_status", aie2::mm_trace_status},
      {"mm_trace_event0", aie2::mm_trace_event0},
      {"mm_trace_event1", aie2::mm_trace_event1},
      {"mm_timer_trig_event_low_value", aie2::mm_timer_trig_event_low_value},
      {"mm_timer_trig_event_high_value", aie2::mm_timer_trig_event_high_value},
      {"mm_timer_low", aie2::mm_timer_low},
      {"mm_timer_high", aie2::mm_timer_high},
      {"mm_watchpoint0", aie2::mm_watchpoint0},
      {"mm_watchpoint1", aie2::mm_watchpoint1},
      {"mm_event_status0", aie2::mm_event_status0},
      {"mm_event_status1", aie2::mm_event_status1},
      {"mm_event_status2", aie2::mm_event_status2},
      {"mm_event_status3", aie2::mm_event_status3},
      {"mm_reserved0", aie2::mm_reserved0},
      {"mm_reserved1", aie2::mm_reserved1},
      {"mm_reserved2", aie2::mm_reserved2},
      {"mm_reserved3", aie2::mm_reserved3},
      {"mm_combo_event_inputs", aie2::mm_combo_event_inputs},
      {"mm_combo_event_control", aie2::mm_combo_event_control},
      {"mm_edge_detection_event_control", aie2::mm_edge_detection_event_control},
      {"mm_event_group_0_enable", aie2::mm_event_group_0_enable},
      {"mm_event_group_watchpoint_enable", aie2::mm_event_group_watchpoint_enable},
      {"mm_event_group_dma_enable", aie2::mm_event_group_dma_enable},
      {"mm_event_group_lock_enable", aie2::mm_event_group_lock_enable},
      {"mm_event_group_memory_conflict_enable", aie2::mm_event_group_memory_conflict_enable},
      {"mm_event_group_error_enable", aie2::mm_event_group_error_enable},
      {"mm_event_group_broadcast_enable", aie2::mm_event_group_broadcast_enable},
      {"mm_event_group_user_event_enable", aie2::mm_event_group_user_event_enable},
      {"mm_spare_reg", aie2::mm_spare_reg},
      {"mm_memory_control", aie2::mm_memory_control},
      {"mm_dma_bd0_0", aie2::mm_dma_bd0_0},
      {"mm_dma_bd0_1", aie2::mm_dma_bd0_1},
      {"mm_dma_bd0_2", aie2::mm_dma_bd0_2},
      {"mm_dma_bd0_3", aie2::mm_dma_bd0_3},
      {"mm_dma_bd0_4", aie2::mm_dma_bd0_4},
      {"mm_dma_bd0_5", aie2::mm_dma_bd0_5},
      {"mm_dma_bd1_0", aie2::mm_dma_bd1_0},
      {"mm_dma_bd1_1", aie2::mm_dma_bd1_1},
      {"mm_dma_bd1_2", aie2::mm_dma_bd1_2},
      {"mm_dma_bd1_3", aie2::mm_dma_bd1_3},
      {"mm_dma_bd1_4", aie2::mm_dma_bd1_4},
      {"mm_dma_bd1_5", aie2::mm_dma_bd1_5},
      {"mm_dma_bd2_0", aie2::mm_dma_bd2_0},
      {"mm_dma_bd2_1", aie2::mm_dma_bd2_1},
      {"mm_dma_bd2_2", aie2::mm_dma_bd2_2},
      {"mm_dma_bd2_3", aie2::mm_dma_bd2_3},
      {"mm_dma_bd2_4", aie2::mm_dma_bd2_4},
      {"mm_dma_bd2_5", aie2::mm_dma_bd2_5},
      {"mm_dma_bd3_0", aie2::mm_dma_bd3_0},
      {"mm_dma_bd3_1", aie2::mm_dma_bd3_1},
      {"mm_dma_bd3_2", aie2::mm_dma_bd3_2},
      {"mm_dma_bd3_3", aie2::mm_dma_bd3_3},
      {"mm_dma_bd3_4", aie2::mm_dma_bd3_4},
      {"mm_dma_bd3_5", aie2::mm_dma_bd3_5},
      {"mm_dma_bd4_0", aie2::mm_dma_bd4_0},
      {"mm_dma_bd4_1", aie2::mm_dma_bd4_1},
      {"mm_dma_bd4_2", aie2::mm_dma_bd4_2},
      {"mm_dma_bd4_3", aie2::mm_dma_bd4_3},
      {"mm_dma_bd4_4", aie2::mm_dma_bd4_4},
      {"mm_dma_bd4_5", aie2::mm_dma_bd4_5},
      {"mm_dma_bd5_0", aie2::mm_dma_bd5_0},
      {"mm_dma_bd5_1", aie2::mm_dma_bd5_1},
      {"mm_dma_bd5_2", aie2::mm_dma_bd5_2},
      {"mm_dma_bd5_3", aie2::mm_dma_bd5_3},
      {"mm_dma_bd5_4", aie2::mm_dma_bd5_4},
      {"mm_dma_bd5_5", aie2::mm_dma_bd5_5},
      {"mm_dma_bd6_0", aie2::mm_dma_bd6_0},
      {"mm_dma_bd6_1", aie2::mm_dma_bd6_1},
      {"mm_dma_bd6_2", aie2::mm_dma_bd6_2},
      {"mm_dma_bd6_3", aie2::mm_dma_bd6_3},
      {"mm_dma_bd6_4", aie2::mm_dma_bd6_4},
      {"mm_dma_bd6_5", aie2::mm_dma_bd6_5},
      {"mm_dma_bd7_0", aie2::mm_dma_bd7_0},
      {"mm_dma_bd7_1", aie2::mm_dma_bd7_1},
      {"mm_dma_bd7_2", aie2::mm_dma_bd7_2},
      {"mm_dma_bd7_3", aie2::mm_dma_bd7_3},
      {"mm_dma_bd7_4", aie2::mm_dma_bd7_4},
      {"mm_dma_bd7_5", aie2::mm_dma_bd7_5},
      {"mm_dma_bd8_0", aie2::mm_dma_bd8_0},
      {"mm_dma_bd8_1", aie2::mm_dma_bd8_1},
      {"mm_dma_bd8_2", aie2::mm_dma_bd8_2},
      {"mm_dma_bd8_3", aie2::mm_dma_bd8_3},
      {"mm_dma_bd8_4", aie2::mm_dma_bd8_4},
      {"mm_dma_bd8_5", aie2::mm_dma_bd8_5},
      {"mm_dma_bd9_0", aie2::mm_dma_bd9_0},
      {"mm_dma_bd9_1", aie2::mm_dma_bd9_1},
      {"mm_dma_bd9_2", aie2::mm_dma_bd9_2},
      {"mm_dma_bd9_3", aie2::mm_dma_bd9_3},
      {"mm_dma_bd9_4", aie2::mm_dma_bd9_4},
      {"mm_dma_bd9_5", aie2::mm_dma_bd9_5},
      {"mm_dma_bd10_0", aie2::mm_dma_bd10_0},
      {"mm_dma_bd10_1", aie2::mm_dma_bd10_1},
      {"mm_dma_bd10_2", aie2::mm_dma_bd10_2},
      {"mm_dma_bd10_3", aie2::mm_dma_bd10_3},
      {"mm_dma_bd10_4", aie2::mm_dma_bd10_4},
      {"mm_dma_bd10_5", aie2::mm_dma_bd10_5},
      {"mm_dma_bd11_0", aie2::mm_dma_bd11_0},
      {"mm_dma_bd11_1", aie2::mm_dma_bd11_1},
      {"mm_dma_bd11_2", aie2::mm_dma_bd11_2},
      {"mm_dma_bd11_3", aie2::mm_dma_bd11_3},
      {"mm_dma_bd11_4", aie2::mm_dma_bd11_4},
      {"mm_dma_bd11_5", aie2::mm_dma_bd11_5},
      {"mm_dma_bd12_0", aie2::mm_dma_bd12_0},
      {"mm_dma_bd12_1", aie2::mm_dma_bd12_1},
      {"mm_dma_bd12_2", aie2::mm_dma_bd12_2},
      {"mm_dma_bd12_3", aie2::mm_dma_bd12_3},
      {"mm_dma_bd12_4", aie2::mm_dma_bd12_4},
      {"mm_dma_bd12_5", aie2::mm_dma_bd12_5},
      {"mm_dma_bd13_0", aie2::mm_dma_bd13_0},
      {"mm_dma_bd13_1", aie2::mm_dma_bd13_1},
      {"mm_dma_bd13_2", aie2::mm_dma_bd13_2},
      {"mm_dma_bd13_3", aie2::mm_dma_bd13_3},
      {"mm_dma_bd13_4", aie2::mm_dma_bd13_4},
      {"mm_dma_bd13_5", aie2::mm_dma_bd13_5},
      {"mm_dma_bd14_0", aie2::mm_dma_bd14_0},
      {"mm_dma_bd14_1", aie2::mm_dma_bd14_1},
      {"mm_dma_bd14_2", aie2::mm_dma_bd14_2},
      {"mm_dma_bd14_3", aie2::mm_dma_bd14_3},
      {"mm_dma_bd14_4", aie2::mm_dma_bd14_4},
      {"mm_dma_bd14_5", aie2::mm_dma_bd14_5},
      {"mm_dma_bd15_0", aie2::mm_dma_bd15_0},
      {"mm_dma_bd15_1", aie2::mm_dma_bd15_1},
      {"mm_dma_bd15_2", aie2::mm_dma_bd15_2},
      {"mm_dma_bd15_3", aie2::mm_dma_bd15_3},
      {"mm_dma_bd15_4", aie2::mm_dma_bd15_4},
      {"mm_dma_bd15_5", aie2::mm_dma_bd15_5},
      {"mm_dma_s2mm_0_ctrl", aie2::mm_dma_s2mm_0_ctrl},
      {"mm_dma_s2mm_0_start_queue", aie2::mm_dma_s2mm_0_start_queue},
      {"mm_dma_s2mm_1_ctrl", aie2::mm_dma_s2mm_1_ctrl},
      {"mm_dma_s2mm_1_start_queue", aie2::mm_dma_s2mm_1_start_queue},
      {"mm_dma_mm2s_0_ctrl", aie2::mm_dma_mm2s_0_ctrl},
      {"mm_dma_mm2s_0_start_queue", aie2::mm_dma_mm2s_0_start_queue},
      {"mm_dma_mm2s_1_ctrl", aie2::mm_dma_mm2s_1_ctrl},
      {"mm_dma_mm2s_1_start_queue", aie2::mm_dma_mm2s_1_start_queue},
      {"mm_dma_s2mm_status_0", aie2::mm_dma_s2mm_status_0},
      {"mm_dma_s2mm_status_1", aie2::mm_dma_s2mm_status_1},
      {"mm_dma_mm2s_status_0", aie2::mm_dma_mm2s_status_0},
      {"mm_dma_mm2s_status_1", aie2::mm_dma_mm2s_status_1},
      {"mm_dma_s2mm_current_write_count_0", aie2::mm_dma_s2mm_current_write_count_0},
      {"mm_dma_s2mm_current_write_count_1", aie2::mm_dma_s2mm_current_write_count_1},
      {"mm_dma_s2mm_fot_count_fifo_pop_0", aie2::mm_dma_s2mm_fot_count_fifo_pop_0},
      {"mm_dma_s2mm_fot_count_fifo_pop_1", aie2::mm_dma_s2mm_fot_count_fifo_pop_1},
      {"mm_lock0_value", aie2::mm_lock0_value},
      {"mm_lock1_value", aie2::mm_lock1_value},
      {"mm_lock2_value", aie2::mm_lock2_value},
      {"mm_lock3_value", aie2::mm_lock3_value},
      {"mm_lock4_value", aie2::mm_lock4_value},
      {"mm_lock5_value", aie2::mm_lock5_value},
      {"mm_lock6_value", aie2::mm_lock6_value},
      {"mm_lock7_value", aie2::mm_lock7_value},
      {"mm_lock8_value", aie2::mm_lock8_value},
      {"mm_lock9_value", aie2::mm_lock9_value},
      {"mm_lock10_value", aie2::mm_lock10_value},
      {"mm_lock11_value", aie2::mm_lock11_value},
      {"mm_lock12_value", aie2::mm_lock12_value},
      {"mm_lock13_value", aie2::mm_lock13_value},
      {"mm_lock14_value", aie2::mm_lock14_value},
      {"mm_lock15_value", aie2::mm_lock15_value},
      {"mm_locks_event_selection_0", aie2::mm_locks_event_selection_0},
      {"mm_locks_event_selection_1", aie2::mm_locks_event_selection_1},
      {"mm_locks_event_selection_2", aie2::mm_locks_event_selection_2},
      {"mm_locks_event_selection_3", aie2::mm_locks_event_selection_3},
      {"mm_locks_event_selection_4", aie2::mm_locks_event_selection_4},
      {"mm_locks_event_selection_5", aie2::mm_locks_event_selection_5},
      {"mm_locks_event_selection_6", aie2::mm_locks_event_selection_6},
      {"mm_locks_event_selection_7", aie2::mm_locks_event_selection_7},
      {"mm_locks_overflow", aie2::mm_locks_overflow},
      {"mm_locks_underflow", aie2::mm_locks_underflow},
      {"mm_lock_request", aie2::mm_lock_request},
      {"mem_performance_control0", aie2::mem_performance_control0},
      {"mem_performance_control1", aie2::mem_performance_control1},
      {"mem_performance_control2", aie2::mem_performance_control2},
      {"mem_performance_counter0", aie2::mem_performance_counter0},
      {"mem_performance_counter1", aie2::mem_performance_counter1},
      {"mem_performance_counter2", aie2::mem_performance_counter2},
      {"mem_performance_counter3", aie2::mem_performance_counter3},
      {"mem_performance_counter0_event_value", aie2::mem_performance_counter0_event_value},
      {"mem_performance_counter1_event_value", aie2::mem_performance_counter1_event_value},
      {"mem_performance_counter2_event_value", aie2::mem_performance_counter2_event_value},
      {"mem_performance_counter3_event_value", aie2::mem_performance_counter3_event_value},
      {"mem_checkbit_error_generation", aie2::mem_checkbit_error_generation},
      {"mem_ecc_scrubbing_event", aie2::mem_ecc_scrubbing_event},
      {"mem_ecc_failing_address", aie2::mem_ecc_failing_address},
      {"mem_timer_control", aie2::mem_timer_control},
      {"mem_event_generate", aie2::mem_event_generate},
      {"mem_event_broadcast0", aie2::mem_event_broadcast0},
      {"mem_event_broadcast1", aie2::mem_event_broadcast1},
      {"mem_event_broadcast2", aie2::mem_event_broadcast2},
      {"mem_event_broadcast3", aie2::mem_event_broadcast3},
      {"mem_event_broadcast4", aie2::mem_event_broadcast4},
      {"mem_event_broadcast5", aie2::mem_event_broadcast5},
      {"mem_event_broadcast6", aie2::mem_event_broadcast6},
      {"mem_event_broadcast7", aie2::mem_event_broadcast7},
      {"mem_event_broadcast8", aie2::mem_event_broadcast8},
      {"mem_event_broadcast9", aie2::mem_event_broadcast9},
      {"mem_event_broadcast10", aie2::mem_event_broadcast10},
      {"mem_event_broadcast11", aie2::mem_event_broadcast11},
      {"mem_event_broadcast12", aie2::mem_event_broadcast12},
      {"mem_event_broadcast13", aie2::mem_event_broadcast13},
      {"mem_event_broadcast14", aie2::mem_event_broadcast14},
      {"mem_event_broadcast15", aie2::mem_event_broadcast15},
      {"mem_event_broadcast_a_block_south_set", aie2::mem_event_broadcast_a_block_south_set},
      {"mem_event_broadcast_a_block_south_clr", aie2::mem_event_broadcast_a_block_south_clr},
      {"mem_event_broadcast_a_block_south_value", aie2::mem_event_broadcast_a_block_south_value},
      {"mem_event_broadcast_a_block_west_set", aie2::mem_event_broadcast_a_block_west_set},
      {"mem_event_broadcast_a_block_west_clr", aie2::mem_event_broadcast_a_block_west_clr},
      {"mem_event_broadcast_a_block_west_value", aie2::mem_event_broadcast_a_block_west_value},
      {"mem_event_broadcast_a_block_north_set", aie2::mem_event_broadcast_a_block_north_set},
      {"mem_event_broadcast_a_block_north_clr", aie2::mem_event_broadcast_a_block_north_clr},
      {"mem_event_broadcast_a_block_north_value", aie2::mem_event_broadcast_a_block_north_value},
      {"mem_event_broadcast_a_block_east_set", aie2::mem_event_broadcast_a_block_east_set},
      {"mem_event_broadcast_a_block_east_clr", aie2::mem_event_broadcast_a_block_east_clr},
      {"mem_event_broadcast_a_block_east_value", aie2::mem_event_broadcast_a_block_east_value},
      {"mem_event_broadcast_b_block_south_set", aie2::mem_event_broadcast_b_block_south_set},
      {"mem_event_broadcast_b_block_south_clr", aie2::mem_event_broadcast_b_block_south_clr},
      {"mem_event_broadcast_b_block_south_value", aie2::mem_event_broadcast_b_block_south_value},
      {"mem_event_broadcast_b_block_west_set", aie2::mem_event_broadcast_b_block_west_set},
      {"mem_event_broadcast_b_block_west_clr", aie2::mem_event_broadcast_b_block_west_clr},
      {"mem_event_broadcast_b_block_west_value", aie2::mem_event_broadcast_b_block_west_value},
      {"mem_event_broadcast_b_block_north_set", aie2::mem_event_broadcast_b_block_north_set},
      {"mem_event_broadcast_b_block_north_clr", aie2::mem_event_broadcast_b_block_north_clr},
      {"mem_event_broadcast_b_block_north_value", aie2::mem_event_broadcast_b_block_north_value},
      {"mem_event_broadcast_b_block_east_set", aie2::mem_event_broadcast_b_block_east_set},
      {"mem_event_broadcast_b_block_east_clr", aie2::mem_event_broadcast_b_block_east_clr},
      {"mem_event_broadcast_b_block_east_value", aie2::mem_event_broadcast_b_block_east_value},
      {"mem_trace_control0", aie2::mem_trace_control0},
      {"mem_trace_control1", aie2::mem_trace_control1},
      {"mem_trace_status", aie2::mem_trace_status},
      {"mem_trace_event0", aie2::mem_trace_event0},
      {"mem_trace_event1", aie2::mem_trace_event1},
      {"mem_timer_trig_event_low_value", aie2::mem_timer_trig_event_low_value},
      {"mem_timer_trig_event_high_value", aie2::mem_timer_trig_event_high_value},
      {"mem_timer_low", aie2::mem_timer_low},
      {"mem_timer_high", aie2::mem_timer_high},
      {"mem_watchpoint0", aie2::mem_watchpoint0},
      {"mem_watchpoint1", aie2::mem_watchpoint1},
      {"mem_watchpoint2", aie2::mem_watchpoint2},
      {"mem_watchpoint3", aie2::mem_watchpoint3},
      {"mem_event_status0", aie2::mem_event_status0},
      {"mem_event_status1", aie2::mem_event_status1},
      {"mem_event_status2", aie2::mem_event_status2},
      {"mem_event_status3", aie2::mem_event_status3},
      {"mem_event_status4", aie2::mem_event_status4},
      {"mem_event_status5", aie2::mem_event_status5},
      {"mem_reserved0", aie2::mem_reserved0},
      {"mem_reserved1", aie2::mem_reserved1},
      {"mem_reserved2", aie2::mem_reserved2},
      {"mem_reserved3", aie2::mem_reserved3},
      {"mem_combo_event_inputs", aie2::mem_combo_event_inputs},
      {"mem_combo_event_control", aie2::mem_combo_event_control},
      {"mem_edge_detection_event_control", aie2::mem_edge_detection_event_control},
      {"mem_event_group_0_enable", aie2::mem_event_group_0_enable},
      {"mem_event_group_watchpoint_enable", aie2::mem_event_group_watchpoint_enable},
      {"mem_event_group_dma_enable", aie2::mem_event_group_dma_enable},
      {"mem_event_group_lock_enable", aie2::mem_event_group_lock_enable},
      {"mem_event_group_stream_switch_enable", aie2::mem_event_group_stream_switch_enable},
      {"mem_event_group_memory_conflict_enable", aie2::mem_event_group_memory_conflict_enable},
      {"mem_event_group_error_enable", aie2::mem_event_group_error_enable},
      {"mem_event_group_broadcast_enable", aie2::mem_event_group_broadcast_enable},
      {"mem_event_group_user_event_enable", aie2::mem_event_group_user_event_enable},
      {"mem_spare_reg", aie2::mem_spare_reg},
      {"mem_tile_control", aie2::mem_tile_control},
      {"mem_cssd_trigger", aie2::mem_cssd_trigger},
      {"mem_memory_control", aie2::mem_memory_control},
      {"mem_dma_bd0_0", aie2::mem_dma_bd0_0},
      {"mem_dma_bd0_1", aie2::mem_dma_bd0_1},
      {"mem_dma_bd0_2", aie2::mem_dma_bd0_2},
      {"mem_dma_bd0_3", aie2::mem_dma_bd0_3},
      {"mem_dma_bd0_4", aie2::mem_dma_bd0_4},
      {"mem_dma_bd0_5", aie2::mem_dma_bd0_5},
      {"mem_dma_bd0_6", aie2::mem_dma_bd0_6},
      {"mem_dma_bd0_7", aie2::mem_dma_bd0_7},
      {"mem_dma_bd1_0", aie2::mem_dma_bd1_0},
      {"mem_dma_bd1_1", aie2::mem_dma_bd1_1},
      {"mem_dma_bd1_2", aie2::mem_dma_bd1_2},
      {"mem_dma_bd1_3", aie2::mem_dma_bd1_3},
      {"mem_dma_bd1_4", aie2::mem_dma_bd1_4},
      {"mem_dma_bd1_5", aie2::mem_dma_bd1_5},
      {"mem_dma_bd1_6", aie2::mem_dma_bd1_6},
      {"mem_dma_bd1_7", aie2::mem_dma_bd1_7},
      {"mem_dma_bd2_0", aie2::mem_dma_bd2_0},
      {"mem_dma_bd2_1", aie2::mem_dma_bd2_1},
      {"mem_dma_bd2_2", aie2::mem_dma_bd2_2},
      {"mem_dma_bd2_3", aie2::mem_dma_bd2_3},
      {"mem_dma_bd2_4", aie2::mem_dma_bd2_4},
      {"mem_dma_bd2_5", aie2::mem_dma_bd2_5},
      {"mem_dma_bd2_6", aie2::mem_dma_bd2_6},
      {"mem_dma_bd2_7", aie2::mem_dma_bd2_7},
      {"mem_dma_bd3_0", aie2::mem_dma_bd3_0},
      {"mem_dma_bd3_1", aie2::mem_dma_bd3_1},
      {"mem_dma_bd3_2", aie2::mem_dma_bd3_2},
      {"mem_dma_bd3_3", aie2::mem_dma_bd3_3},
      {"mem_dma_bd3_4", aie2::mem_dma_bd3_4},
      {"mem_dma_bd3_5", aie2::mem_dma_bd3_5},
      {"mem_dma_bd3_6", aie2::mem_dma_bd3_6},
      {"mem_dma_bd3_7", aie2::mem_dma_bd3_7},
      {"mem_dma_bd4_0", aie2::mem_dma_bd4_0},
      {"mem_dma_bd4_1", aie2::mem_dma_bd4_1},
      {"mem_dma_bd4_2", aie2::mem_dma_bd4_2},
      {"mem_dma_bd4_3", aie2::mem_dma_bd4_3},
      {"mem_dma_bd4_4", aie2::mem_dma_bd4_4},
      {"mem_dma_bd4_5", aie2::mem_dma_bd4_5},
      {"mem_dma_bd4_6", aie2::mem_dma_bd4_6},
      {"mem_dma_bd4_7", aie2::mem_dma_bd4_7},
      {"mem_dma_bd5_0", aie2::mem_dma_bd5_0},
      {"mem_dma_bd5_1", aie2::mem_dma_bd5_1},
      {"mem_dma_bd5_2", aie2::mem_dma_bd5_2},
      {"mem_dma_bd5_3", aie2::mem_dma_bd5_3},
      {"mem_dma_bd5_4", aie2::mem_dma_bd5_4},
      {"mem_dma_bd5_5", aie2::mem_dma_bd5_5},
      {"mem_dma_bd5_6", aie2::mem_dma_bd5_6},
      {"mem_dma_bd5_7", aie2::mem_dma_bd5_7},
      {"mem_dma_bd6_0", aie2::mem_dma_bd6_0},
      {"mem_dma_bd6_1", aie2::mem_dma_bd6_1},
      {"mem_dma_bd6_2", aie2::mem_dma_bd6_2},
      {"mem_dma_bd6_3", aie2::mem_dma_bd6_3},
      {"mem_dma_bd6_4", aie2::mem_dma_bd6_4},
      {"mem_dma_bd6_5", aie2::mem_dma_bd6_5},
      {"mem_dma_bd6_6", aie2::mem_dma_bd6_6},
      {"mem_dma_bd6_7", aie2::mem_dma_bd6_7},
      {"mem_dma_bd7_0", aie2::mem_dma_bd7_0},
      {"mem_dma_bd7_1", aie2::mem_dma_bd7_1},
      {"mem_dma_bd7_2", aie2::mem_dma_bd7_2},
      {"mem_dma_bd7_3", aie2::mem_dma_bd7_3},
      {"mem_dma_bd7_4", aie2::mem_dma_bd7_4},
      {"mem_dma_bd7_5", aie2::mem_dma_bd7_5},
      {"mem_dma_bd7_6", aie2::mem_dma_bd7_6},
      {"mem_dma_bd7_7", aie2::mem_dma_bd7_7},
      {"mem_dma_bd8_0", aie2::mem_dma_bd8_0},
      {"mem_dma_bd8_1", aie2::mem_dma_bd8_1},
      {"mem_dma_bd8_2", aie2::mem_dma_bd8_2},
      {"mem_dma_bd8_3", aie2::mem_dma_bd8_3},
      {"mem_dma_bd8_4", aie2::mem_dma_bd8_4},
      {"mem_dma_bd8_5", aie2::mem_dma_bd8_5},
      {"mem_dma_bd8_6", aie2::mem_dma_bd8_6},
      {"mem_dma_bd8_7", aie2::mem_dma_bd8_7},
      {"mem_dma_bd9_0", aie2::mem_dma_bd9_0},
      {"mem_dma_bd9_1", aie2::mem_dma_bd9_1},
      {"mem_dma_bd9_2", aie2::mem_dma_bd9_2},
      {"mem_dma_bd9_3", aie2::mem_dma_bd9_3},
      {"mem_dma_bd9_4", aie2::mem_dma_bd9_4},
      {"mem_dma_bd9_5", aie2::mem_dma_bd9_5},
      {"mem_dma_bd9_6", aie2::mem_dma_bd9_6},
      {"mem_dma_bd9_7", aie2::mem_dma_bd9_7},
      {"mem_dma_bd10_0", aie2::mem_dma_bd10_0},
      {"mem_dma_bd10_1", aie2::mem_dma_bd10_1},
      {"mem_dma_bd10_2", aie2::mem_dma_bd10_2},
      {"mem_dma_bd10_3", aie2::mem_dma_bd10_3},
      {"mem_dma_bd10_4", aie2::mem_dma_bd10_4},
      {"mem_dma_bd10_5", aie2::mem_dma_bd10_5},
      {"mem_dma_bd10_6", aie2::mem_dma_bd10_6},
      {"mem_dma_bd10_7", aie2::mem_dma_bd10_7},
      {"mem_dma_bd11_0", aie2::mem_dma_bd11_0},
      {"mem_dma_bd11_1", aie2::mem_dma_bd11_1},
      {"mem_dma_bd11_2", aie2::mem_dma_bd11_2},
      {"mem_dma_bd11_3", aie2::mem_dma_bd11_3},
      {"mem_dma_bd11_4", aie2::mem_dma_bd11_4},
      {"mem_dma_bd11_5", aie2::mem_dma_bd11_5},
      {"mem_dma_bd11_6", aie2::mem_dma_bd11_6},
      {"mem_dma_bd11_7", aie2::mem_dma_bd11_7},
      {"mem_dma_bd12_0", aie2::mem_dma_bd12_0},
      {"mem_dma_bd12_1", aie2::mem_dma_bd12_1},
      {"mem_dma_bd12_2", aie2::mem_dma_bd12_2},
      {"mem_dma_bd12_3", aie2::mem_dma_bd12_3},
      {"mem_dma_bd12_4", aie2::mem_dma_bd12_4},
      {"mem_dma_bd12_5", aie2::mem_dma_bd12_5},
      {"mem_dma_bd12_6", aie2::mem_dma_bd12_6},
      {"mem_dma_bd12_7", aie2::mem_dma_bd12_7},
      {"mem_dma_bd13_0", aie2::mem_dma_bd13_0},
      {"mem_dma_bd13_1", aie2::mem_dma_bd13_1},
      {"mem_dma_bd13_2", aie2::mem_dma_bd13_2},
      {"mem_dma_bd13_3", aie2::mem_dma_bd13_3},
      {"mem_dma_bd13_4", aie2::mem_dma_bd13_4},
      {"mem_dma_bd13_5", aie2::mem_dma_bd13_5},
      {"mem_dma_bd13_6", aie2::mem_dma_bd13_6},
      {"mem_dma_bd13_7", aie2::mem_dma_bd13_7},
      {"mem_dma_bd14_0", aie2::mem_dma_bd14_0},
      {"mem_dma_bd14_1", aie2::mem_dma_bd14_1},
      {"mem_dma_bd14_2", aie2::mem_dma_bd14_2},
      {"mem_dma_bd14_3", aie2::mem_dma_bd14_3},
      {"mem_dma_bd14_4", aie2::mem_dma_bd14_4},
      {"mem_dma_bd14_5", aie2::mem_dma_bd14_5},
      {"mem_dma_bd14_6", aie2::mem_dma_bd14_6},
      {"mem_dma_bd14_7", aie2::mem_dma_bd14_7},
      {"mem_dma_bd15_0", aie2::mem_dma_bd15_0},
      {"mem_dma_bd15_1", aie2::mem_dma_bd15_1},
      {"mem_dma_bd15_2", aie2::mem_dma_bd15_2},
      {"mem_dma_bd15_3", aie2::mem_dma_bd15_3},
      {"mem_dma_bd15_4", aie2::mem_dma_bd15_4},
      {"mem_dma_bd15_5", aie2::mem_dma_bd15_5},
      {"mem_dma_bd15_6", aie2::mem_dma_bd15_6},
      {"mem_dma_bd15_7", aie2::mem_dma_bd15_7},
      {"mem_dma_bd16_0", aie2::mem_dma_bd16_0},
      {"mem_dma_bd16_1", aie2::mem_dma_bd16_1},
      {"mem_dma_bd16_2", aie2::mem_dma_bd16_2},
      {"mem_dma_bd16_3", aie2::mem_dma_bd16_3},
      {"mem_dma_bd16_4", aie2::mem_dma_bd16_4},
      {"mem_dma_bd16_5", aie2::mem_dma_bd16_5},
      {"mem_dma_bd16_6", aie2::mem_dma_bd16_6},
      {"mem_dma_bd16_7", aie2::mem_dma_bd16_7},
      {"mem_dma_bd17_0", aie2::mem_dma_bd17_0},
      {"mem_dma_bd17_1", aie2::mem_dma_bd17_1},
      {"mem_dma_bd17_2", aie2::mem_dma_bd17_2},
      {"mem_dma_bd17_3", aie2::mem_dma_bd17_3},
      {"mem_dma_bd17_4", aie2::mem_dma_bd17_4},
      {"mem_dma_bd17_5", aie2::mem_dma_bd17_5},
      {"mem_dma_bd17_6", aie2::mem_dma_bd17_6},
      {"mem_dma_bd17_7", aie2::mem_dma_bd17_7},
      {"mem_dma_bd18_0", aie2::mem_dma_bd18_0},
      {"mem_dma_bd18_1", aie2::mem_dma_bd18_1},
      {"mem_dma_bd18_2", aie2::mem_dma_bd18_2},
      {"mem_dma_bd18_3", aie2::mem_dma_bd18_3},
      {"mem_dma_bd18_4", aie2::mem_dma_bd18_4},
      {"mem_dma_bd18_5", aie2::mem_dma_bd18_5},
      {"mem_dma_bd18_6", aie2::mem_dma_bd18_6},
      {"mem_dma_bd18_7", aie2::mem_dma_bd18_7},
      {"mem_dma_bd19_0", aie2::mem_dma_bd19_0},
      {"mem_dma_bd19_1", aie2::mem_dma_bd19_1},
      {"mem_dma_bd19_2", aie2::mem_dma_bd19_2},
      {"mem_dma_bd19_3", aie2::mem_dma_bd19_3},
      {"mem_dma_bd19_4", aie2::mem_dma_bd19_4},
      {"mem_dma_bd19_5", aie2::mem_dma_bd19_5},
      {"mem_dma_bd19_6", aie2::mem_dma_bd19_6},
      {"mem_dma_bd19_7", aie2::mem_dma_bd19_7},
      {"mem_dma_bd20_0", aie2::mem_dma_bd20_0},
      {"mem_dma_bd20_1", aie2::mem_dma_bd20_1},
      {"mem_dma_bd20_2", aie2::mem_dma_bd20_2},
      {"mem_dma_bd20_3", aie2::mem_dma_bd20_3},
      {"mem_dma_bd20_4", aie2::mem_dma_bd20_4},
      {"mem_dma_bd20_5", aie2::mem_dma_bd20_5},
      {"mem_dma_bd20_6", aie2::mem_dma_bd20_6},
      {"mem_dma_bd20_7", aie2::mem_dma_bd20_7},
      {"mem_dma_bd21_0", aie2::mem_dma_bd21_0},
      {"mem_dma_bd21_1", aie2::mem_dma_bd21_1},
      {"mem_dma_bd21_2", aie2::mem_dma_bd21_2},
      {"mem_dma_bd21_3", aie2::mem_dma_bd21_3},
      {"mem_dma_bd21_4", aie2::mem_dma_bd21_4},
      {"mem_dma_bd21_5", aie2::mem_dma_bd21_5},
      {"mem_dma_bd21_6", aie2::mem_dma_bd21_6},
      {"mem_dma_bd21_7", aie2::mem_dma_bd21_7},
      {"mem_dma_bd22_0", aie2::mem_dma_bd22_0},
      {"mem_dma_bd22_1", aie2::mem_dma_bd22_1},
      {"mem_dma_bd22_2", aie2::mem_dma_bd22_2},
      {"mem_dma_bd22_3", aie2::mem_dma_bd22_3},
      {"mem_dma_bd22_4", aie2::mem_dma_bd22_4},
      {"mem_dma_bd22_5", aie2::mem_dma_bd22_5},
      {"mem_dma_bd22_6", aie2::mem_dma_bd22_6},
      {"mem_dma_bd22_7", aie2::mem_dma_bd22_7},
      {"mem_dma_bd23_0", aie2::mem_dma_bd23_0},
      {"mem_dma_bd23_1", aie2::mem_dma_bd23_1},
      {"mem_dma_bd23_2", aie2::mem_dma_bd23_2},
      {"mem_dma_bd23_3", aie2::mem_dma_bd23_3},
      {"mem_dma_bd23_4", aie2::mem_dma_bd23_4},
      {"mem_dma_bd23_5", aie2::mem_dma_bd23_5},
      {"mem_dma_bd23_6", aie2::mem_dma_bd23_6},
      {"mem_dma_bd23_7", aie2::mem_dma_bd23_7},
      {"mem_dma_bd24_0", aie2::mem_dma_bd24_0},
      {"mem_dma_bd24_1", aie2::mem_dma_bd24_1},
      {"mem_dma_bd24_2", aie2::mem_dma_bd24_2},
      {"mem_dma_bd24_3", aie2::mem_dma_bd24_3},
      {"mem_dma_bd24_4", aie2::mem_dma_bd24_4},
      {"mem_dma_bd24_5", aie2::mem_dma_bd24_5},
      {"mem_dma_bd24_6", aie2::mem_dma_bd24_6},
      {"mem_dma_bd24_7", aie2::mem_dma_bd24_7},
      {"mem_dma_bd25_0", aie2::mem_dma_bd25_0},
      {"mem_dma_bd25_1", aie2::mem_dma_bd25_1},
      {"mem_dma_bd25_2", aie2::mem_dma_bd25_2},
      {"mem_dma_bd25_3", aie2::mem_dma_bd25_3},
      {"mem_dma_bd25_4", aie2::mem_dma_bd25_4},
      {"mem_dma_bd25_5", aie2::mem_dma_bd25_5},
      {"mem_dma_bd25_6", aie2::mem_dma_bd25_6},
      {"mem_dma_bd25_7", aie2::mem_dma_bd25_7},
      {"mem_dma_bd26_0", aie2::mem_dma_bd26_0},
      {"mem_dma_bd26_1", aie2::mem_dma_bd26_1},
      {"mem_dma_bd26_2", aie2::mem_dma_bd26_2},
      {"mem_dma_bd26_3", aie2::mem_dma_bd26_3},
      {"mem_dma_bd26_4", aie2::mem_dma_bd26_4},
      {"mem_dma_bd26_5", aie2::mem_dma_bd26_5},
      {"mem_dma_bd26_6", aie2::mem_dma_bd26_6},
      {"mem_dma_bd26_7", aie2::mem_dma_bd26_7},
      {"mem_dma_bd27_0", aie2::mem_dma_bd27_0},
      {"mem_dma_bd27_1", aie2::mem_dma_bd27_1},
      {"mem_dma_bd27_2", aie2::mem_dma_bd27_2},
      {"mem_dma_bd27_3", aie2::mem_dma_bd27_3},
      {"mem_dma_bd27_4", aie2::mem_dma_bd27_4},
      {"mem_dma_bd27_5", aie2::mem_dma_bd27_5},
      {"mem_dma_bd27_6", aie2::mem_dma_bd27_6},
      {"mem_dma_bd27_7", aie2::mem_dma_bd27_7},
      {"mem_dma_bd28_0", aie2::mem_dma_bd28_0},
      {"mem_dma_bd28_1", aie2::mem_dma_bd28_1},
      {"mem_dma_bd28_2", aie2::mem_dma_bd28_2},
      {"mem_dma_bd28_3", aie2::mem_dma_bd28_3},
      {"mem_dma_bd28_4", aie2::mem_dma_bd28_4},
      {"mem_dma_bd28_5", aie2::mem_dma_bd28_5},
      {"mem_dma_bd28_6", aie2::mem_dma_bd28_6},
      {"mem_dma_bd28_7", aie2::mem_dma_bd28_7},
      {"mem_dma_bd29_0", aie2::mem_dma_bd29_0},
      {"mem_dma_bd29_1", aie2::mem_dma_bd29_1},
      {"mem_dma_bd29_2", aie2::mem_dma_bd29_2},
      {"mem_dma_bd29_3", aie2::mem_dma_bd29_3},
      {"mem_dma_bd29_4", aie2::mem_dma_bd29_4},
      {"mem_dma_bd29_5", aie2::mem_dma_bd29_5},
      {"mem_dma_bd29_6", aie2::mem_dma_bd29_6},
      {"mem_dma_bd29_7", aie2::mem_dma_bd29_7},
      {"mem_dma_bd30_0", aie2::mem_dma_bd30_0},
      {"mem_dma_bd30_1", aie2::mem_dma_bd30_1},
      {"mem_dma_bd30_2", aie2::mem_dma_bd30_2},
      {"mem_dma_bd30_3", aie2::mem_dma_bd30_3},
      {"mem_dma_bd30_4", aie2::mem_dma_bd30_4},
      {"mem_dma_bd30_5", aie2::mem_dma_bd30_5},
      {"mem_dma_bd30_6", aie2::mem_dma_bd30_6},
      {"mem_dma_bd30_7", aie2::mem_dma_bd30_7},
      {"mem_dma_bd31_0", aie2::mem_dma_bd31_0},
      {"mem_dma_bd31_1", aie2::mem_dma_bd31_1},
      {"mem_dma_bd31_2", aie2::mem_dma_bd31_2},
      {"mem_dma_bd31_3", aie2::mem_dma_bd31_3},
      {"mem_dma_bd31_4", aie2::mem_dma_bd31_4},
      {"mem_dma_bd31_5", aie2::mem_dma_bd31_5},
      {"mem_dma_bd31_6", aie2::mem_dma_bd31_6},
      {"mem_dma_bd31_7", aie2::mem_dma_bd31_7},
      {"mem_dma_bd32_0", aie2::mem_dma_bd32_0},
      {"mem_dma_bd32_1", aie2::mem_dma_bd32_1},
      {"mem_dma_bd32_2", aie2::mem_dma_bd32_2},
      {"mem_dma_bd32_3", aie2::mem_dma_bd32_3},
      {"mem_dma_bd32_4", aie2::mem_dma_bd32_4},
      {"mem_dma_bd32_5", aie2::mem_dma_bd32_5},
      {"mem_dma_bd32_6", aie2::mem_dma_bd32_6},
      {"mem_dma_bd32_7", aie2::mem_dma_bd32_7},
      {"mem_dma_bd33_0", aie2::mem_dma_bd33_0},
      {"mem_dma_bd33_1", aie2::mem_dma_bd33_1},
      {"mem_dma_bd33_2", aie2::mem_dma_bd33_2},
      {"mem_dma_bd33_3", aie2::mem_dma_bd33_3},
      {"mem_dma_bd33_4", aie2::mem_dma_bd33_4},
      {"mem_dma_bd33_5", aie2::mem_dma_bd33_5},
      {"mem_dma_bd33_6", aie2::mem_dma_bd33_6},
      {"mem_dma_bd33_7", aie2::mem_dma_bd33_7},
      {"mem_dma_bd34_0", aie2::mem_dma_bd34_0},
      {"mem_dma_bd34_1", aie2::mem_dma_bd34_1},
      {"mem_dma_bd34_2", aie2::mem_dma_bd34_2},
      {"mem_dma_bd34_3", aie2::mem_dma_bd34_3},
      {"mem_dma_bd34_4", aie2::mem_dma_bd34_4},
      {"mem_dma_bd34_5", aie2::mem_dma_bd34_5},
      {"mem_dma_bd34_6", aie2::mem_dma_bd34_6},
      {"mem_dma_bd34_7", aie2::mem_dma_bd34_7},
      {"mem_dma_bd35_0", aie2::mem_dma_bd35_0},
      {"mem_dma_bd35_1", aie2::mem_dma_bd35_1},
      {"mem_dma_bd35_2", aie2::mem_dma_bd35_2},
      {"mem_dma_bd35_3", aie2::mem_dma_bd35_3},
      {"mem_dma_bd35_4", aie2::mem_dma_bd35_4},
      {"mem_dma_bd35_5", aie2::mem_dma_bd35_5},
      {"mem_dma_bd35_6", aie2::mem_dma_bd35_6},
      {"mem_dma_bd35_7", aie2::mem_dma_bd35_7},
      {"mem_dma_bd36_0", aie2::mem_dma_bd36_0},
      {"mem_dma_bd36_1", aie2::mem_dma_bd36_1},
      {"mem_dma_bd36_2", aie2::mem_dma_bd36_2},
      {"mem_dma_bd36_3", aie2::mem_dma_bd36_3},
      {"mem_dma_bd36_4", aie2::mem_dma_bd36_4},
      {"mem_dma_bd36_5", aie2::mem_dma_bd36_5},
      {"mem_dma_bd36_6", aie2::mem_dma_bd36_6},
      {"mem_dma_bd36_7", aie2::mem_dma_bd36_7},
      {"mem_dma_bd37_0", aie2::mem_dma_bd37_0},
      {"mem_dma_bd37_1", aie2::mem_dma_bd37_1},
      {"mem_dma_bd37_2", aie2::mem_dma_bd37_2},
      {"mem_dma_bd37_3", aie2::mem_dma_bd37_3},
      {"mem_dma_bd37_4", aie2::mem_dma_bd37_4},
      {"mem_dma_bd37_5", aie2::mem_dma_bd37_5},
      {"mem_dma_bd37_6", aie2::mem_dma_bd37_6},
      {"mem_dma_bd37_7", aie2::mem_dma_bd37_7},
      {"mem_dma_bd38_0", aie2::mem_dma_bd38_0},
      {"mem_dma_bd38_1", aie2::mem_dma_bd38_1},
      {"mem_dma_bd38_2", aie2::mem_dma_bd38_2},
      {"mem_dma_bd38_3", aie2::mem_dma_bd38_3},
      {"mem_dma_bd38_4", aie2::mem_dma_bd38_4},
      {"mem_dma_bd38_5", aie2::mem_dma_bd38_5},
      {"mem_dma_bd38_6", aie2::mem_dma_bd38_6},
      {"mem_dma_bd38_7", aie2::mem_dma_bd38_7},
      {"mem_dma_bd39_0", aie2::mem_dma_bd39_0},
      {"mem_dma_bd39_1", aie2::mem_dma_bd39_1},
      {"mem_dma_bd39_2", aie2::mem_dma_bd39_2},
      {"mem_dma_bd39_3", aie2::mem_dma_bd39_3},
      {"mem_dma_bd39_4", aie2::mem_dma_bd39_4},
      {"mem_dma_bd39_5", aie2::mem_dma_bd39_5},
      {"mem_dma_bd39_6", aie2::mem_dma_bd39_6},
      {"mem_dma_bd39_7", aie2::mem_dma_bd39_7},
      {"mem_dma_bd40_0", aie2::mem_dma_bd40_0},
      {"mem_dma_bd40_1", aie2::mem_dma_bd40_1},
      {"mem_dma_bd40_2", aie2::mem_dma_bd40_2},
      {"mem_dma_bd40_3", aie2::mem_dma_bd40_3},
      {"mem_dma_bd40_4", aie2::mem_dma_bd40_4},
      {"mem_dma_bd40_5", aie2::mem_dma_bd40_5},
      {"mem_dma_bd40_6", aie2::mem_dma_bd40_6},
      {"mem_dma_bd40_7", aie2::mem_dma_bd40_7},
      {"mem_dma_bd41_0", aie2::mem_dma_bd41_0},
      {"mem_dma_bd41_1", aie2::mem_dma_bd41_1},
      {"mem_dma_bd41_2", aie2::mem_dma_bd41_2},
      {"mem_dma_bd41_3", aie2::mem_dma_bd41_3},
      {"mem_dma_bd41_4", aie2::mem_dma_bd41_4},
      {"mem_dma_bd41_5", aie2::mem_dma_bd41_5},
      {"mem_dma_bd41_6", aie2::mem_dma_bd41_6},
      {"mem_dma_bd41_7", aie2::mem_dma_bd41_7},
      {"mem_dma_bd42_0", aie2::mem_dma_bd42_0},
      {"mem_dma_bd42_1", aie2::mem_dma_bd42_1},
      {"mem_dma_bd42_2", aie2::mem_dma_bd42_2},
      {"mem_dma_bd42_3", aie2::mem_dma_bd42_3},
      {"mem_dma_bd42_4", aie2::mem_dma_bd42_4},
      {"mem_dma_bd42_5", aie2::mem_dma_bd42_5},
      {"mem_dma_bd42_6", aie2::mem_dma_bd42_6},
      {"mem_dma_bd42_7", aie2::mem_dma_bd42_7},
      {"mem_dma_bd43_0", aie2::mem_dma_bd43_0},
      {"mem_dma_bd43_1", aie2::mem_dma_bd43_1},
      {"mem_dma_bd43_2", aie2::mem_dma_bd43_2},
      {"mem_dma_bd43_3", aie2::mem_dma_bd43_3},
      {"mem_dma_bd43_4", aie2::mem_dma_bd43_4},
      {"mem_dma_bd43_5", aie2::mem_dma_bd43_5},
      {"mem_dma_bd43_6", aie2::mem_dma_bd43_6},
      {"mem_dma_bd43_7", aie2::mem_dma_bd43_7},
      {"mem_dma_bd44_0", aie2::mem_dma_bd44_0},
      {"mem_dma_bd44_1", aie2::mem_dma_bd44_1},
      {"mem_dma_bd44_2", aie2::mem_dma_bd44_2},
      {"mem_dma_bd44_3", aie2::mem_dma_bd44_3},
      {"mem_dma_bd44_4", aie2::mem_dma_bd44_4},
      {"mem_dma_bd44_5", aie2::mem_dma_bd44_5},
      {"mem_dma_bd44_6", aie2::mem_dma_bd44_6},
      {"mem_dma_bd44_7", aie2::mem_dma_bd44_7},
      {"mem_dma_bd45_0", aie2::mem_dma_bd45_0},
      {"mem_dma_bd45_1", aie2::mem_dma_bd45_1},
      {"mem_dma_bd45_2", aie2::mem_dma_bd45_2},
      {"mem_dma_bd45_3", aie2::mem_dma_bd45_3},
      {"mem_dma_bd45_4", aie2::mem_dma_bd45_4},
      {"mem_dma_bd45_5", aie2::mem_dma_bd45_5},
      {"mem_dma_bd45_6", aie2::mem_dma_bd45_6},
      {"mem_dma_bd45_7", aie2::mem_dma_bd45_7},
      {"mem_dma_bd46_0", aie2::mem_dma_bd46_0},
      {"mem_dma_bd46_1", aie2::mem_dma_bd46_1},
      {"mem_dma_bd46_2", aie2::mem_dma_bd46_2},
      {"mem_dma_bd46_3", aie2::mem_dma_bd46_3},
      {"mem_dma_bd46_4", aie2::mem_dma_bd46_4},
      {"mem_dma_bd46_5", aie2::mem_dma_bd46_5},
      {"mem_dma_bd46_6", aie2::mem_dma_bd46_6},
      {"mem_dma_bd46_7", aie2::mem_dma_bd46_7},
      {"mem_dma_bd47_0", aie2::mem_dma_bd47_0},
      {"mem_dma_bd47_1", aie2::mem_dma_bd47_1},
      {"mem_dma_bd47_2", aie2::mem_dma_bd47_2},
      {"mem_dma_bd47_3", aie2::mem_dma_bd47_3},
      {"mem_dma_bd47_4", aie2::mem_dma_bd47_4},
      {"mem_dma_bd47_5", aie2::mem_dma_bd47_5},
      {"mem_dma_bd47_6", aie2::mem_dma_bd47_6},
      {"mem_dma_bd47_7", aie2::mem_dma_bd47_7},
      {"mem_dma_s2mm_0_ctrl", aie2::mem_dma_s2mm_0_ctrl},
      {"mem_dma_s2mm_0_start_queue", aie2::mem_dma_s2mm_0_start_queue},
      {"mem_dma_s2mm_1_ctrl", aie2::mem_dma_s2mm_1_ctrl},
      {"mem_dma_s2mm_1_start_queue", aie2::mem_dma_s2mm_1_start_queue},
      {"mem_dma_s2mm_2_ctrl", aie2::mem_dma_s2mm_2_ctrl},
      {"mem_dma_s2mm_2_start_queue", aie2::mem_dma_s2mm_2_start_queue},
      {"mem_dma_s2mm_3_ctrl", aie2::mem_dma_s2mm_3_ctrl},
      {"mem_dma_s2mm_3_start_queue", aie2::mem_dma_s2mm_3_start_queue},
      {"mem_dma_s2mm_4_ctrl", aie2::mem_dma_s2mm_4_ctrl},
      {"mem_dma_s2mm_4_start_queue", aie2::mem_dma_s2mm_4_start_queue},
      {"mem_dma_s2mm_5_ctrl", aie2::mem_dma_s2mm_5_ctrl},
      {"mem_dma_s2mm_5_start_queue", aie2::mem_dma_s2mm_5_start_queue},
      {"mem_dma_mm2s_0_ctrl", aie2::mem_dma_mm2s_0_ctrl},
      {"mem_dma_mm2s_0_start_queue", aie2::mem_dma_mm2s_0_start_queue},
      {"mem_dma_mm2s_1_ctrl", aie2::mem_dma_mm2s_1_ctrl},
      {"mem_dma_mm2s_1_start_queue", aie2::mem_dma_mm2s_1_start_queue},
      {"mem_dma_mm2s_2_ctrl", aie2::mem_dma_mm2s_2_ctrl},
      {"mem_dma_mm2s_2_start_queue", aie2::mem_dma_mm2s_2_start_queue},
      {"mem_dma_mm2s_3_ctrl", aie2::mem_dma_mm2s_3_ctrl},
      {"mem_dma_mm2s_3_start_queue", aie2::mem_dma_mm2s_3_start_queue},
      {"mem_dma_mm2s_4_ctrl", aie2::mem_dma_mm2s_4_ctrl},
      {"mem_dma_mm2s_4_start_queue", aie2::mem_dma_mm2s_4_start_queue},
      {"mem_dma_mm2s_5_ctrl", aie2::mem_dma_mm2s_5_ctrl},
      {"mem_dma_mm2s_5_start_queue", aie2::mem_dma_mm2s_5_start_queue},
      {"mem_dma_s2mm_status_0", aie2::mem_dma_s2mm_status_0},
      {"mem_dma_s2mm_status_1", aie2::mem_dma_s2mm_status_1},
      {"mem_dma_s2mm_status_2", aie2::mem_dma_s2mm_status_2},
      {"mem_dma_s2mm_status_3", aie2::mem_dma_s2mm_status_3},
      {"mem_dma_s2mm_status_4", aie2::mem_dma_s2mm_status_4},
      {"mem_dma_s2mm_status_5", aie2::mem_dma_s2mm_status_5},
      {"mem_dma_mm2s_status_0", aie2::mem_dma_mm2s_status_0},
      {"mem_dma_mm2s_status_1", aie2::mem_dma_mm2s_status_1},
      {"mem_dma_mm2s_status_2", aie2::mem_dma_mm2s_status_2},
      {"mem_dma_mm2s_status_3", aie2::mem_dma_mm2s_status_3},
      {"mem_dma_mm2s_status_4", aie2::mem_dma_mm2s_status_4},
      {"mem_dma_mm2s_status_5", aie2::mem_dma_mm2s_status_5},
      {"mem_dma_event_channel_selection", aie2::mem_dma_event_channel_selection},
      {"mem_dma_s2mm_current_write_count_0", aie2::mem_dma_s2mm_current_write_count_0},
      {"mem_dma_s2mm_current_write_count_1", aie2::mem_dma_s2mm_current_write_count_1},
      {"mem_dma_s2mm_current_write_count_2", aie2::mem_dma_s2mm_current_write_count_2},
      {"mem_dma_s2mm_current_write_count_3", aie2::mem_dma_s2mm_current_write_count_3},
      {"mem_dma_s2mm_current_write_count_4", aie2::mem_dma_s2mm_current_write_count_4},
      {"mem_dma_s2mm_current_write_count_5", aie2::mem_dma_s2mm_current_write_count_5},
      {"mem_dma_s2mm_fot_count_fifo_pop_0", aie2::mem_dma_s2mm_fot_count_fifo_pop_0},
      {"mem_dma_s2mm_fot_count_fifo_pop_1", aie2::mem_dma_s2mm_fot_count_fifo_pop_1},
      {"mem_dma_s2mm_fot_count_fifo_pop_2", aie2::mem_dma_s2mm_fot_count_fifo_pop_2},
      {"mem_dma_s2mm_fot_count_fifo_pop_3", aie2::mem_dma_s2mm_fot_count_fifo_pop_3},
      {"mem_dma_s2mm_fot_count_fifo_pop_4", aie2::mem_dma_s2mm_fot_count_fifo_pop_4},
      {"mem_dma_s2mm_fot_count_fifo_pop_5", aie2::mem_dma_s2mm_fot_count_fifo_pop_5},
      {"mem_stream_switch_master_config_dma0", aie2::mem_stream_switch_master_config_dma0},
      {"mem_stream_switch_master_config_dma1", aie2::mem_stream_switch_master_config_dma1},
      {"mem_stream_switch_master_config_dma2", aie2::mem_stream_switch_master_config_dma2},
      {"mem_stream_switch_master_config_dma3", aie2::mem_stream_switch_master_config_dma3},
      {"mem_stream_switch_master_config_dma4", aie2::mem_stream_switch_master_config_dma4},
      {"mem_stream_switch_master_config_dma5", aie2::mem_stream_switch_master_config_dma5},
      {"mem_stream_switch_master_config_tile_ctrl", aie2::mem_stream_switch_master_config_tile_ctrl},
      {"mem_stream_switch_master_config_south0", aie2::mem_stream_switch_master_config_south0},
      {"mem_stream_switch_master_config_south1", aie2::mem_stream_switch_master_config_south1},
      {"mem_stream_switch_master_config_south2", aie2::mem_stream_switch_master_config_south2},
      {"mem_stream_switch_master_config_south3", aie2::mem_stream_switch_master_config_south3},
      {"mem_stream_switch_master_config_north0", aie2::mem_stream_switch_master_config_north0},
      {"mem_stream_switch_master_config_north1", aie2::mem_stream_switch_master_config_north1},
      {"mem_stream_switch_master_config_north2", aie2::mem_stream_switch_master_config_north2},
      {"mem_stream_switch_master_config_north3", aie2::mem_stream_switch_master_config_north3},
      {"mem_stream_switch_master_config_north4", aie2::mem_stream_switch_master_config_north4},
      {"mem_stream_switch_master_config_north5", aie2::mem_stream_switch_master_config_north5},
      {"mem_stream_switch_slave_config_dma_0", aie2::mem_stream_switch_slave_config_dma_0},
      {"mem_stream_switch_slave_config_dma_1", aie2::mem_stream_switch_slave_config_dma_1},
      {"mem_stream_switch_slave_config_dma_2", aie2::mem_stream_switch_slave_config_dma_2},
      {"mem_stream_switch_slave_config_dma_3", aie2::mem_stream_switch_slave_config_dma_3},
      {"mem_stream_switch_slave_config_dma_4", aie2::mem_stream_switch_slave_config_dma_4},
      {"mem_stream_switch_slave_config_dma_5", aie2::mem_stream_switch_slave_config_dma_5},
      {"mem_stream_switch_slave_config_tile_ctrl", aie2::mem_stream_switch_slave_config_tile_ctrl},
      {"mem_stream_switch_slave_config_south_0", aie2::mem_stream_switch_slave_config_south_0},
      {"mem_stream_switch_slave_config_south_1", aie2::mem_stream_switch_slave_config_south_1},
      {"mem_stream_switch_slave_config_south_2", aie2::mem_stream_switch_slave_config_south_2},
      {"mem_stream_switch_slave_config_south_3", aie2::mem_stream_switch_slave_config_south_3},
      {"mem_stream_switch_slave_config_south_4", aie2::mem_stream_switch_slave_config_south_4},
      {"mem_stream_switch_slave_config_south_5", aie2::mem_stream_switch_slave_config_south_5},
      {"mem_stream_switch_slave_config_north_0", aie2::mem_stream_switch_slave_config_north_0},
      {"mem_stream_switch_slave_config_north_1", aie2::mem_stream_switch_slave_config_north_1},
      {"mem_stream_switch_slave_config_north_2", aie2::mem_stream_switch_slave_config_north_2},
      {"mem_stream_switch_slave_config_north_3", aie2::mem_stream_switch_slave_config_north_3},
      {"mem_stream_switch_slave_config_trace", aie2::mem_stream_switch_slave_config_trace},
      {"mem_stream_switch_slave_dma_0_slot0", aie2::mem_stream_switch_slave_dma_0_slot0},
      {"mem_stream_switch_slave_dma_0_slot1", aie2::mem_stream_switch_slave_dma_0_slot1},
      {"mem_stream_switch_slave_dma_0_slot2", aie2::mem_stream_switch_slave_dma_0_slot2},
      {"mem_stream_switch_slave_dma_0_slot3", aie2::mem_stream_switch_slave_dma_0_slot3},
      {"mem_stream_switch_slave_dma_1_slot0", aie2::mem_stream_switch_slave_dma_1_slot0},
      {"mem_stream_switch_slave_dma_1_slot1", aie2::mem_stream_switch_slave_dma_1_slot1},
      {"mem_stream_switch_slave_dma_1_slot2", aie2::mem_stream_switch_slave_dma_1_slot2},
      {"mem_stream_switch_slave_dma_1_slot3", aie2::mem_stream_switch_slave_dma_1_slot3},
      {"mem_stream_switch_slave_dma_2_slot0", aie2::mem_stream_switch_slave_dma_2_slot0},
      {"mem_stream_switch_slave_dma_2_slot1", aie2::mem_stream_switch_slave_dma_2_slot1},
      {"mem_stream_switch_slave_dma_2_slot2", aie2::mem_stream_switch_slave_dma_2_slot2},
      {"mem_stream_switch_slave_dma_2_slot3", aie2::mem_stream_switch_slave_dma_2_slot3},
      {"mem_stream_switch_slave_dma_3_slot0", aie2::mem_stream_switch_slave_dma_3_slot0},
      {"mem_stream_switch_slave_dma_3_slot1", aie2::mem_stream_switch_slave_dma_3_slot1},
      {"mem_stream_switch_slave_dma_3_slot2", aie2::mem_stream_switch_slave_dma_3_slot2},
      {"mem_stream_switch_slave_dma_3_slot3", aie2::mem_stream_switch_slave_dma_3_slot3},
      {"mem_stream_switch_slave_dma_4_slot0", aie2::mem_stream_switch_slave_dma_4_slot0},
      {"mem_stream_switch_slave_dma_4_slot1", aie2::mem_stream_switch_slave_dma_4_slot1},
      {"mem_stream_switch_slave_dma_4_slot2", aie2::mem_stream_switch_slave_dma_4_slot2},
      {"mem_stream_switch_slave_dma_4_slot3", aie2::mem_stream_switch_slave_dma_4_slot3},
      {"mem_stream_switch_slave_dma_5_slot0", aie2::mem_stream_switch_slave_dma_5_slot0},
      {"mem_stream_switch_slave_dma_5_slot1", aie2::mem_stream_switch_slave_dma_5_slot1},
      {"mem_stream_switch_slave_dma_5_slot2", aie2::mem_stream_switch_slave_dma_5_slot2},
      {"mem_stream_switch_slave_dma_5_slot3", aie2::mem_stream_switch_slave_dma_5_slot3},
      {"mem_stream_switch_slave_tile_ctrl_slot0", aie2::mem_stream_switch_slave_tile_ctrl_slot0},
      {"mem_stream_switch_slave_tile_ctrl_slot1", aie2::mem_stream_switch_slave_tile_ctrl_slot1},
      {"mem_stream_switch_slave_tile_ctrl_slot2", aie2::mem_stream_switch_slave_tile_ctrl_slot2},
      {"mem_stream_switch_slave_tile_ctrl_slot3", aie2::mem_stream_switch_slave_tile_ctrl_slot3},
      {"mem_stream_switch_slave_south_0_slot0", aie2::mem_stream_switch_slave_south_0_slot0},
      {"mem_stream_switch_slave_south_0_slot1", aie2::mem_stream_switch_slave_south_0_slot1},
      {"mem_stream_switch_slave_south_0_slot2", aie2::mem_stream_switch_slave_south_0_slot2},
      {"mem_stream_switch_slave_south_0_slot3", aie2::mem_stream_switch_slave_south_0_slot3},
      {"mem_stream_switch_slave_south_1_slot0", aie2::mem_stream_switch_slave_south_1_slot0},
      {"mem_stream_switch_slave_south_1_slot1", aie2::mem_stream_switch_slave_south_1_slot1},
      {"mem_stream_switch_slave_south_1_slot2", aie2::mem_stream_switch_slave_south_1_slot2},
      {"mem_stream_switch_slave_south_1_slot3", aie2::mem_stream_switch_slave_south_1_slot3},
      {"mem_stream_switch_slave_south_2_slot0", aie2::mem_stream_switch_slave_south_2_slot0},
      {"mem_stream_switch_slave_south_2_slot1", aie2::mem_stream_switch_slave_south_2_slot1},
      {"mem_stream_switch_slave_south_2_slot2", aie2::mem_stream_switch_slave_south_2_slot2},
      {"mem_stream_switch_slave_south_2_slot3", aie2::mem_stream_switch_slave_south_2_slot3},
      {"mem_stream_switch_slave_south_3_slot0", aie2::mem_stream_switch_slave_south_3_slot0},
      {"mem_stream_switch_slave_south_3_slot1", aie2::mem_stream_switch_slave_south_3_slot1},
      {"mem_stream_switch_slave_south_3_slot2", aie2::mem_stream_switch_slave_south_3_slot2},
      {"mem_stream_switch_slave_south_3_slot3", aie2::mem_stream_switch_slave_south_3_slot3},
      {"mem_stream_switch_slave_south_4_slot0", aie2::mem_stream_switch_slave_south_4_slot0},
      {"mem_stream_switch_slave_south_4_slot1", aie2::mem_stream_switch_slave_south_4_slot1},
      {"mem_stream_switch_slave_south_4_slot2", aie2::mem_stream_switch_slave_south_4_slot2},
      {"mem_stream_switch_slave_south_4_slot3", aie2::mem_stream_switch_slave_south_4_slot3},
      {"mem_stream_switch_slave_south_5_slot0", aie2::mem_stream_switch_slave_south_5_slot0},
      {"mem_stream_switch_slave_south_5_slot1", aie2::mem_stream_switch_slave_south_5_slot1},
      {"mem_stream_switch_slave_south_5_slot2", aie2::mem_stream_switch_slave_south_5_slot2},
      {"mem_stream_switch_slave_south_5_slot3", aie2::mem_stream_switch_slave_south_5_slot3},
      {"mem_stream_switch_slave_north_0_slot0", aie2::mem_stream_switch_slave_north_0_slot0},
      {"mem_stream_switch_slave_north_0_slot1", aie2::mem_stream_switch_slave_north_0_slot1},
      {"mem_stream_switch_slave_north_0_slot2", aie2::mem_stream_switch_slave_north_0_slot2},
      {"mem_stream_switch_slave_north_0_slot3", aie2::mem_stream_switch_slave_north_0_slot3},
      {"mem_stream_switch_slave_north_1_slot0", aie2::mem_stream_switch_slave_north_1_slot0},
      {"mem_stream_switch_slave_north_1_slot1", aie2::mem_stream_switch_slave_north_1_slot1},
      {"mem_stream_switch_slave_north_1_slot2", aie2::mem_stream_switch_slave_north_1_slot2},
      {"mem_stream_switch_slave_north_1_slot3", aie2::mem_stream_switch_slave_north_1_slot3},
      {"mem_stream_switch_slave_north_2_slot0", aie2::mem_stream_switch_slave_north_2_slot0},
      {"mem_stream_switch_slave_north_2_slot1", aie2::mem_stream_switch_slave_north_2_slot1},
      {"mem_stream_switch_slave_north_2_slot2", aie2::mem_stream_switch_slave_north_2_slot2},
      {"mem_stream_switch_slave_north_2_slot3", aie2::mem_stream_switch_slave_north_2_slot3},
      {"mem_stream_switch_slave_north_3_slot0", aie2::mem_stream_switch_slave_north_3_slot0},
      {"mem_stream_switch_slave_north_3_slot1", aie2::mem_stream_switch_slave_north_3_slot1},
      {"mem_stream_switch_slave_north_3_slot2", aie2::mem_stream_switch_slave_north_3_slot2},
      {"mem_stream_switch_slave_north_3_slot3", aie2::mem_stream_switch_slave_north_3_slot3},
      {"mem_stream_switch_slave_trace_slot0", aie2::mem_stream_switch_slave_trace_slot0},
      {"mem_stream_switch_slave_trace_slot1", aie2::mem_stream_switch_slave_trace_slot1},
      {"mem_stream_switch_slave_trace_slot2", aie2::mem_stream_switch_slave_trace_slot2},
      {"mem_stream_switch_slave_trace_slot3", aie2::mem_stream_switch_slave_trace_slot3},
      {"mem_stream_switch_deterministic_merge_arb0_slave0_1", aie2::mem_stream_switch_deterministic_merge_arb0_slave0_1},
      {"mem_stream_switch_deterministic_merge_arb0_slave2_3", aie2::mem_stream_switch_deterministic_merge_arb0_slave2_3},
      {"mem_stream_switch_deterministic_merge_arb0_ctrl", aie2::mem_stream_switch_deterministic_merge_arb0_ctrl},
      {"mem_stream_switch_deterministic_merge_arb1_slave0_1", aie2::mem_stream_switch_deterministic_merge_arb1_slave0_1},
      {"mem_stream_switch_deterministic_merge_arb1_slave2_3", aie2::mem_stream_switch_deterministic_merge_arb1_slave2_3},
      {"mem_stream_switch_deterministic_merge_arb1_ctrl", aie2::mem_stream_switch_deterministic_merge_arb1_ctrl},
      {"mem_stream_switch_event_port_selection_0", aie2::mem_stream_switch_event_port_selection_0},
      {"mem_stream_switch_event_port_selection_1", aie2::mem_stream_switch_event_port_selection_1},
      {"mem_stream_switch_parity_status", aie2::mem_stream_switch_parity_status},
      {"mem_stream_switch_parity_injection", aie2::mem_stream_switch_parity_injection},
      {"mem_tile_control_packet_handler_status", aie2::mem_tile_control_packet_handler_status},
      {"mem_stream_switch_adaptive_clock_gate_status", aie2::mem_stream_switch_adaptive_clock_gate_status},
      {"mem_stream_switch_adaptive_clock_gate_abort_period", aie2::mem_stream_switch_adaptive_clock_gate_abort_period},
      {"mem_lock0_value", aie2::mem_lock0_value},
      {"mem_lock1_value", aie2::mem_lock1_value},
      {"mem_lock2_value", aie2::mem_lock2_value},
      {"mem_lock3_value", aie2::mem_lock3_value},
      {"mem_lock4_value", aie2::mem_lock4_value},
      {"mem_lock5_value", aie2::mem_lock5_value},
      {"mem_lock6_value", aie2::mem_lock6_value},
      {"mem_lock7_value", aie2::mem_lock7_value},
      {"mem_lock8_value", aie2::mem_lock8_value},
      {"mem_lock9_value", aie2::mem_lock9_value},
      {"mem_lock10_value", aie2::mem_lock10_value},
      {"mem_lock11_value", aie2::mem_lock11_value},
      {"mem_lock12_value", aie2::mem_lock12_value},
      {"mem_lock13_value", aie2::mem_lock13_value},
      {"mem_lock14_value", aie2::mem_lock14_value},
      {"mem_lock15_value", aie2::mem_lock15_value},
      {"mem_lock16_value", aie2::mem_lock16_value},
      {"mem_lock17_value", aie2::mem_lock17_value},
      {"mem_lock18_value", aie2::mem_lock18_value},
      {"mem_lock19_value", aie2::mem_lock19_value},
      {"mem_lock20_value", aie2::mem_lock20_value},
      {"mem_lock21_value", aie2::mem_lock21_value},
      {"mem_lock22_value", aie2::mem_lock22_value},
      {"mem_lock23_value", aie2::mem_lock23_value},
      {"mem_lock24_value", aie2::mem_lock24_value},
      {"mem_lock25_value", aie2::mem_lock25_value},
      {"mem_lock26_value", aie2::mem_lock26_value},
      {"mem_lock27_value", aie2::mem_lock27_value},
      {"mem_lock28_value", aie2::mem_lock28_value},
      {"mem_lock29_value", aie2::mem_lock29_value},
      {"mem_lock30_value", aie2::mem_lock30_value},
      {"mem_lock31_value", aie2::mem_lock31_value},
      {"mem_lock32_value", aie2::mem_lock32_value},
      {"mem_lock33_value", aie2::mem_lock33_value},
      {"mem_lock34_value", aie2::mem_lock34_value},
      {"mem_lock35_value", aie2::mem_lock35_value},
      {"mem_lock36_value", aie2::mem_lock36_value},
      {"mem_lock37_value", aie2::mem_lock37_value},
      {"mem_lock38_value", aie2::mem_lock38_value},
      {"mem_lock39_value", aie2::mem_lock39_value},
      {"mem_lock40_value", aie2::mem_lock40_value},
      {"mem_lock41_value", aie2::mem_lock41_value},
      {"mem_lock42_value", aie2::mem_lock42_value},
      {"mem_lock43_value", aie2::mem_lock43_value},
      {"mem_lock44_value", aie2::mem_lock44_value},
      {"mem_lock45_value", aie2::mem_lock45_value},
      {"mem_lock46_value", aie2::mem_lock46_value},
      {"mem_lock47_value", aie2::mem_lock47_value},
      {"mem_lock48_value", aie2::mem_lock48_value},
      {"mem_lock49_value", aie2::mem_lock49_value},
      {"mem_lock50_value", aie2::mem_lock50_value},
      {"mem_lock51_value", aie2::mem_lock51_value},
      {"mem_lock52_value", aie2::mem_lock52_value},
      {"mem_lock53_value", aie2::mem_lock53_value},
      {"mem_lock54_value", aie2::mem_lock54_value},
      {"mem_lock55_value", aie2::mem_lock55_value},
      {"mem_lock56_value", aie2::mem_lock56_value},
      {"mem_lock57_value", aie2::mem_lock57_value},
      {"mem_lock58_value", aie2::mem_lock58_value},
      {"mem_lock59_value", aie2::mem_lock59_value},
      {"mem_lock60_value", aie2::mem_lock60_value},
      {"mem_lock61_value", aie2::mem_lock61_value},
      {"mem_lock62_value", aie2::mem_lock62_value},
      {"mem_lock63_value", aie2::mem_lock63_value},
      {"mem_locks_event_selection_0", aie2::mem_locks_event_selection_0},
      {"mem_locks_event_selection_1", aie2::mem_locks_event_selection_1},
      {"mem_locks_event_selection_2", aie2::mem_locks_event_selection_2},
      {"mem_locks_event_selection_3", aie2::mem_locks_event_selection_3},
      {"mem_locks_event_selection_4", aie2::mem_locks_event_selection_4},
      {"mem_locks_event_selection_5", aie2::mem_locks_event_selection_5},
      {"mem_locks_event_selection_6", aie2::mem_locks_event_selection_6},
      {"mem_locks_event_selection_7", aie2::mem_locks_event_selection_7},
      {"mem_locks_overflow_0", aie2::mem_locks_overflow_0},
      {"mem_locks_overflow_1", aie2::mem_locks_overflow_1},
      {"mem_locks_underflow_0", aie2::mem_locks_underflow_0},
      {"mem_locks_underflow_1", aie2::mem_locks_underflow_1},
      {"mem_lock_request", aie2::mem_lock_request},
      {"mem_module_clock_control", aie2::mem_module_clock_control},
      {"mem_module_reset_control", aie2::mem_module_reset_control},
      {"shim_lock_step_size", aie2::shim_lock_step_size},
      {"shim_dma_bd_step_size", aie2::shim_dma_bd_step_size},
      {"shim_dma_s2mm_step_size", aie2::shim_dma_s2mm_step_size},
      {"shim_lock0_value", aie2::shim_lock0_value},
      {"shim_lock1_value", aie2::shim_lock1_value},
      {"shim_lock2_value", aie2::shim_lock2_value},
      {"shim_lock3_value", aie2::shim_lock3_value},
      {"shim_lock4_value", aie2::shim_lock4_value},
      {"shim_lock5_value", aie2::shim_lock5_value},
      {"shim_lock6_value", aie2::shim_lock6_value},
      {"shim_lock7_value", aie2::shim_lock7_value},
      {"shim_lock8_value", aie2::shim_lock8_value},
      {"shim_lock9_value", aie2::shim_lock9_value},
      {"shim_lock10_value", aie2::shim_lock10_value},
      {"shim_lock11_value", aie2::shim_lock11_value},
      {"shim_lock12_value", aie2::shim_lock12_value},
      {"shim_lock13_value", aie2::shim_lock13_value},
      {"shim_lock14_value", aie2::shim_lock14_value},
      {"shim_lock15_value", aie2::shim_lock15_value},
      {"shim_dma_bd0_0", aie2::shim_dma_bd0_0},
      {"shim_dma_bd0_1", aie2::shim_dma_bd0_1},
      {"shim_dma_bd0_2", aie2::shim_dma_bd0_2},
      {"shim_dma_bd0_3", aie2::shim_dma_bd0_3},
      {"shim_dma_bd0_4", aie2::shim_dma_bd0_4},
      {"shim_dma_bd0_5", aie2::shim_dma_bd0_5},
      {"shim_dma_bd0_6", aie2::shim_dma_bd0_6},
      {"shim_dma_bd0_7", aie2::shim_dma_bd0_7},
      {"shim_dma_bd1_0", aie2::shim_dma_bd1_0},
      {"shim_dma_bd1_1", aie2::shim_dma_bd1_1},
      {"shim_dma_bd1_2", aie2::shim_dma_bd1_2},
      {"shim_dma_bd1_3", aie2::shim_dma_bd1_3},
      {"shim_dma_bd1_4", aie2::shim_dma_bd1_4},
      {"shim_dma_bd1_5", aie2::shim_dma_bd1_5},
      {"shim_dma_bd1_6", aie2::shim_dma_bd1_6},
      {"shim_dma_bd1_7", aie2::shim_dma_bd1_7},
      {"shim_dma_bd2_0", aie2::shim_dma_bd2_0},
      {"shim_dma_bd2_1", aie2::shim_dma_bd2_1},
      {"shim_dma_bd2_2", aie2::shim_dma_bd2_2},
      {"shim_dma_bd2_3", aie2::shim_dma_bd2_3},
      {"shim_dma_bd2_4", aie2::shim_dma_bd2_4},
      {"shim_dma_bd2_5", aie2::shim_dma_bd2_5},
      {"shim_dma_bd2_6", aie2::shim_dma_bd2_6},
      {"shim_dma_bd2_7", aie2::shim_dma_bd2_7},
      {"shim_dma_bd3_0", aie2::shim_dma_bd3_0},
      {"shim_dma_bd3_1", aie2::shim_dma_bd3_1},
      {"shim_dma_bd3_2", aie2::shim_dma_bd3_2},
      {"shim_dma_bd3_3", aie2::shim_dma_bd3_3},
      {"shim_dma_bd3_4", aie2::shim_dma_bd3_4},
      {"shim_dma_bd3_5", aie2::shim_dma_bd3_5},
      {"shim_dma_bd3_6", aie2::shim_dma_bd3_6},
      {"shim_dma_bd3_7", aie2::shim_dma_bd3_7},
      {"shim_dma_bd4_0", aie2::shim_dma_bd4_0},
      {"shim_dma_bd4_1", aie2::shim_dma_bd4_1},
      {"shim_dma_bd4_2", aie2::shim_dma_bd4_2},
      {"shim_dma_bd4_3", aie2::shim_dma_bd4_3},
      {"shim_dma_bd4_4", aie2::shim_dma_bd4_4},
      {"shim_dma_bd4_5", aie2::shim_dma_bd4_5},
      {"shim_dma_bd4_6", aie2::shim_dma_bd4_6},
      {"shim_dma_bd4_7", aie2::shim_dma_bd4_7},
      {"shim_dma_bd5_0", aie2::shim_dma_bd5_0},
      {"shim_dma_bd5_1", aie2::shim_dma_bd5_1},
      {"shim_dma_bd5_2", aie2::shim_dma_bd5_2},
      {"shim_dma_bd5_3", aie2::shim_dma_bd5_3},
      {"shim_dma_bd5_4", aie2::shim_dma_bd5_4},
      {"shim_dma_bd5_5", aie2::shim_dma_bd5_5},
      {"shim_dma_bd5_6", aie2::shim_dma_bd5_6},
      {"shim_dma_bd5_7", aie2::shim_dma_bd5_7},
      {"shim_dma_bd6_0", aie2::shim_dma_bd6_0},
      {"shim_dma_bd6_1", aie2::shim_dma_bd6_1},
      {"shim_dma_bd6_2", aie2::shim_dma_bd6_2},
      {"shim_dma_bd6_3", aie2::shim_dma_bd6_3},
      {"shim_dma_bd6_4", aie2::shim_dma_bd6_4},
      {"shim_dma_bd6_5", aie2::shim_dma_bd6_5},
      {"shim_dma_bd6_6", aie2::shim_dma_bd6_6},
      {"shim_dma_bd6_7", aie2::shim_dma_bd6_7},
      {"shim_dma_bd7_0", aie2::shim_dma_bd7_0},
      {"shim_dma_bd7_1", aie2::shim_dma_bd7_1},
      {"shim_dma_bd7_2", aie2::shim_dma_bd7_2},
      {"shim_dma_bd7_3", aie2::shim_dma_bd7_3},
      {"shim_dma_bd7_4", aie2::shim_dma_bd7_4},
      {"shim_dma_bd7_5", aie2::shim_dma_bd7_5},
      {"shim_dma_bd7_6", aie2::shim_dma_bd7_6},
      {"shim_dma_bd7_7", aie2::shim_dma_bd7_7},
      {"shim_dma_bd8_0", aie2::shim_dma_bd8_0},
      {"shim_dma_bd8_1", aie2::shim_dma_bd8_1},
      {"shim_dma_bd8_2", aie2::shim_dma_bd8_2},
      {"shim_dma_bd8_3", aie2::shim_dma_bd8_3},
      {"shim_dma_bd8_4", aie2::shim_dma_bd8_4},
      {"shim_dma_bd8_5", aie2::shim_dma_bd8_5},
      {"shim_dma_bd8_6", aie2::shim_dma_bd8_6},
      {"shim_dma_bd8_7", aie2::shim_dma_bd8_7},
      {"shim_dma_bd9_0", aie2::shim_dma_bd9_0},
      {"shim_dma_bd9_1", aie2::shim_dma_bd9_1},
      {"shim_dma_bd9_2", aie2::shim_dma_bd9_2},
      {"shim_dma_bd9_3", aie2::shim_dma_bd9_3},
      {"shim_dma_bd9_4", aie2::shim_dma_bd9_4},
      {"shim_dma_bd9_5", aie2::shim_dma_bd9_5},
      {"shim_dma_bd9_6", aie2::shim_dma_bd9_6},
      {"shim_dma_bd9_7", aie2::shim_dma_bd9_7},
      {"shim_dma_bd10_0", aie2::shim_dma_bd10_0},
      {"shim_dma_bd10_1", aie2::shim_dma_bd10_1},
      {"shim_dma_bd10_2", aie2::shim_dma_bd10_2},
      {"shim_dma_bd10_3", aie2::shim_dma_bd10_3},
      {"shim_dma_bd10_4", aie2::shim_dma_bd10_4},
      {"shim_dma_bd10_5", aie2::shim_dma_bd10_5},
      {"shim_dma_bd10_6", aie2::shim_dma_bd10_6},
      {"shim_dma_bd10_7", aie2::shim_dma_bd10_7},
      {"shim_dma_bd11_0", aie2::shim_dma_bd11_0},
      {"shim_dma_bd11_1", aie2::shim_dma_bd11_1},
      {"shim_dma_bd11_2", aie2::shim_dma_bd11_2},
      {"shim_dma_bd11_3", aie2::shim_dma_bd11_3},
      {"shim_dma_bd11_4", aie2::shim_dma_bd11_4},
      {"shim_dma_bd11_5", aie2::shim_dma_bd11_5},
      {"shim_dma_bd11_6", aie2::shim_dma_bd11_6},
      {"shim_dma_bd11_7", aie2::shim_dma_bd11_7},
      {"shim_dma_bd12_0", aie2::shim_dma_bd12_0},
      {"shim_dma_bd12_1", aie2::shim_dma_bd12_1},
      {"shim_dma_bd12_2", aie2::shim_dma_bd12_2},
      {"shim_dma_bd12_3", aie2::shim_dma_bd12_3},
      {"shim_dma_bd12_4", aie2::shim_dma_bd12_4},
      {"shim_dma_bd12_5", aie2::shim_dma_bd12_5},
      {"shim_dma_bd12_6", aie2::shim_dma_bd12_6},
      {"shim_dma_bd12_7", aie2::shim_dma_bd12_7},
      {"shim_dma_bd13_0", aie2::shim_dma_bd13_0},
      {"shim_dma_bd13_1", aie2::shim_dma_bd13_1},
      {"shim_dma_bd13_2", aie2::shim_dma_bd13_2},
      {"shim_dma_bd13_3", aie2::shim_dma_bd13_3},
      {"shim_dma_bd13_4", aie2::shim_dma_bd13_4},
      {"shim_dma_bd13_5", aie2::shim_dma_bd13_5},
      {"shim_dma_bd13_6", aie2::shim_dma_bd13_6},
      {"shim_dma_bd13_7", aie2::shim_dma_bd13_7},
      {"shim_dma_bd14_0", aie2::shim_dma_bd14_0},
      {"shim_dma_bd14_1", aie2::shim_dma_bd14_1},
      {"shim_dma_bd14_2", aie2::shim_dma_bd14_2},
      {"shim_dma_bd14_3", aie2::shim_dma_bd14_3},
      {"shim_dma_bd14_4", aie2::shim_dma_bd14_4},
      {"shim_dma_bd14_5", aie2::shim_dma_bd14_5},
      {"shim_dma_bd14_6", aie2::shim_dma_bd14_6},
      {"shim_dma_bd14_7", aie2::shim_dma_bd14_7},
      {"shim_dma_bd15_0", aie2::shim_dma_bd15_0},
      {"shim_dma_bd15_1", aie2::shim_dma_bd15_1},
      {"shim_dma_bd15_2", aie2::shim_dma_bd15_2},
      {"shim_dma_bd15_3", aie2::shim_dma_bd15_3},
      {"shim_dma_bd15_4", aie2::shim_dma_bd15_4},
      {"shim_dma_bd15_5", aie2::shim_dma_bd15_5},
      {"shim_dma_bd15_6", aie2::shim_dma_bd15_6},
      {"shim_dma_bd15_7", aie2::shim_dma_bd15_7},
      {"shim_dma_s2mm_0_ctrl", aie2::shim_dma_s2mm_0_ctrl},
      {"shim_dma_s2mm_0_task_queue", aie2::shim_dma_s2mm_0_task_queue},
      {"shim_dma_s2mm_status_0", aie2::shim_dma_s2mm_status_0},
      {"shim_dma_s2mm_status_1", aie2::shim_dma_s2mm_status_1},
      {"shim_dma_mm2s_status_0", aie2::shim_dma_mm2s_status_0},
      {"shim_dma_mm2s_status_1", aie2::shim_dma_mm2s_status_1},
      {"shim_performance_control0", aie2::shim_performance_control0},
      {"shim_performance_control1", aie2::shim_performance_control1},
      {"shim_performance_counter0", aie2::shim_performance_counter0},
      {"shim_performance_counter1", aie2::shim_performance_counter1},
      {"shim_performance_counter0_event_value", aie2::shim_performance_counter0_event_value},
      {"shim_performance_counter1_event_value", aie2::shim_performance_counter1_event_value},
      {"shim_event_generate", aie2::shim_event_generate},
      {"shim_event_broadcast_a_0", aie2::shim_event_broadcast_a_0},
      {"shim_event_broadcast_a_1", aie2::shim_event_broadcast_a_1},
      {"shim_event_broadcast_a_2", aie2::shim_event_broadcast_a_2},
      {"shim_event_broadcast_a_3", aie2::shim_event_broadcast_a_3},
      {"shim_event_broadcast_a_4", aie2::shim_event_broadcast_a_4},
      {"shim_event_broadcast_a_5", aie2::shim_event_broadcast_a_5},
      {"shim_event_broadcast_a_6", aie2::shim_event_broadcast_a_6},
      {"shim_event_broadcast_a_7", aie2::shim_event_broadcast_a_7},
      {"shim_event_broadcast_a_8", aie2::shim_event_broadcast_a_8},
      {"shim_event_broadcast_a_9", aie2::shim_event_broadcast_a_9},
      {"shim_event_broadcast_a_10", aie2::shim_event_broadcast_a_10},
      {"shim_event_broadcast_a_11", aie2::shim_event_broadcast_a_11},
      {"shim_event_broadcast_a_12", aie2::shim_event_broadcast_a_12},
      {"shim_event_broadcast_a_13", aie2::shim_event_broadcast_a_13},
      {"shim_event_broadcast_a_14", aie2::shim_event_broadcast_a_14},
      {"shim_event_broadcast_a_15", aie2::shim_event_broadcast_a_15},
      {"shim_event_broadcast_a_block_south_set", aie2::shim_event_broadcast_a_block_south_set},
      {"shim_event_broadcast_a_block_south_clr", aie2::shim_event_broadcast_a_block_south_clr},
      {"shim_event_broadcast_a_block_south_value", aie2::shim_event_broadcast_a_block_south_value},
      {"shim_event_broadcast_a_block_west_set", aie2::shim_event_broadcast_a_block_west_set},
      {"shim_event_broadcast_a_block_west_clr", aie2::shim_event_broadcast_a_block_west_clr},
      {"shim_event_broadcast_a_block_west_value", aie2::shim_event_broadcast_a_block_west_value},
      {"shim_event_broadcast_a_block_north_set", aie2::shim_event_broadcast_a_block_north_set},
      {"shim_event_broadcast_a_block_north_clr", aie2::shim_event_broadcast_a_block_north_clr},
      {"shim_event_broadcast_a_block_north_value", aie2::shim_event_broadcast_a_block_north_value},
      {"shim_event_broadcast_a_block_east_set", aie2::shim_event_broadcast_a_block_east_set},
      {"shim_event_broadcast_a_block_east_clr", aie2::shim_event_broadcast_a_block_east_clr},
      {"shim_event_broadcast_a_block_east_value", aie2::shim_event_broadcast_a_block_east_value},
      {"shim_trace_control0", aie2::shim_trace_control0},
      {"shim_trace_control1", aie2::shim_trace_control1},
      {"shim_trace_status", aie2::shim_trace_status},
      {"shim_trace_event0", aie2::shim_trace_event0},
      {"shim_trace_event1", aie2::shim_trace_event1},
      {"shim_timer_trig_event_low_value", aie2::shim_timer_trig_event_low_value},
      {"shim_timer_trig_event_high_value", aie2::shim_timer_trig_event_high_value},
      {"shim_timer_low", aie2::shim_timer_low},
      {"shim_timer_high", aie2::shim_timer_high},
      {"shim_event_status0", aie2::shim_event_status0},
      {"shim_event_status1", aie2::shim_event_status1},
      {"shim_event_status2", aie2::shim_event_status2},
      {"shim_event_status3", aie2::shim_event_status3},
      {"shim_event_group_dma_enable", aie2::shim_event_group_dma_enable},
      {"shim_stream_switch_event_port_selection_0", aie2::shim_stream_switch_event_port_selection_0},
      {"shim_stream_switch_event_port_selection_1", aie2::shim_stream_switch_event_port_selection_1}
    };
  }

  void populateRegValueToNameMap() {
    regValueToName=  {
      {0x000940a0, "mem_event_broadcast_b_block_west_set"},
      {0x0001d220, "shim_dma_s2mm_status_0"},
      {0x0001d224, "shim_dma_s2mm_status_1"},
      {0x000a061c, "mem_dma_s2mm_3_start_queue"},
      {0x000c0230, "mem_lock35_value"},
      {0x000a0614, "mem_dma_s2mm_2_start_queue"},
      {0x000340f4, "shim_timer_trig_event_high_value"},
      {0x00034068, "shim_event_broadcast_a_block_west_value"},
      {0x0001d10c, "mm_dma_bd8_3"},
      {0x000b0228, "mem_stream_switch_slave_dma_2_slot2"},
      {0x0001d104, "mm_dma_bd8_1"},
      {0x000140e0, "shim_lock14_value"},
      {0x0001d100, "mm_dma_bd8_0"},
      {0x0003f2d4, "cm_stream_switch_slave_west_2_slot1"},
      {0x0003f2d0, "cm_stream_switch_slave_west_2_slot0"},
      {0x0003f2dc, "cm_stream_switch_slave_west_2_slot3"},
      {0x0003f2d8, "cm_stream_switch_slave_west_2_slot2"},
      {0x000c0000, "mem_lock0_value"},
      {0x000c0130, "mem_lock19_value"},
      {0x000a0254, "mem_dma_bd18_5"},
      {0x000a0250, "mem_dma_bd18_4"},
      {0x000a025c, "mem_dma_bd18_7"},
      {0x000a0258, "mem_dma_bd18_6"},
      {0x000a0244, "mem_dma_bd18_1"},
      {0x000a0240, "mem_dma_bd18_0"},
      {0x000a024c, "mem_dma_bd18_3"},
      {0x000a0248, "mem_dma_bd18_2"},
      {0x000c02c0, "mem_lock44_value"},
      {0x0001de0c, "mm_dma_s2mm_1_start_queue"},
      {0x00034508, "cm_event_group_core_stall_enable"},
      {0x00094214, "mem_event_status5"},
      {0x00094200, "mem_event_status0"},
      {0x00094204, "mem_event_status1"},
      {0x00094208, "mem_event_status2"},
      {0x0009420c, "mem_event_status3"},
      {0x0003ff38, "cm_stream_switch_adaptive_clock_gate_abort_period"},
      {0x00094500, "mem_event_group_0_enable"},
      {0x0003f104, "cm_stream_switch_slave_config_dma_0"},
      {0x000c01c0, "mem_lock28_value"},
      {0x000a0624, "mem_dma_s2mm_4_start_queue"},
      {0x00014514, "mm_event_group_error_enable"},
      {0x0001f0b0, "mm_lock11_value"},
      {0x000c00c0, "mem_lock12_value"},
      {0x0001d200, "shim_dma_s2mm_0_ctrl"},
      {0x00014074, "mm_event_broadcast_block_north_clr"},
      {0x0001d084, "mm_dma_bd4_1"},
      {0x000340f0, "cm_timer_trig_event_low_value"},
      {0x00031080, "shim_performance_counter0_event_value"},
      {0x000b02d0, "mem_stream_switch_slave_north_0_slot0"},
      {0x000b02d4, "mem_stream_switch_slave_north_0_slot1"},
      {0x000b02d8, "mem_stream_switch_slave_north_0_slot2"},
      {0x000b02dc, "mem_stream_switch_slave_north_0_slot3"},
      {0x0003f278, "cm_stream_switch_slave_south_2_slot2"},
      {0x0003f27c, "cm_stream_switch_slave_south_2_slot3"},
      {0x0003f270, "cm_stream_switch_slave_south_2_slot0"},
      {0x0003f274, "cm_stream_switch_slave_south_2_slot1"},
      {0x0001f050, "mm_lock5_value"},
      {0x00060010, "cm_module_reset_control"},
      {0x000940f8, "mem_timer_low"},
      {0x0001d1f0, "mm_dma_bd15_4"},
      {0x0001d1f4, "mm_dma_bd15_5"},
      {0x000a063c, "mem_dma_mm2s_1_start_queue"},
      {0x0001d1e0, "mm_dma_bd15_0"},
      {0x0001d1e4, "mm_dma_bd15_1"},
      {0x0001d1e8, "mm_dma_bd15_2"},
      {0x0001d1ec, "mm_dma_bd15_3"},
      {0x00094400, "mem_combo_event_inputs"},
      {0x0003f100, "cm_stream_switch_slave_config_aie_core0"},
      {0x000c0200, "mem_lock32_value"},
      {0x000340d8, "shim_trace_status"},
      {0x00014070, "mm_event_broadcast_block_north_set"},
      {0x0001d160, "mm_dma_bd11_0"},
      {0x0001d164, "mm_dma_bd11_1"},
      {0x0001d168, "mm_dma_bd11_2"},
      {0x0001d16c, "mm_dma_bd11_3"},
      {0x0001d170, "mm_dma_bd11_4"},
      {0x0001d174, "mm_dma_bd11_5"},
      {0x0001d1b0, "mm_dma_bd13_4"},
      {0x0001d1b4, "mm_dma_bd13_5"},
      {0x0001d1a8, "mm_dma_bd13_2"},
      {0x0001d1ac, "mm_dma_bd13_3"},
      {0x0001d1a0, "mm_dma_bd13_0"},
      {0x0001d1a4, "mm_dma_bd13_1"},
      {0x0009422c, "mem_reserved3"},
      {0x00094228, "mem_reserved2"},
      {0x00094224, "mem_reserved1"},
      {0x00094220, "mem_reserved0"},
      {0x000140e0, "mm_trace_event0"},
      {0x000140e4, "mm_trace_event1"},
      {0x000a0230, "mem_dma_bd17_4"},
      {0x000a0234, "mem_dma_bd17_5"},
      {0x000a0228, "mem_dma_bd17_2"},
      {0x000a022c, "mem_dma_bd17_3"},
      {0x000a0220, "mem_dma_bd17_0"},
      {0x000a0224, "mem_dma_bd17_1"},
      {0x000a01e0, "mem_dma_bd15_0"},
      {0x000a01e4, "mem_dma_bd15_1"},
      {0x000a01e8, "mem_dma_bd15_2"},
      {0x000a01ec, "mem_dma_bd15_3"},
      {0x000a01f0, "mem_dma_bd15_4"},
      {0x000a01f4, "mem_dma_bd15_5"},
      {0x000a01f8, "mem_dma_bd15_6"},
      {0x000a01fc, "mem_dma_bd15_7"},
      {0x000a06dc, "mem_dma_s2mm_fot_count_fifo_pop_5"},
      {0x000a06d8, "mem_dma_s2mm_fot_count_fifo_pop_4"},
      {0x000a06cc, "mem_dma_s2mm_fot_count_fifo_pop_1"},
      {0x0001f0d0, "mm_lock13_value"},
      {0x000a06d4, "mem_dma_s2mm_fot_count_fifo_pop_3"},
      {0x000a06d0, "mem_dma_s2mm_fot_count_fifo_pop_2"},
      {0x000b0800, "mem_stream_switch_deterministic_merge_arb0_slave0_1"},
      {0x0003f234, "cm_stream_switch_slave_tile_ctrl_slot1"},
      {0x0003f238, "cm_stream_switch_slave_tile_ctrl_slot2"},
      {0x0003f23c, "cm_stream_switch_slave_tile_ctrl_slot3"},
      {0x0003f14c, "cm_stream_switch_slave_config_east_0"},
      {0x0003f150, "cm_stream_switch_slave_config_east_1"},
      {0x0003f154, "cm_stream_switch_slave_config_east_2"},
      {0x0003f158, "cm_stream_switch_slave_config_east_3"},
      {0x000a01a8, "mem_dma_bd13_2"},
      {0x000a01ac, "mem_dma_bd13_3"},
      {0x000a01a0, "mem_dma_bd13_0"},
      {0x000a01a4, "mem_dma_bd13_1"},
      {0x000a01b8, "mem_dma_bd13_6"},
      {0x000a01bc, "mem_dma_bd13_7"},
      {0x000a01b0, "mem_dma_bd13_4"},
      {0x000a01b4, "mem_dma_bd13_5"},
      {0x000a0170, "mem_dma_bd11_4"},
      {0x000a0174, "mem_dma_bd11_5"},
      {0x000a0178, "mem_dma_bd11_6"},
      {0x000a017c, "mem_dma_bd11_7"},
      {0x000a0160, "mem_dma_bd11_0"},
      {0x000a0164, "mem_dma_bd11_1"},
      {0x000a0168, "mem_dma_bd11_2"},
      {0x000a016c, "mem_dma_bd11_3"},
      {0x000940fc, "mem_timer_high"},
      {0x0003f374, "cm_stream_switch_slave_aie_trace_slot1"},
      {0x0003f370, "cm_stream_switch_slave_aie_trace_slot0"},
      {0x0003f37c, "cm_stream_switch_slave_aie_trace_slot3"},
      {0x0003f378, "cm_stream_switch_slave_aie_trace_slot2"},
      {0x00031524, "cm_performance_counter1"},
      {0x00031520, "cm_performance_counter0"},
      {0x0003152c, "cm_performance_counter3"},
      {0x00031528, "cm_performance_counter2"},
      {0x000b0018, "mem_stream_switch_master_config_tile_ctrl"},
      {0x000940c8, "mem_event_broadcast_b_block_east_value"},
      {0x0003f2bc, "cm_stream_switch_slave_west_0_slot3"},
      {0x0003f2b8, "cm_stream_switch_slave_west_0_slot2"},
      {0x0003f2b4, "cm_stream_switch_slave_west_0_slot1"},
      {0x0003f2b0, "cm_stream_switch_slave_west_0_slot0"},
      {0x000940d4, "mem_trace_control1"},
      {0x000940d0, "mem_trace_control0"},
      {0x0001f030, "mm_lock3_value"},
      {0x00014028, "mm_event_broadcast6"},
      {0x0001402c, "mm_event_broadcast7"},
      {0x00034000, "cm_timer_control"},
      {0x00014018, "mm_event_broadcast2"},
      {0x00014040, "shim_lock4_value"},
      {0x0001401c, "mm_event_broadcast3"},
      {0x0001d0cc, "mm_dma_bd6_3"},
      {0x00014014, "mm_event_broadcast1"},
      {0x00014090, "shim_lock9_value"},
      {0x0003ff30, "cm_tile_control_packet_handler_status"},
      {0x00034030, "cm_event_broadcast8"},
      {0x00034034, "cm_event_broadcast9"},
      {0x0003f368, "cm_stream_switch_slave_east_3_slot2"},
      {0x00034024, "cm_event_broadcast5"},
      {0x0003f360, "cm_stream_switch_slave_east_3_slot0"},
      {0x0003f364, "cm_stream_switch_slave_east_3_slot1"},
      {0x00034010, "cm_event_broadcast0"},
      {0x00034014, "cm_event_broadcast1"},
      {0x00034018, "cm_event_broadcast2"},
      {0x0003401c, "cm_event_broadcast3"},
      {0x00091084, "mem_performance_counter1_event_value"},
      {0x000b0268, "mem_stream_switch_slave_tile_ctrl_slot2"},
      {0x000b026c, "mem_stream_switch_slave_tile_ctrl_slot3"},
      {0x0001d158, "shim_dma_bd10_6"},
      {0x0003f02c, "cm_stream_switch_master_config_west2"},
      {0x0003f030, "cm_stream_switch_master_config_west3"},
      {0x0003f024, "cm_stream_switch_master_config_west0"},
      {0x0003f028, "cm_stream_switch_master_config_west1"},
      {0x0003f144, "cm_stream_switch_slave_config_north_2"},
      {0x000a05c0, "mem_dma_bd46_0"},
      {0x000a05c4, "mem_dma_bd46_1"},
      {0x000a05c8, "mem_dma_bd46_2"},
      {0x000a05cc, "mem_dma_bd46_3"},
      {0x000a05d0, "mem_dma_bd46_4"},
      {0x000a05d4, "mem_dma_bd46_5"},
      {0x000a05d8, "mem_dma_bd46_6"},
      {0x000a05dc, "mem_dma_bd46_7"},
      {0x00014100, "mm_watchpoint0"},
      {0x00014104, "mm_watchpoint1"},
      {0x00094068, "mem_event_broadcast_a_block_west_value"},
      {0x0001de18, "mm_dma_mm2s_1_ctrl"},
      {0x000140f0, "mm_timer_trig_event_low_value"},
      {0x000b0808, "mem_stream_switch_deterministic_merge_arb0_ctrl"},
      {0x0003f28c, "cm_stream_switch_slave_south_3_slot3"},
      {0x0003f288, "cm_stream_switch_slave_south_3_slot2"},
      {0x000b0030, "mem_stream_switch_master_config_north1"},
      {0x000b002c, "mem_stream_switch_master_config_north0"},
      {0x000b0040, "mem_stream_switch_master_config_north5"},
      {0x000b003c, "mem_stream_switch_master_config_north4"},
      {0x000140d4, "mm_trace_control1"},
      {0x000140d0, "mm_trace_control0"},
      {0x00034078, "shim_event_broadcast_a_block_north_value"},
      {0x00034054, "shim_event_broadcast_a_block_south_clr"},
      {0x00094074, "mem_event_broadcast_a_block_north_clr"},
      {0x00014064, "mm_event_broadcast_block_west_clr"},
      {0x000c02e0, "mem_lock46_value"},
      {0x0001d114, "mm_dma_bd8_5"},
      {0x0001d110, "mm_dma_bd8_4"},
      {0x000b022c, "mem_stream_switch_slave_dma_2_slot3"},
      {0x0001d108, "mm_dma_bd8_2"},
      {0x000b0224, "mem_stream_switch_slave_dma_2_slot1"},
      {0x000b0220, "mem_stream_switch_slave_dma_2_slot0"},
      {0x0001de00, "mm_dma_s2mm_0_ctrl"},
      {0x0003158c, "cm_performance_counter3_event_value"},
      {0x00091088, "mem_performance_counter2_event_value"},
      {0x0003f108, "cm_stream_switch_slave_config_dma_1"},
      {0x000a052c, "mem_dma_bd41_3"},
      {0x000a0528, "mem_dma_bd41_2"},
      {0x000a0524, "mem_dma_bd41_1"},
      {0x000a0520, "mem_dma_bd41_0"},
      {0x000a053c, "mem_dma_bd41_7"},
      {0x000a0538, "mem_dma_bd41_6"},
      {0x000a0534, "mem_dma_bd41_5"},
      {0x000a0530, "mem_dma_bd41_4"},
      {0x000a0574, "mem_dma_bd43_5"},
      {0x000a0570, "mem_dma_bd43_4"},
      {0x000a057c, "mem_dma_bd43_7"},
      {0x000a0578, "mem_dma_bd43_6"},
      {0x000a0564, "mem_dma_bd43_1"},
      {0x000a0560, "mem_dma_bd43_0"},
      {0x000a056c, "mem_dma_bd43_3"},
      {0x000a0568, "mem_dma_bd43_2"},
      {0x000c0350, "mem_lock53_value"},
      {0x000140b0, "shim_lock11_value"},
      {0x000b0244, "mem_stream_switch_slave_dma_4_slot1"},
      {0x000b0240, "mem_stream_switch_slave_dma_4_slot0"},
      {0x000b024c, "mem_stream_switch_slave_dma_4_slot3"},
      {0x000b0248, "mem_stream_switch_slave_dma_4_slot2"},
      {0x000a05bc, "mem_dma_bd45_7"},
      {0x000a05b8, "mem_dma_bd45_6"},
      {0x000a05b4, "mem_dma_bd45_5"},
      {0x000a05b0, "mem_dma_bd45_4"},
      {0x000a05ac, "mem_dma_bd45_3"},
      {0x000a05a8, "mem_dma_bd45_2"},
      {0x000a05a4, "mem_dma_bd45_1"},
      {0x000a05a0, "mem_dma_bd45_0"},
      {0x00014050, "shim_lock5_value"},
      {0x00092110, "mem_ecc_scrubbing_event"},
      {0x00034084, "shim_event_broadcast_a_block_east_clr"},
      {0x0003f290, "cm_stream_switch_slave_south_4_slot0"},
      {0x0003f294, "cm_stream_switch_slave_south_4_slot1"},
      {0x0003f298, "cm_stream_switch_slave_south_4_slot2"},
      {0x0003f29c, "cm_stream_switch_slave_south_4_slot3"},
      {0x00094520, "mem_event_group_user_event_enable"},
      {0x00014000, "shim_lock0_value"},
      {0x00092000, "mem_checkbit_error_generation"},
      {0x00011000, "mm_performance_control0"},
      {0x00011008, "mm_performance_control1"},
      {0x0003f134, "cm_stream_switch_slave_config_west_2"},
      {0x0003f138, "cm_stream_switch_slave_config_west_3"},
      {0x0003f12c, "cm_stream_switch_slave_config_west_0"},
      {0x0003f130, "cm_stream_switch_slave_config_west_1"},
      {0x00014518, "mm_event_group_broadcast_enable"},
      {0x00034070, "shim_event_broadcast_a_block_north_set"},
      {0x00094210, "mem_event_status4"},
      {0x0001d1c4, "shim_dma_bd14_1"},
      {0x000a010c, "mem_dma_bd8_3"},
      {0x000a0108, "mem_dma_bd8_2"},
      {0x000a0104, "mem_dma_bd8_1"},
      {0x000a0100, "mem_dma_bd8_0"},
      {0x000a011c, "mem_dma_bd8_7"},
      {0x000a0118, "mem_dma_bd8_6"},
      {0x000a0114, "mem_dma_bd8_5"},
      {0x000a0110, "mem_dma_bd8_4"},
      {0x000a00c4, "mem_dma_bd6_1"},
      {0x000a00c0, "mem_dma_bd6_0"},
      {0x000a00cc, "mem_dma_bd6_3"},
      {0x000a00c8, "mem_dma_bd6_2"},
      {0x000a00d4, "mem_dma_bd6_5"},
      {0x000a00d0, "mem_dma_bd6_4"},
      {0x000a00dc, "mem_dma_bd6_7"},
      {0x000a00d8, "mem_dma_bd6_6"},
      {0x000940b0, "mem_event_broadcast_b_block_north_set"},
      {0x00096040, "mem_cssd_trigger"},
      {0x0001d0f0, "mm_dma_bd7_4"},
      {0x0001d0f4, "mm_dma_bd7_5"},
      {0x0001d0e0, "mm_dma_bd7_0"},
      {0x0001d0e4, "mm_dma_bd7_1"},
      {0x0001d0e8, "mm_dma_bd7_2"},
      {0x0001d0ec, "mm_dma_bd7_3"},
      {0x000b02f8, "mem_stream_switch_slave_north_2_slot2"},
      {0x000b02fc, "mem_stream_switch_slave_north_2_slot3"},
      {0x000b02f0, "mem_stream_switch_slave_north_2_slot0"},
      {0x000b02f4, "mem_stream_switch_slave_north_2_slot1"},
      {0x0001d0a8, "mm_dma_bd5_2"},
      {0x0001d0ac, "mm_dma_bd5_3"},
      {0x0001d0a0, "mm_dma_bd5_0"},
      {0x0001d0a4, "mm_dma_bd5_1"},
      {0x0001d0b0, "mm_dma_bd5_4"},
      {0x0001d0b4, "mm_dma_bd5_5"},
      {0x00094518, "mem_event_group_error_enable"},
      {0x0001f020, "mm_lock2_value"},
      {0x00034060, "shim_event_broadcast_a_block_west_set"},
      {0x000a0028, "mem_dma_bd1_2"},
      {0x000a002c, "mem_dma_bd1_3"},
      {0x000a0020, "mem_dma_bd1_0"},
      {0x000a0024, "mem_dma_bd1_1"},
      {0x000a0038, "mem_dma_bd1_6"},
      {0x000a003c, "mem_dma_bd1_7"},
      {0x000a0030, "mem_dma_bd1_4"},
      {0x000a0034, "mem_dma_bd1_5"},
      {0x000340d0, "shim_trace_control0"},
      {0x000340d4, "shim_trace_control1"},
      {0x000b0f00, "mem_stream_switch_event_port_selection_0"},
      {0x000b0f04, "mem_stream_switch_event_port_selection_1"},
      {0x0001d068, "mm_dma_bd3_2"},
      {0x0001d06c, "mm_dma_bd3_3"},
      {0x0001d070, "mm_dma_bd3_4"},
      {0x0001d074, "mm_dma_bd3_5"},
      {0x00034074, "cm_event_broadcast_block_north_clr"},
      {0x0001d030, "mm_dma_bd1_4"},
      {0x0001d034, "mm_dma_bd1_5"},
      {0x0001d028, "mm_dma_bd1_2"},
      {0x0001d02c, "mm_dma_bd1_3"},
      {0x00032008, "cm_enable_events"},
      {0x0001d024, "mm_dma_bd1_1"},
      {0x000c0250, "mem_lock37_value"},
      {0x000b0110, "mem_stream_switch_slave_config_dma_4"},
      {0x000b0114, "mem_stream_switch_slave_config_dma_5"},
      {0x000b0100, "mem_stream_switch_slave_config_dma_0"},
      {0x000b0104, "mem_stream_switch_slave_config_dma_1"},
      {0x000b0108, "mem_stream_switch_slave_config_dma_2"},
      {0x00094508, "mem_event_group_dma_enable"},
      {0x00094028, "mem_event_broadcast6"},
      {0x0009402c, "mem_event_broadcast7"},
      {0x00094020, "mem_event_broadcast4"},
      {0x00094024, "mem_event_broadcast5"},
      {0x00094018, "mem_event_broadcast2"},
      {0x0009401c, "mem_event_broadcast3"},
      {0x00094010, "mem_event_broadcast0"},
      {0x00094014, "mem_event_broadcast1"},
      {0x000b0020, "mem_stream_switch_master_config_south1"},
      {0x000b001c, "mem_stream_switch_master_config_south0"},
      {0x000b0028, "mem_stream_switch_master_config_south3"},
      {0x000b0024, "mem_stream_switch_master_config_south2"},
      {0x000a0060, "mem_dma_bd3_0"},
      {0x000a0064, "mem_dma_bd3_1"},
      {0x000a0068, "mem_dma_bd3_2"},
      {0x000a006c, "mem_dma_bd3_3"},
      {0x00032004, "cm_core_status"},
      {0x0001f120, "mm_locks_overflow"},
      {0x000c0390, "mem_lock57_value"},
      {0x00034080, "cm_event_broadcast_block_east_set"},
      {0x0001df14, "mm_dma_mm2s_status_1"},
      {0x0001df10, "mm_dma_mm2s_status_0"},
      {0x000c0020, "mem_lock2_value"},
      {0x0003f240, "cm_stream_switch_slave_fifo_0_slot0"},
      {0x0001f080, "mm_lock8_value"},
      {0x00034518, "cm_event_group_stream_switch_enable"},
      {0x0001f000, "mm_lock0_value"},
      {0x000a0404, "mem_dma_bd32_1"},
      {0x000a0400, "mem_dma_bd32_0"},
      {0x000140d8, "mm_trace_status"},
      {0x000a0408, "mem_dma_bd32_2"},
      {0x000a0414, "mem_dma_bd32_5"},
      {0x000a0410, "mem_dma_bd32_4"},
      {0x000a041c, "mem_dma_bd32_7"},
      {0x000a0418, "mem_dma_bd32_6"},
      {0x000c0040, "mem_lock4_value"},
      {0x0003f204, "cm_stream_switch_slave_aie_core0_slot1"},
      {0x0003f200, "cm_stream_switch_slave_aie_core0_slot0"},
      {0x0003f20c, "cm_stream_switch_slave_aie_core0_slot3"},
      {0x0003f208, "cm_stream_switch_slave_aie_core0_slot2"},
      {0x0001451c, "mm_event_group_user_event_enable"},
      {0x00014000, "mm_timer_control"},
      {0x000a04dc, "mem_dma_bd38_7"},
      {0x000a04d8, "mem_dma_bd38_6"},
      {0x000a04d4, "mem_dma_bd38_5"},
      {0x000a04d0, "mem_dma_bd38_4"},
      {0x000a04cc, "mem_dma_bd38_3"},
      {0x000a04c8, "mem_dma_bd38_2"},
      {0x000a04c4, "mem_dma_bd38_1"},
      {0x000a04c0, "mem_dma_bd38_0"},
      {0x0003f2e8, "cm_stream_switch_slave_west_3_slot2"},
      {0x0003f2ec, "cm_stream_switch_slave_west_3_slot3"},
      {0x00014404, "mm_combo_event_control"},
      {0x0003f314, "cm_stream_switch_slave_north_2_slot1"},
      {0x0003f318, "cm_stream_switch_slave_north_2_slot2"},
      {0x0003f31c, "cm_stream_switch_slave_north_2_slot3"},
      {0x00031020, "shim_performance_counter0"},
      {0x00031024, "shim_performance_counter1"},
      {0x00014508, "mm_event_group_dma_enable"},
      {0x00014030, "shim_lock3_value"},
      {0x000340f4, "cm_timer_trig_event_high_value"},
      {0x000c0190, "mem_lock25_value"},
      {0x000d0000, "mem_lock_request"},
      {0x0001f128, "mm_locks_underflow"},
      {0x00034088, "cm_event_broadcast_block_east_value"},
      {0x00011084, "mm_performance_counter1_event_value"},
      {0x00034058, "cm_event_broadcast_block_south_value"},
      {0x0003f340, "cm_stream_switch_slave_east_1_slot0"},
      {0x0003f344, "cm_stream_switch_slave_east_1_slot1"},
      {0x0003f348, "cm_stream_switch_slave_east_1_slot2"},
      {0x0003f34c, "cm_stream_switch_slave_east_1_slot3"},
      {0x000c02a0, "mem_lock42_value"},
      {0x000a0468, "mem_dma_bd35_2"},
      {0x000a046c, "mem_dma_bd35_3"},
      {0x000a0460, "mem_dma_bd35_0"},
      {0x000140f4, "mm_timer_trig_event_high_value"},
      {0x000a0478, "mem_dma_bd35_6"},
      {0x0003ff20, "cm_stream_switch_parity_injection"},
      {0x000a0470, "mem_dma_bd35_4"},
      {0x000a0474, "mem_dma_bd35_5"},
      {0x000a04b0, "mem_dma_bd37_4"},
      {0x000a04b4, "mem_dma_bd37_5"},
      {0x000a04b8, "mem_dma_bd37_6"},
      {0x000a04bc, "mem_dma_bd37_7"},
      {0x000a04a0, "mem_dma_bd37_0"},
      {0x000a04a4, "mem_dma_bd37_1"},
      {0x000a04a8, "mem_dma_bd37_2"},
      {0x000a04ac, "mem_dma_bd37_3"},
      {0x000a03f8, "mem_dma_bd31_6"},
      {0x000a03fc, "mem_dma_bd31_7"},
      {0x000a03f0, "mem_dma_bd31_4"},
      {0x000a03f4, "mem_dma_bd31_5"},
      {0x000a03e8, "mem_dma_bd31_2"},
      {0x000a03ec, "mem_dma_bd31_3"},
      {0x000a03e0, "mem_dma_bd31_0"},
      {0x000a03e4, "mem_dma_bd31_1"},
      {0x000a0608, "mem_dma_s2mm_1_ctrl"},
      {0x0003f038, "cm_stream_switch_master_config_north1"},
      {0x0003f034, "cm_stream_switch_master_config_north0"},
      {0x0003f040, "cm_stream_switch_master_config_north3"},
      {0x0003f03c, "cm_stream_switch_master_config_north2"},
      {0x0003f048, "cm_stream_switch_master_config_north5"},
      {0x0003f044, "cm_stream_switch_master_config_north4"},
      {0x0001d044, "shim_dma_bd2_1"},
      {0x000fff00, "mem_module_clock_control"},
      {0x0001de10, "mm_dma_mm2s_0_ctrl"},
      {0x000a0674, "mem_dma_s2mm_status_5"},
      {0x000a0670, "mem_dma_s2mm_status_4"},
      {0x000a066c, "mem_dma_s2mm_status_3"},
      {0x000a0668, "mem_dma_s2mm_status_2"},
      {0x000a0664, "mem_dma_s2mm_status_1"},
      {0x000a0660, "mem_dma_s2mm_status_0"},
      {0x00014060, "mm_event_broadcast_block_west_set"},
      {0x00060000, "cm_module_clock_control"},
      {0x000b0f10, "mem_stream_switch_parity_status"},
      {0x00094408, "mem_edge_detection_event_control"},
      {0x00094034, "mem_event_broadcast9"},
      {0x0003f128, "cm_stream_switch_slave_config_south_5"},
      {0x0003f380, "cm_stream_switch_slave_mem_trace_slot0"},
      {0x0003f2a4, "cm_stream_switch_slave_south_5_slot1"},
      {0x0003f2a0, "cm_stream_switch_slave_south_5_slot0"},
      {0x0003f2ac, "cm_stream_switch_slave_south_5_slot3"},
      {0x0003f2a8, "cm_stream_switch_slave_south_5_slot2"},
      {0x000340d4, "cm_trace_control1"},
      {0x000340d0, "cm_trace_control0"},
      {0x0003f118, "cm_stream_switch_slave_config_south_1"},
      {0x000c0310, "mem_lock49_value"},
      {0x000c02f0, "mem_lock47_value"},
      {0x0001d188, "shim_dma_bd12_2"},
      {0x0001d18c, "shim_dma_bd12_3"},
      {0x0001d180, "shim_dma_bd12_0"},
      {0x0001d184, "shim_dma_bd12_1"},
      {0x0001d198, "shim_dma_bd12_6"},
      {0x0001d19c, "shim_dma_bd12_7"},
      {0x0001d190, "shim_dma_bd12_4"},
      {0x0001d194, "shim_dma_bd12_5"},
      {0x0001d150, "shim_dma_bd10_4"},
      {0x0001d154, "shim_dma_bd10_5"},
      {0x000b0260, "mem_stream_switch_slave_tile_ctrl_slot0"},
      {0x000b0264, "mem_stream_switch_slave_tile_ctrl_slot1"},
      {0x0001d140, "shim_dma_bd10_0"},
      {0x0001d144, "shim_dma_bd10_1"},
      {0x0001d148, "shim_dma_bd10_2"},
      {0x0001d14c, "shim_dma_bd10_3"},
      {0x000c00d0, "mem_lock13_value"},
      {0x0003f2cc, "cm_stream_switch_slave_west_1_slot3"},
      {0x000b025c, "mem_stream_switch_slave_dma_5_slot3"},
      {0x00014088, "mm_event_broadcast_block_east_value"},
      {0x00014054, "mm_event_broadcast_block_south_clr"},
      {0x000a0300, "mem_dma_bd24_0"},
      {0x000a0304, "mem_dma_bd24_1"},
      {0x000a0308, "mem_dma_bd24_2"},
      {0x000a030c, "mem_dma_bd24_3"},
      {0x000940a4, "mem_event_broadcast_b_block_west_clr"},
      {0x000b0f20, "mem_stream_switch_parity_injection"},
      {0x000a0610, "mem_dma_s2mm_2_ctrl"},
      {0x0001d064, "mm_dma_bd3_1"},
      {0x000c01e0, "mem_lock30_value"},
      {0x00034504, "cm_event_group_pc_enable"},
      {0x000a02d8, "mem_dma_bd22_6"},
      {0x0001d0cc, "shim_dma_bd6_3"},
      {0x000a02d0, "mem_dma_bd22_4"},
      {0x0001d0c4, "shim_dma_bd6_1"},
      {0x000c03c0, "mem_lock60_value"},
      {0x0001d0d8, "shim_dma_bd6_6"},
      {0x0001df04, "mm_dma_s2mm_status_1"},
      {0x0001df00, "mm_dma_s2mm_status_0"},
      {0x0001d0dc, "shim_dma_bd6_7"},
      {0x0001d0d0, "shim_dma_bd6_4"},
      {0x0001d0d4, "shim_dma_bd6_5"},
      {0x000c00a0, "mem_lock10_value"},
      {0x000a062c, "mem_dma_s2mm_5_start_queue"},
      {0x0001d020, "mm_dma_bd1_0"},
      {0x000a0380, "mem_dma_bd28_0"},
      {0x000a0384, "mem_dma_bd28_1"},
      {0x000a0388, "mem_dma_bd28_2"},
      {0x000a038c, "mem_dma_bd28_3"},
      {0x000a0390, "mem_dma_bd28_4"},
      {0x000a0394, "mem_dma_bd28_5"},
      {0x000a0398, "mem_dma_bd28_6"},
      {0x000a039c, "mem_dma_bd28_7"},
      {0x00034514, "cm_event_group_errors1_enable"},
      {0x000940e0, "mem_trace_event0"},
      {0x000940e4, "mem_trace_event1"},
      {0x000a0288, "mem_dma_bd20_2"},
      {0x000a028c, "mem_dma_bd20_3"},
      {0x000a0290, "mem_dma_bd20_4"},
      {0x000a0294, "mem_dma_bd20_5"},
      {0x000a0298, "mem_dma_bd20_6"},
      {0x000a029c, "mem_dma_bd20_7"},
      {0x000c03a0, "mem_lock58_value"},
      {0x00094084, "mem_event_broadcast_a_block_east_clr"},
      {0x000a060c, "mem_dma_s2mm_1_start_queue"},
      {0x0001d1e4, "shim_dma_bd15_1"},
      {0x0001d1e0, "shim_dma_bd15_0"},
      {0x0001d1ec, "shim_dma_bd15_3"},
      {0x0001d1e8, "shim_dma_bd15_2"},
      {0x0001d1f4, "shim_dma_bd15_5"},
      {0x0001d1f0, "shim_dma_bd15_4"},
      {0x00014084, "mm_event_broadcast_block_east_clr"},
      {0x0001d1f8, "shim_dma_bd15_6"},
      {0x0001d170, "shim_dma_bd11_4"},
      {0x000940a8, "mem_event_broadcast_b_block_west_value"},
      {0x00094030, "mem_event_broadcast8"},
      {0x000b010c, "mem_stream_switch_slave_config_dma_3"},
      {0x00014510, "mm_event_group_memory_conflict_enable"},
      {0x000c0260, "mem_lock38_value"},
      {0x00034034, "shim_event_broadcast_a_9"},
      {0x00034030, "shim_event_broadcast_a_8"},
      {0x0003402c, "shim_event_broadcast_a_7"},
      {0x00034028, "shim_event_broadcast_a_6"},
      {0x00034024, "shim_event_broadcast_a_5"},
      {0x00034020, "shim_event_broadcast_a_4"},
      {0x0003401c, "shim_event_broadcast_a_3"},
      {0x00034018, "shim_event_broadcast_a_2"},
      {0x00034014, "shim_event_broadcast_a_1"},
      {0x00034010, "shim_event_broadcast_a_0"},
      {0x000a0620, "mem_dma_s2mm_4_ctrl"},
      {0x0001d0b4, "shim_dma_bd5_5"},
      {0x0001d0b0, "shim_dma_bd5_4"},
      {0x0001d0bc, "shim_dma_bd5_7"},
      {0x0001d0b8, "shim_dma_bd5_6"},
      {0x0001d0a4, "shim_dma_bd5_1"},
      {0x0001d0a0, "shim_dma_bd5_0"},
      {0x0001d0ac, "shim_dma_bd5_3"},
      {0x0001d0a8, "shim_dma_bd5_2"},
      {0x0003200c, "cm_reset_event"},
      {0x000140f0, "shim_lock15_value"},
      {0x0001d07c, "shim_dma_bd3_7"},
      {0x0001d078, "shim_dma_bd3_6"},
      {0x0001d074, "shim_dma_bd3_5"},
      {0x0001d070, "shim_dma_bd3_4"},
      {0x0001d06c, "shim_dma_bd3_3"},
      {0x0001d068, "shim_dma_bd3_2"},
      {0x0001d064, "shim_dma_bd3_1"},
      {0x0001d060, "shim_dma_bd3_0"},
      {0x000c0180, "mem_lock24_value"},
      {0x000a0070, "mem_dma_bd3_4"},
      {0x000a0074, "mem_dma_bd3_5"},
      {0x000a0078, "mem_dma_bd3_6"},
      {0x000a007c, "mem_dma_bd3_7"},
      {0x00012124, "mm_parity_failing_address"},
      {0x000b0000, "mem_stream_switch_master_config_dma0"},
      {0x000b0004, "mem_stream_switch_master_config_dma1"},
      {0x000b0008, "mem_stream_switch_master_config_dma2"},
      {0x000b000c, "mem_stream_switch_master_config_dma3"},
      {0x000b0010, "mem_stream_switch_master_config_dma4"},
      {0x000b0014, "mem_stream_switch_master_config_dma5"},
      {0x000a036c, "mem_dma_bd27_3"},
      {0x000a0368, "mem_dma_bd27_2"},
      {0x000a0364, "mem_dma_bd27_1"},
      {0x000a0360, "mem_dma_bd27_0"},
      {0x000a037c, "mem_dma_bd27_7"},
      {0x000a0378, "mem_dma_bd27_6"},
      {0x000a0374, "mem_dma_bd27_5"},
      {0x000a0370, "mem_dma_bd27_4"},
      {0x000a0334, "mem_dma_bd25_5"},
      {0x000a0330, "mem_dma_bd25_4"},
      {0x000a033c, "mem_dma_bd25_7"},
      {0x000a0338, "mem_dma_bd25_6"},
      {0x000a0324, "mem_dma_bd25_1"},
      {0x000a0320, "mem_dma_bd25_0"},
      {0x000a032c, "mem_dma_bd25_3"},
      {0x000a0328, "mem_dma_bd25_2"},
      {0x0001d0ec, "shim_dma_bd7_3"},
      {0x0001d0e8, "shim_dma_bd7_2"},
      {0x000a02f4, "mem_dma_bd23_5"},
      {0x0001d0e0, "shim_dma_bd7_0"},
      {0x0001d0fc, "shim_dma_bd7_7"},
      {0x0001d0f8, "shim_dma_bd7_6"},
      {0x0001d0f4, "shim_dma_bd7_5"},
      {0x0001d0f0, "shim_dma_bd7_4"},
      {0x000b0204, "mem_stream_switch_slave_dma_0_slot1"},
      {0x000b0200, "mem_stream_switch_slave_dma_0_slot0"},
      {0x000b020c, "mem_stream_switch_slave_dma_0_slot3"},
      {0x000b0208, "mem_stream_switch_slave_dma_0_slot2"},
      {0x000a0638, "mem_dma_mm2s_1_ctrl"},
      {0x00034404, "cm_combo_event_control"},
      {0x000b0144, "mem_stream_switch_slave_config_trace"},
      {0x000a0280, "mem_dma_bd20_0"},
      {0x000a0284, "mem_dma_bd20_1"},
      {0x00032018, "cm_debug_control2"},
      {0x00032014, "cm_debug_control1"},
      {0x00032010, "cm_debug_control0"},
      {0x0001d100, "shim_dma_bd8_0"},
      {0x00014010, "mm_event_broadcast0"},
      {0x0001d108, "shim_dma_bd8_2"},
      {0x0001d10c, "shim_dma_bd8_3"},
      {0x0001d110, "shim_dma_bd8_4"},
      {0x0001d114, "shim_dma_bd8_5"},
      {0x0001d118, "shim_dma_bd8_6"},
      {0x0001d11c, "shim_dma_bd8_7"},
      {0x000c02b0, "mem_lock43_value"},
      {0x000340f8, "shim_timer_low"},
      {0x0001de08, "mm_dma_s2mm_1_ctrl"},
      {0x0009451c, "mem_event_group_broadcast_enable"},
      {0x0003f308, "cm_stream_switch_slave_north_1_slot2"},
      {0x0003f304, "cm_stream_switch_slave_north_1_slot1"},
      {0x0003f300, "cm_stream_switch_slave_north_1_slot0"},
      {0x0001d000, "shim_dma_bd0_0"},
      {0x0001d004, "shim_dma_bd0_1"},
      {0x0001d008, "shim_dma_bd0_2"},
      {0x0001d00c, "shim_dma_bd0_3"},
      {0x0001d010, "shim_dma_bd0_4"},
      {0x0001d014, "shim_dma_bd0_5"},
      {0x0001d018, "shim_dma_bd0_6"},
      {0x0001d01c, "shim_dma_bd0_7"},
      {0x000940b4, "mem_event_broadcast_b_block_north_clr"},
      {0x000b0210, "mem_stream_switch_slave_dma_1_slot0"},
      {0x000b0214, "mem_stream_switch_slave_dma_1_slot1"},
      {0x000b0218, "mem_stream_switch_slave_dma_1_slot2"},
      {0x000b021c, "mem_stream_switch_slave_dma_1_slot3"},
      {0x00034408, "cm_edge_detection_event_control"},
      {0x000c0070, "mem_lock7_value"},
      {0x0003201c, "cm_debug_status"},
      {0x000b030c, "mem_stream_switch_slave_north_3_slot3"},
      {0x000b0308, "mem_stream_switch_slave_north_3_slot2"},
      {0x000b0304, "mem_stream_switch_slave_north_3_slot1"},
      {0x000b0300, "mem_stream_switch_slave_north_3_slot0"},
      {0x0003202c, "cm_pc_event3"},
      {0x00032028, "cm_pc_event2"},
      {0x00032024, "cm_pc_event1"},
      {0x00032020, "cm_pc_event0"},
      {0x0003f000, "cm_stream_switch_master_config_aie_core0"},
      {0x000a0238, "mem_dma_bd17_6"},
      {0x000a023c, "mem_dma_bd17_7"},
      {0x0001f118, "mm_locks_event_selection_6"},
      {0x0001f11c, "mm_locks_event_selection_7"},
      {0x000a06c4, "mem_dma_s2mm_current_write_count_5"},
      {0x0001f114, "mm_locks_event_selection_5"},
      {0x0001f108, "mm_locks_event_selection_2"},
      {0x000a06b8, "mem_dma_s2mm_current_write_count_2"},
      {0x0001f100, "mm_locks_event_selection_0"},
      {0x0001f104, "mm_locks_event_selection_1"},
      {0x000a02f8, "mem_dma_bd23_6"},
      {0x00040000, "mm_lock_request"},
      {0x000940f0, "mem_timer_trig_event_low_value"},
      {0x0009404c, "mem_event_broadcast15"},
      {0x00094038, "mem_event_broadcast10"},
      {0x0009403c, "mem_event_broadcast11"},
      {0x00094040, "mem_event_broadcast12"},
      {0x00094044, "mem_event_broadcast13"},
      {0x0003f324, "cm_stream_switch_slave_north_3_slot1"},
      {0x0003f320, "cm_stream_switch_slave_north_3_slot0"},
      {0x0003f32c, "cm_stream_switch_slave_north_3_slot3"},
      {0x0003f328, "cm_stream_switch_slave_north_3_slot2"},
      {0x0003ff10, "cm_stream_switch_parity_status"},
      {0x00036030, "cm_tile_control"},
      {0x00031580, "cm_performance_counter0_event_value"},
      {0x000b02a4, "mem_stream_switch_slave_south_3_slot1"},
      {0x000b02a0, "mem_stream_switch_slave_south_3_slot0"},
      {0x000b02ac, "mem_stream_switch_slave_south_3_slot3"},
      {0x000b02a8, "mem_stream_switch_slave_south_3_slot2"},
      {0x000c0160, "mem_lock22_value"},
      {0x000340f8, "cm_timer_low"},
      {0x00094080, "mem_event_broadcast_a_block_east_set"},
      {0x00014504, "mm_event_group_watchpoint_enable"},
      {0x000c0080, "mem_lock8_value"},
      {0x000c0210, "mem_lock33_value"},
      {0x000b012c, "mem_stream_switch_slave_config_south_4"},
      {0x000a064c, "mem_dma_mm2s_3_start_queue"},
      {0x00094098, "mem_event_broadcast_b_block_south_value"},
      {0x000140d0, "shim_lock13_value"},
      {0x000940b8, "mem_event_broadcast_b_block_north_value"},
      {0x0003f04c, "cm_stream_switch_master_config_east0"},
      {0x0003f050, "cm_stream_switch_master_config_east1"},
      {0x0003f054, "cm_stream_switch_master_config_east2"},
      {0x0003f058, "cm_stream_switch_master_config_east3"},
      {0x0001d1fc, "shim_dma_bd15_7"},
      {0x000a06c8, "mem_dma_s2mm_fot_count_fifo_pop_0"},
      {0x0003f230, "cm_stream_switch_slave_tile_ctrl_slot0"},
      {0x000c0320, "mem_lock50_value"},
      {0x00034504, "shim_event_group_dma_enable"},
      {0x0003450c, "cm_event_group_core_program_flow_enable"},
      {0x00014050, "mm_event_broadcast_block_south_set"},
      {0x0001d1d4, "mm_dma_bd14_5"},
      {0x0001d1d0, "mm_dma_bd14_4"},
      {0x0001d1c4, "mm_dma_bd14_1"},
      {0x0001d1c0, "mm_dma_bd14_0"},
      {0x0001d1cc, "mm_dma_bd14_3"},
      {0x0001d1c8, "mm_dma_bd14_2"},
      {0x00034044, "cm_event_broadcast13"},
      {0x00016000, "mm_spare_reg"},
      {0x0003f148, "cm_stream_switch_slave_config_north_3"},
      {0x0003f00c, "cm_stream_switch_master_config_tile_ctrl"},
      {0x0003f140, "cm_stream_switch_slave_config_north_1"},
      {0x0003f13c, "cm_stream_switch_slave_config_north_0"},
      {0x00094504, "mem_event_group_watchpoint_enable"},
      {0x00096030, "mem_tile_control"},
      {0x0001d144, "mm_dma_bd10_1"},
      {0x0001d140, "mm_dma_bd10_0"},
      {0x0001d14c, "mm_dma_bd10_3"},
      {0x0001d148, "mm_dma_bd10_2"},
      {0x0001d154, "mm_dma_bd10_5"},
      {0x0001d150, "mm_dma_bd10_4"},
      {0x0001d194, "mm_dma_bd12_5"},
      {0x0001d190, "mm_dma_bd12_4"},
      {0x0001d18c, "mm_dma_bd12_3"},
      {0x0001d188, "mm_dma_bd12_2"},
      {0x0001d184, "mm_dma_bd12_1"},
      {0x0001d180, "mm_dma_bd12_0"},
      {0x000a01c4, "mem_dma_bd14_1"},
      {0x000a01c0, "mem_dma_bd14_0"},
      {0x000a01cc, "mem_dma_bd14_3"},
      {0x000a01c8, "mem_dma_bd14_2"},
      {0x000a01d4, "mem_dma_bd14_5"},
      {0x000a01d0, "mem_dma_bd14_4"},
      {0x000a01dc, "mem_dma_bd14_7"},
      {0x000a01d8, "mem_dma_bd14_6"},
      {0x00034058, "shim_event_broadcast_a_block_south_value"},
      {0x00032030, "cm_error_halt_control"},
      {0x00094070, "mem_event_broadcast_a_block_north_set"},
      {0x00096000, "mem_spare_reg"},
      {0x000a018c, "mem_dma_bd12_3"},
      {0x000a0188, "mem_dma_bd12_2"},
      {0x000a0184, "mem_dma_bd12_1"},
      {0x000a0180, "mem_dma_bd12_0"},
      {0x000a019c, "mem_dma_bd12_7"},
      {0x000a0198, "mem_dma_bd12_6"},
      {0x000a0194, "mem_dma_bd12_5"},
      {0x000a0190, "mem_dma_bd12_4"},
      {0x00031008, "shim_performance_control1"},
      {0x00031000, "shim_performance_control0"},
      {0x0003ff00, "cm_stream_switch_event_port_selection_0"},
      {0x0003ff04, "cm_stream_switch_event_port_selection_1"},
      {0x000a0154, "mem_dma_bd10_5"},
      {0x000a0150, "mem_dma_bd10_4"},
      {0x000a015c, "mem_dma_bd10_7"},
      {0x000a0158, "mem_dma_bd10_6"},
      {0x000a0144, "mem_dma_bd10_1"},
      {0x000a0140, "mem_dma_bd10_0"},
      {0x000a014c, "mem_dma_bd10_3"},
      {0x000a0148, "mem_dma_bd10_2"},
      {0x000a021c, "mem_dma_bd16_7"},
      {0x000a0218, "mem_dma_bd16_6"},
      {0x000a0214, "mem_dma_bd16_5"},
      {0x000a0210, "mem_dma_bd16_4"},
      {0x000a020c, "mem_dma_bd16_3"},
      {0x000a0208, "mem_dma_bd16_2"},
      {0x000a0204, "mem_dma_bd16_1"},
      {0x000a0200, "mem_dma_bd16_0"},
      {0x0001d060, "mm_dma_bd3_0"},
      {0x00094514, "mem_event_group_memory_conflict_enable"},
      {0x0003f334, "cm_stream_switch_slave_east_0_slot1"},
      {0x0003f330, "cm_stream_switch_slave_east_0_slot0"},
      {0x0003f33c, "cm_stream_switch_slave_east_0_slot3"},
      {0x0003f338, "cm_stream_switch_slave_east_0_slot2"},
      {0x00034080, "shim_event_broadcast_a_block_east_set"},
      {0x00014200, "mm_event_status0"},
      {0x00014204, "mm_event_status1"},
      {0x00014208, "mm_event_status2"},
      {0x0001420c, "mm_event_status3"},
      {0x00034074, "shim_event_broadcast_a_block_north_clr"},
      {0x0001f040, "mm_lock4_value"},
      {0x000a0630, "mem_dma_mm2s_0_ctrl"},
      {0x00031588, "cm_performance_counter2_event_value"},
      {0x000340f0, "shim_timer_trig_event_low_value"},
      {0x0003ff34, "cm_stream_switch_adaptive_clock_gate_status"},
      {0x000b0804, "mem_stream_switch_deterministic_merge_arb0_slave2_3"},
      {0x000a00b0, "mem_dma_bd5_4"},
      {0x00094000, "mem_timer_control"},
      {0x000c0420, "mem_locks_overflow_0"},
      {0x000c0424, "mem_locks_overflow_1"},
      {0x000a0270, "mem_dma_bd19_4"},
      {0x000a0274, "mem_dma_bd19_5"},
      {0x000a0278, "mem_dma_bd19_6"},
      {0x000a027c, "mem_dma_bd19_7"},
      {0x000a0260, "mem_dma_bd19_0"},
      {0x000a0264, "mem_dma_bd19_1"},
      {0x000a0268, "mem_dma_bd19_2"},
      {0x000a026c, "mem_dma_bd19_3"},
      {0x000b0128, "mem_stream_switch_slave_config_south_3"},
      {0x000b0124, "mem_stream_switch_slave_config_south_2"},
      {0x000b0120, "mem_stream_switch_slave_config_south_1"},
      {0x000b011c, "mem_stream_switch_slave_config_south_0"},
      {0x000b0130, "mem_stream_switch_slave_config_south_5"},
      {0x0003f244, "cm_stream_switch_slave_fifo_0_slot1"},
      {0x0003f248, "cm_stream_switch_slave_fifo_0_slot2"},
      {0x0001450c, "mm_event_group_lock_enable"},
      {0x0003f24c, "cm_stream_switch_slave_fifo_0_slot3"},
      {0x000c0030, "mem_lock3_value"},
      {0x000a0654, "mem_dma_mm2s_4_start_queue"},
      {0x000a0600, "mem_dma_s2mm_0_ctrl"},
      {0x00094510, "mem_event_group_stream_switch_enable"},
      {0x000c0370, "mem_lock55_value"},
      {0x000a0694, "mem_dma_mm2s_status_5"},
      {0x000a0690, "mem_dma_mm2s_status_4"},
      {0x000a0684, "mem_dma_mm2s_status_1"},
      {0x000a0680, "mem_dma_mm2s_status_0"},
      {0x000a068c, "mem_dma_mm2s_status_3"},
      {0x000a0688, "mem_dma_mm2s_status_2"},
      {0x00094058, "mem_event_broadcast_a_block_south_value"},
      {0x000940c0, "mem_event_broadcast_b_block_east_set"},
      {0x000a0648, "mem_dma_mm2s_3_ctrl"},
      {0x00034060, "cm_event_broadcast_block_west_set"},
      {0x000b0f38, "mem_stream_switch_adaptive_clock_gate_abort_period"},
      {0x00034084, "cm_event_broadcast_block_east_clr"},
      {0x0009108c, "mem_performance_counter3_event_value"},
      {0x00014020, "shim_lock2_value"},
      {0x00011024, "mm_performance_counter1"},
      {0x00011020, "mm_performance_counter0"},
      {0x000a065c, "mem_dma_mm2s_5_start_queue"},
      {0x00014068, "mm_event_broadcast_block_west_value"},
      {0x000b027c, "mem_stream_switch_slave_south_0_slot3"},
      {0x000b0270, "mem_stream_switch_slave_south_0_slot0"},
      {0x000b0274, "mem_stream_switch_slave_south_0_slot1"},
      {0x00032000, "cm_core_control"},
      {0x000c0010, "mem_lock1_value"},
      {0x000a0508, "mem_dma_bd40_2"},
      {0x000a050c, "mem_dma_bd40_3"},
      {0x000a0500, "mem_dma_bd40_0"},
      {0x000a0504, "mem_dma_bd40_1"},
      {0x000a0518, "mem_dma_bd40_6"},
      {0x000a051c, "mem_dma_bd40_7"},
      {0x000a0510, "mem_dma_bd40_4"},
      {0x000a0514, "mem_dma_bd40_5"},
      {0x000a0550, "mem_dma_bd42_4"},
      {0x000a0554, "mem_dma_bd42_5"},
      {0x000a0558, "mem_dma_bd42_6"},
      {0x000a055c, "mem_dma_bd42_7"},
      {0x000a0540, "mem_dma_bd42_0"},
      {0x000a0544, "mem_dma_bd42_1"},
      {0x000a0548, "mem_dma_bd42_2"},
      {0x000a054c, "mem_dma_bd42_3"},
      {0x000a0598, "mem_dma_bd44_6"},
      {0x000a059c, "mem_dma_bd44_7"},
      {0x000a0590, "mem_dma_bd44_4"},
      {0x000a0594, "mem_dma_bd44_5"},
      {0x000a0588, "mem_dma_bd44_2"},
      {0x000a058c, "mem_dma_bd44_3"},
      {0x000a0580, "mem_dma_bd44_0"},
      {0x000a0584, "mem_dma_bd44_1"},
      {0x0003f810, "cm_stream_switch_deterministic_merge_arb1_slave0_1"},
      {0x00094088, "mem_event_broadcast_a_block_east_value"},
      {0x00034054, "cm_event_broadcast_block_south_clr"},
      {0x000c01a0, "mem_lock26_value"},
      {0x00014070, "shim_lock7_value"},
      {0x000b02b8, "mem_stream_switch_slave_south_4_slot2"},
      {0x000b02bc, "mem_stream_switch_slave_south_4_slot3"},
      {0x000b02b0, "mem_stream_switch_slave_south_4_slot0"},
      {0x000b02b4, "mem_stream_switch_slave_south_4_slot1"},
      {0x0003420c, "shim_event_status3"},
      {0x00034208, "shim_event_status2"},
      {0x00034204, "shim_event_status1"},
      {0x00034200, "shim_event_status0"},
      {0x000c0110, "mem_lock17_value"},
      {0x00096048, "mem_memory_control"},
      {0x0003f228, "cm_stream_switch_slave_dma_1_slot2"},
      {0x000a0604, "mem_dma_s2mm_0_start_queue"},
      {0x0003f22c, "cm_stream_switch_slave_dma_1_slot3"},
      {0x0003f220, "cm_stream_switch_slave_dma_1_slot0"},
      {0x0001df24, "mm_dma_s2mm_fot_count_fifo_pop_1"},
      {0x0001df20, "mm_dma_s2mm_fot_count_fifo_pop_0"},
      {0x0003f224, "cm_stream_switch_slave_dma_1_slot1"},
      {0x000940d8, "mem_trace_status"},
      {0x00012110, "mm_ecc_scrubbing_event"},
      {0x0003f020, "cm_stream_switch_master_config_south3"},
      {0x0003f01c, "cm_stream_switch_master_config_south2"},
      {0x0003f018, "cm_stream_switch_master_config_south1"},
      {0x0003f014, "cm_stream_switch_master_config_south0"},
      {0x00031084, "shim_performance_counter1_event_value"},
      {0x00014010, "shim_lock1_value"},
      {0x000340e4, "shim_trace_event1"},
      {0x000340e0, "shim_trace_event0"},
      {0x0001421c, "mm_reserved3"},
      {0x000a040c, "mem_dma_bd32_3"},
      {0x00014214, "mm_reserved1"},
      {0x00014210, "mm_reserved0"},
      {0x000c042c, "mem_locks_underflow_1"},
      {0x000c0428, "mem_locks_underflow_0"},
      {0x000b028c, "mem_stream_switch_slave_south_1_slot3"},
      {0x000c0140, "mem_lock20_value"},
      {0x000b0288, "mem_stream_switch_slave_south_1_slot2"},
      {0x000b0284, "mem_stream_switch_slave_south_1_slot1"},
      {0x000b0280, "mem_stream_switch_slave_south_1_slot0"},
      {0x00014038, "mm_event_broadcast10"},
      {0x0001403c, "mm_event_broadcast11"},
      {0x00014040, "mm_event_broadcast12"},
      {0x00014044, "mm_event_broadcast13"},
      {0x00014048, "mm_event_broadcast14"},
      {0x0001404c, "mm_event_broadcast15"},
      {0x0003f160, "cm_stream_switch_slave_config_mem_trace"},
      {0x000c0418, "mem_locks_event_selection_6"},
      {0x000c041c, "mem_locks_event_selection_7"},
      {0x000c0410, "mem_locks_event_selection_4"},
      {0x000c0414, "mem_locks_event_selection_5"},
      {0x000c0408, "mem_locks_event_selection_2"},
      {0x000c040c, "mem_locks_event_selection_3"},
      {0x000c0400, "mem_locks_event_selection_0"},
      {0x000c0404, "mem_locks_event_selection_1"},
      {0x00094060, "mem_event_broadcast_a_block_west_set"},
      {0x0003f2e0, "cm_stream_switch_slave_west_3_slot0"},
      {0x000c0340, "mem_lock52_value"},
      {0x000c01d0, "mem_lock29_value"},
      {0x000a02fc, "mem_dma_bd23_7"},
      {0x000c03e0, "mem_lock62_value"},
      {0x0001d0e4, "shim_dma_bd7_1"},
      {0x000c01b0, "mem_lock27_value"},
      {0x0003f2e4, "cm_stream_switch_slave_west_3_slot1"},
      {0x000a02f0, "mem_dma_bd23_4"},
      {0x000a02ec, "mem_dma_bd23_3"},
      {0x000a02e8, "mem_dma_bd23_2"},
      {0x000a02e4, "mem_dma_bd23_1"},
      {0x000a02e0, "mem_dma_bd23_0"},
      {0x0003451c, "cm_event_group_broadcast_enable"},
      {0x000a0650, "mem_dma_mm2s_4_ctrl"},
      {0x00094050, "mem_event_broadcast_a_block_south_set"},
      {0x00011080, "mm_performance_counter0_event_value"},
      {0x00032038, "cm_core_processor_bus"},
      {0x0001d08c, "mm_dma_bd4_3"},
      {0x0001d088, "mm_dma_bd4_2"},
      {0x00034068, "cm_event_broadcast_block_west_value"},
      {0x0001d080, "mm_dma_bd4_0"},
      {0x0001d094, "mm_dma_bd4_5"},
      {0x0001d090, "mm_dma_bd4_4"},
      {0x00014408, "mm_edge_detection_event_control"},
      {0x00012120, "mm_ecc_failing_address"},
      {0x0001d044, "mm_dma_bd2_1"},
      {0x0001d040, "mm_dma_bd2_0"},
      {0x0001d04c, "mm_dma_bd2_3"},
      {0x0001d048, "mm_dma_bd2_2"},
      {0x0001d054, "mm_dma_bd2_5"},
      {0x0001d050, "mm_dma_bd2_4"},
      {0x0001d014, "mm_dma_bd0_5"},
      {0x0001d010, "mm_dma_bd0_4"},
      {0x0001d00c, "mm_dma_bd0_3"},
      {0x0001d008, "mm_dma_bd0_2"},
      {0x0001d004, "mm_dma_bd0_1"},
      {0x0001d000, "mm_dma_bd0_0"},
      {0x00014030, "mm_event_broadcast8"},
      {0x00014034, "mm_event_broadcast9"},
      {0x0001d0d4, "mm_dma_bd6_5"},
      {0x0001d0d0, "mm_dma_bd6_4"},
      {0x00014020, "mm_event_broadcast4"},
      {0x00014024, "mm_event_broadcast5"},
      {0x0001d0c4, "mm_dma_bd6_1"},
      {0x0001d0c0, "mm_dma_bd6_0"},
      {0x000a0464, "mem_dma_bd35_1"},
      {0x0001d0c8, "mm_dma_bd6_2"},
      {0x000a05e4, "mem_dma_bd47_1"},
      {0x000a05e0, "mem_dma_bd47_0"},
      {0x000a05ec, "mem_dma_bd47_3"},
      {0x000a05e8, "mem_dma_bd47_2"},
      {0x000a05f4, "mem_dma_bd47_5"},
      {0x000a05f0, "mem_dma_bd47_4"},
      {0x000a05fc, "mem_dma_bd47_7"},
      {0x000a05f8, "mem_dma_bd47_6"},
      {0x0001f010, "mm_lock1_value"},
      {0x000b0814, "mem_stream_switch_deterministic_merge_arb1_slave2_3"},
      {0x00031500, "cm_performance_control0"},
      {0x00031504, "cm_performance_control1"},
      {0x00031508, "cm_performance_control2"},
      {0x000b0810, "mem_stream_switch_deterministic_merge_arb1_slave0_1"},
      {0x00034044, "shim_event_broadcast_a_13"},
      {0x00034040, "shim_event_broadcast_a_12"},
      {0x0003403c, "shim_event_broadcast_a_11"},
      {0x00034038, "shim_event_broadcast_a_10"},
      {0x0003404c, "shim_event_broadcast_a_15"},
      {0x00034048, "shim_event_broadcast_a_14"},
      {0x000a0618, "mem_dma_s2mm_3_ctrl"},
      {0x00094054, "mem_event_broadcast_a_block_south_clr"},
      {0x00014078, "mm_event_broadcast_block_north_value"},
      {0x00014080, "mm_event_broadcast_block_east_set"},
      {0x000b0278, "mem_stream_switch_slave_south_0_slot2"},
      {0x000a0094, "mem_dma_bd4_5"},
      {0x000c0380, "mem_lock56_value"},
      {0x0001d104, "shim_dma_bd8_1"},
      {0x0001f060, "mm_lock6_value"},
      {0x0001d130, "mm_dma_bd9_4"},
      {0x0001d134, "mm_dma_bd9_5"},
      {0x0001d128, "mm_dma_bd9_2"},
      {0x0001d12c, "mm_dma_bd9_3"},
      {0x0001d120, "mm_dma_bd9_0"},
      {0x0001d124, "mm_dma_bd9_1"},
      {0x000c02d0, "mem_lock45_value"},
      {0x0003ff04, "shim_stream_switch_event_port_selection_1"},
      {0x0003ff00, "shim_stream_switch_event_port_selection_0"},
      {0x0003f2c8, "cm_stream_switch_slave_west_1_slot2"},
      {0x0009450c, "mem_event_group_lock_enable"},
      {0x0003f2c0, "cm_stream_switch_slave_west_1_slot0"},
      {0x0003f2c4, "cm_stream_switch_slave_west_1_slot1"},
      {0x0003f110, "cm_stream_switch_slave_config_fifo_0"},
      {0x000c0290, "mem_lock41_value"},
      {0x0001f090, "mm_lock9_value"},
      {0x00036060, "cm_accumulator_control"},
      {0x0003f250, "cm_stream_switch_slave_south_0_slot0"},
      {0x0003f254, "cm_stream_switch_slave_south_0_slot1"},
      {0x0003f258, "cm_stream_switch_slave_south_0_slot2"},
      {0x0003f25c, "cm_stream_switch_slave_south_0_slot3"},
      {0x000140c0, "shim_lock12_value"},
      {0x000c03b0, "mem_lock59_value"},
      {0x0003f310, "cm_stream_switch_slave_north_2_slot0"},
      {0x0001f0a0, "mm_lock10_value"},
      {0x0003f21c, "cm_stream_switch_slave_dma_0_slot3"},
      {0x0003f218, "cm_stream_switch_slave_dma_0_slot2"},
      {0x0003f214, "cm_stream_switch_slave_dma_0_slot1"},
      {0x0003f210, "cm_stream_switch_slave_dma_0_slot0"},
      {0x0003f30c, "cm_stream_switch_slave_north_1_slot3"},
      {0x00034208, "cm_event_status2"},
      {0x0003420c, "cm_event_status3"},
      {0x00034200, "cm_event_status0"},
      {0x00034204, "cm_event_status1"},
      {0x00034048, "cm_event_broadcast14"},
      {0x0003404c, "cm_event_broadcast15"},
      {0x00034040, "cm_event_broadcast12"},
      {0x00034500, "cm_event_group_0_enable"},
      {0x00034038, "cm_event_broadcast10"},
      {0x0003403c, "cm_event_broadcast11"},
      {0x000c0060, "mem_lock6_value"},
      {0x00034088, "shim_event_broadcast_a_block_east_value"},
      {0x0001de1c, "mm_dma_mm2s_1_start_queue"},
      {0x000a00e0, "mem_dma_bd7_0"},
      {0x000a00e4, "mem_dma_bd7_1"},
      {0x000a00e8, "mem_dma_bd7_2"},
      {0x000a00ec, "mem_dma_bd7_3"},
      {0x000a00f0, "mem_dma_bd7_4"},
      {0x000a00f4, "mem_dma_bd7_5"},
      {0x000a00f8, "mem_dma_bd7_6"},
      {0x000a00fc, "mem_dma_bd7_7"},
      {0x000a044c, "mem_dma_bd34_3"},
      {0x000a0448, "mem_dma_bd34_2"},
      {0x000a0444, "mem_dma_bd34_1"},
      {0x000a0440, "mem_dma_bd34_0"},
      {0x000a045c, "mem_dma_bd34_7"},
      {0x000a0458, "mem_dma_bd34_6"},
      {0x000a0454, "mem_dma_bd34_5"},
      {0x000a0450, "mem_dma_bd34_4"},
      {0x000a00b8, "mem_dma_bd5_6"},
      {0x000a00bc, "mem_dma_bd5_7"},
      {0x00094064, "mem_event_broadcast_a_block_west_clr"},
      {0x000a00b4, "mem_dma_bd5_5"},
      {0x000a00a8, "mem_dma_bd5_2"},
      {0x000a00ac, "mem_dma_bd5_3"},
      {0x000a00a0, "mem_dma_bd5_0"},
      {0x000a00a4, "mem_dma_bd5_1"},
      {0x000a0494, "mem_dma_bd36_5"},
      {0x000a0490, "mem_dma_bd36_4"},
      {0x000a049c, "mem_dma_bd36_7"},
      {0x000a0498, "mem_dma_bd36_6"},
      {0x000a0484, "mem_dma_bd36_1"},
      {0x000a0480, "mem_dma_bd36_0"},
      {0x000a048c, "mem_dma_bd36_3"},
      {0x000a0488, "mem_dma_bd36_2"},
      {0x000a03dc, "mem_dma_bd30_7"},
      {0x000a03d8, "mem_dma_bd30_6"},
      {0x000a03d4, "mem_dma_bd30_5"},
      {0x000a03d0, "mem_dma_bd30_4"},
      {0x000a03cc, "mem_dma_bd30_3"},
      {0x000a03c8, "mem_dma_bd30_2"},
      {0x000a03c4, "mem_dma_bd30_1"},
      {0x000a03c0, "mem_dma_bd30_0"},
      {0x000b0298, "mem_stream_switch_slave_south_2_slot2"},
      {0x00014008, "mm_event_generate"},
      {0x00031584, "cm_performance_counter1_event_value"},
      {0x000c00f0, "mem_lock15_value"},
      {0x00034050, "cm_event_broadcast_block_south_set"},
      {0x000a0128, "mem_dma_bd9_2"},
      {0x000a012c, "mem_dma_bd9_3"},
      {0x000a0120, "mem_dma_bd9_0"},
      {0x000a0124, "mem_dma_bd9_1"},
      {0x000a0138, "mem_dma_bd9_6"},
      {0x000a013c, "mem_dma_bd9_7"},
      {0x000a0130, "mem_dma_bd9_4"},
      {0x000a0134, "mem_dma_bd9_5"},
      {0x000c0170, "mem_lock23_value"},
      {0x000a000c, "mem_dma_bd0_3"},
      {0x000a0008, "mem_dma_bd0_2"},
      {0x000a0004, "mem_dma_bd0_1"},
      {0x000a0000, "mem_dma_bd0_0"},
      {0x000a001c, "mem_dma_bd0_7"},
      {0x000a0018, "mem_dma_bd0_6"},
      {0x000a0014, "mem_dma_bd0_5"},
      {0x000a0010, "mem_dma_bd0_4"},
      {0x000b0f34, "mem_stream_switch_adaptive_clock_gate_status"},
      {0x00034020, "cm_event_broadcast4"},
      {0x0003f36c, "cm_stream_switch_slave_east_3_slot3"},
      {0x00034028, "cm_event_broadcast6"},
      {0x0003402c, "cm_event_broadcast7"},
      {0x000140a0, "shim_lock10_value"},
      {0x000a009c, "mem_dma_bd4_7"},
      {0x000a0098, "mem_dma_bd4_6"},
      {0x00034510, "cm_event_group_errors0_enable"},
      {0x000a0090, "mem_dma_bd4_4"},
      {0x000a008c, "mem_dma_bd4_3"},
      {0x000a0088, "mem_dma_bd4_2"},
      {0x000a0084, "mem_dma_bd4_1"},
      {0x000a0080, "mem_dma_bd4_0"},
      {0x000a0054, "mem_dma_bd2_5"},
      {0x000a0050, "mem_dma_bd2_4"},
      {0x000a005c, "mem_dma_bd2_7"},
      {0x000a0058, "mem_dma_bd2_6"},
      {0x000a0044, "mem_dma_bd2_1"},
      {0x000a0040, "mem_dma_bd2_0"},
      {0x000a004c, "mem_dma_bd2_3"},
      {0x000a0048, "mem_dma_bd2_2"},
      {0x0003f10c, "cm_stream_switch_slave_config_tile_ctrl"},
      {0x000b0290, "mem_stream_switch_slave_south_2_slot0"},
      {0x000b0294, "mem_stream_switch_slave_south_2_slot1"},
      {0x0003f804, "cm_stream_switch_deterministic_merge_arb0_slave2_3"},
      {0x000b029c, "mem_stream_switch_slave_south_2_slot3"},
      {0x00094108, "mem_watchpoint2"},
      {0x0009410c, "mem_watchpoint3"},
      {0x00094100, "mem_watchpoint0"},
      {0x00094104, "mem_watchpoint1"},
      {0x0001de14, "mm_dma_mm2s_0_start_queue"},
      {0x0003f800, "cm_stream_switch_deterministic_merge_arb0_slave0_1"},
      {0x00014058, "mm_event_broadcast_block_south_value"},
      {0x000c0220, "mem_lock34_value"},
      {0x00094078, "mem_event_broadcast_a_block_north_value"},
      {0x0003f010, "cm_stream_switch_master_config_fifo0"},
      {0x00094094, "mem_event_broadcast_b_block_south_clr"},
      {0x000340fc, "cm_timer_high"},
      {0x000c0240, "mem_lock36_value"},
      {0x00016010, "mm_memory_control"},
      {0x00032034, "cm_error_halt_event"},
      {0x000a04f8, "mem_dma_bd39_6"},
      {0x00014400, "mm_combo_event_inputs"},
      {0x000a04f0, "mem_dma_bd39_4"},
      {0x000a04f4, "mem_dma_bd39_5"},
      {0x000a04e8, "mem_dma_bd39_2"},
      {0x000a04ec, "mem_dma_bd39_3"},
      {0x000a04e0, "mem_dma_bd39_0"},
      {0x000a04e4, "mem_dma_bd39_1"},
      {0x0009102c, "mem_performance_counter3"},
      {0x00091028, "mem_performance_counter2"},
      {0x00091024, "mem_performance_counter1"},
      {0x00091020, "mem_performance_counter0"},
      {0x0003f814, "cm_stream_switch_deterministic_merge_arb1_slave2_3"},
      {0x000c01f0, "mem_lock31_value"},
      {0x000c0150, "mem_lock21_value"},
      {0x0003f35c, "cm_stream_switch_slave_east_2_slot3"},
      {0x0003f358, "cm_stream_switch_slave_east_2_slot2"},
      {0x0003f354, "cm_stream_switch_slave_east_2_slot1"},
      {0x0003f350, "cm_stream_switch_slave_east_2_slot0"},
      {0x000a0420, "mem_dma_bd33_0"},
      {0x000a0424, "mem_dma_bd33_1"},
      {0x000a0428, "mem_dma_bd33_2"},
      {0x000a042c, "mem_dma_bd33_3"},
      {0x000a0430, "mem_dma_bd33_4"},
      {0x000a0434, "mem_dma_bd33_5"},
      {0x000a0438, "mem_dma_bd33_6"},
      {0x000a043c, "mem_dma_bd33_7"},
      {0x000b0f30, "mem_tile_control_packet_handler_status"},
      {0x0003f264, "cm_stream_switch_slave_south_1_slot1"},
      {0x0003f260, "cm_stream_switch_slave_south_1_slot0"},
      {0x0003f26c, "cm_stream_switch_slave_south_1_slot3"},
      {0x0003f268, "cm_stream_switch_slave_south_1_slot2"},
      {0x00034078, "cm_event_broadcast_block_north_value"},
      {0x00034400, "cm_combo_event_inputs"},
      {0x0001f110, "mm_locks_event_selection_4"},
      {0x000a06c0, "mem_dma_s2mm_current_write_count_4"},
      {0x000a06bc, "mem_dma_s2mm_current_write_count_3"},
      {0x0003f15c, "cm_stream_switch_slave_config_aie_trace"},
      {0x0001f10c, "mm_locks_event_selection_3"},
      {0x10, "shim_lock_step_size"},
      {0x000a06b4, "mem_dma_s2mm_current_write_count_1"},
      {0x000a06b0, "mem_dma_s2mm_current_write_count_0"},
      {0x00034070, "cm_event_broadcast_block_north_set"},
      {0x00034008, "shim_event_generate"},
      {0x000c0100, "mem_lock16_value"},
      {0x000b0818, "mem_stream_switch_deterministic_merge_arb1_ctrl"},
      {0x000140fc, "mm_timer_high"},
      {0x000c0360, "mem_lock54_value"},
      {0x8, "shim_dma_s2mm_step_size"},
      {0x00014500, "mm_event_group_0_enable"},
      {0x0001df1c, "mm_dma_s2mm_current_write_count_1"},
      {0x0001df18, "mm_dma_s2mm_current_write_count_0"},
      {0x000a0628, "mem_dma_s2mm_5_ctrl"},
      {0x0001d204, "shim_dma_s2mm_0_task_queue"},
      {0x000a047c, "mem_dma_bd35_7"},
      {0x000940c4, "mem_event_broadcast_b_block_east_clr"},
      {0x000340e0, "cm_trace_event0"},
      {0x000340e4, "cm_trace_event1"},
      {0x000c0050, "mem_lock5_value"},
      {0x00094048, "mem_event_broadcast14"},
      {0x000b0238, "mem_stream_switch_slave_dma_3_slot2"},
      {0x000b023c, "mem_stream_switch_slave_dma_3_slot3"},
      {0x000b0230, "mem_stream_switch_slave_dma_3_slot0"},
      {0x000b0234, "mem_stream_switch_slave_dma_3_slot1"},
      {0x000c03d0, "mem_lock61_value"},
      {0x000c03f0, "mem_lock63_value"},
      {0x0003f2f8, "cm_stream_switch_slave_north_0_slot2"},
      {0x0003f2fc, "cm_stream_switch_slave_north_0_slot3"},
      {0x0003f2f0, "cm_stream_switch_slave_north_0_slot0"},
      {0x0003f2f4, "cm_stream_switch_slave_north_0_slot1"},
      {0x0001d1c0, "shim_dma_bd14_0"},
      {0x0003f818, "cm_stream_switch_deterministic_merge_arb1_ctrl"},
      {0x0001d1c8, "shim_dma_bd14_2"},
      {0x0001d1cc, "shim_dma_bd14_3"},
      {0x0001d1d0, "shim_dma_bd14_4"},
      {0x0001d1d4, "shim_dma_bd14_5"},
      {0x0001d1d8, "shim_dma_bd14_6"},
      {0x0001d1dc, "shim_dma_bd14_7"},
      {0x000b0314, "mem_stream_switch_slave_trace_slot1"},
      {0x000b0310, "mem_stream_switch_slave_trace_slot0"},
      {0x000b031c, "mem_stream_switch_slave_trace_slot3"},
      {0x000b0318, "mem_stream_switch_slave_trace_slot2"},
      {0x0001d15c, "shim_dma_bd10_7"},
      {0x000c0120, "mem_lock18_value"},
      {0x000a04fc, "mem_dma_bd39_7"},
      {0x00094008, "mem_event_generate"},
      {0x0001f0f0, "mm_lock15_value"},
      {0x00014080, "shim_lock8_value"},
      {0x0003f004, "cm_stream_switch_master_config_dma0"},
      {0x0003f008, "cm_stream_switch_master_config_dma1"},
      {0x0001d090, "shim_dma_bd4_4"},
      {0x0001d094, "shim_dma_bd4_5"},
      {0x0001d098, "shim_dma_bd4_6"},
      {0x0001d09c, "shim_dma_bd4_7"},
      {0x0001d080, "shim_dma_bd4_0"},
      {0x0001d084, "shim_dma_bd4_1"},
      {0x0001d088, "shim_dma_bd4_2"},
      {0x0001d08c, "shim_dma_bd4_3"},
      {0x0001d058, "shim_dma_bd2_6"},
      {0x0001d05c, "shim_dma_bd2_7"},
      {0x0001d050, "shim_dma_bd2_4"},
      {0x0001d054, "shim_dma_bd2_5"},
      {0x0001d048, "shim_dma_bd2_2"},
      {0x0001d04c, "shim_dma_bd2_3"},
      {0x0001d040, "shim_dma_bd2_0"},
      {0x000b0118, "mem_stream_switch_slave_config_tile_ctrl"},
      {0x00014218, "mm_reserved2"},
      {0x00014060, "shim_lock6_value"},
      {0x0003f384, "cm_stream_switch_slave_mem_trace_slot1"},
      {0x0003f124, "cm_stream_switch_slave_config_south_4"},
      {0x0003f38c, "cm_stream_switch_slave_mem_trace_slot3"},
      {0x0003f388, "cm_stream_switch_slave_mem_trace_slot2"},
      {0x000a0644, "mem_dma_mm2s_2_start_queue"},
      {0x0003f114, "cm_stream_switch_slave_config_south_0"},
      {0x0003f120, "cm_stream_switch_slave_config_south_3"},
      {0x0003f11c, "cm_stream_switch_slave_config_south_2"},
      {0x000a0348, "mem_dma_bd26_2"},
      {0x000a034c, "mem_dma_bd26_3"},
      {0x000a0340, "mem_dma_bd26_0"},
      {0x000a0344, "mem_dma_bd26_1"},
      {0x000a0358, "mem_dma_bd26_6"},
      {0x000a035c, "mem_dma_bd26_7"},
      {0x000a0350, "mem_dma_bd26_4"},
      {0x000a0354, "mem_dma_bd26_5"},
      {0x000a0310, "mem_dma_bd24_4"},
      {0x000a0314, "mem_dma_bd24_5"},
      {0x000a0318, "mem_dma_bd24_6"},
      {0x000a031c, "mem_dma_bd24_7"},
      {0x000b02e4, "mem_stream_switch_slave_north_1_slot1"},
      {0x000b02e0, "mem_stream_switch_slave_north_1_slot0"},
      {0x000b02ec, "mem_stream_switch_slave_north_1_slot3"},
      {0x000b02e8, "mem_stream_switch_slave_north_1_slot2"},
      {0x0001d0c8, "shim_dma_bd6_2"},
      {0x000a02dc, "mem_dma_bd22_7"},
      {0x0001d0c0, "shim_dma_bd6_0"},
      {0x000a02d4, "mem_dma_bd22_5"},
      {0x000a02c8, "mem_dma_bd22_2"},
      {0x000a02cc, "mem_dma_bd22_3"},
      {0x000a02c0, "mem_dma_bd22_0"},
      {0x000a02c4, "mem_dma_bd22_1"},
      {0x0001f0e0, "mm_lock14_value"},
      {0x0001d1ac, "shim_dma_bd13_3"},
      {0x0001d1a8, "shim_dma_bd13_2"},
      {0x0001d1a4, "shim_dma_bd13_1"},
      {0x0001d1a0, "shim_dma_bd13_0"},
      {0x0001d1bc, "shim_dma_bd13_7"},
      {0x0001d1b8, "shim_dma_bd13_6"},
      {0x0001d1b4, "shim_dma_bd13_5"},
      {0x0001d1b0, "shim_dma_bd13_4"},
      {0x000c0270, "mem_lock39_value"},
      {0x0001d174, "shim_dma_bd11_5"},
      {0x000c0280, "mem_lock40_value"},
      {0x0001d17c, "shim_dma_bd11_7"},
      {0x0001d178, "shim_dma_bd11_6"},
      {0x0001d164, "shim_dma_bd11_1"},
      {0x0001d160, "shim_dma_bd11_0"},
      {0x0001d16c, "shim_dma_bd11_3"},
      {0x0001d168, "shim_dma_bd11_2"},
      {0x0001d124, "shim_dma_bd9_1"},
      {0x0001d120, "shim_dma_bd9_0"},
      {0x0001d12c, "shim_dma_bd9_3"},
      {0x0001d128, "shim_dma_bd9_2"},
      {0x0001d134, "shim_dma_bd9_5"},
      {0x0001d130, "shim_dma_bd9_4"},
      {0x0001d13c, "shim_dma_bd9_7"},
      {0x0001d138, "shim_dma_bd9_6"},
      {0x00034050, "shim_event_broadcast_a_block_south_set"},
      {0x00034064, "cm_event_broadcast_block_west_clr"},
      {0x0001d024, "shim_dma_bd1_1"},
      {0x0001d020, "shim_dma_bd1_0"},
      {0x0001d02c, "shim_dma_bd1_3"},
      {0x0001d028, "shim_dma_bd1_2"},
      {0x0001d034, "shim_dma_bd1_5"},
      {0x0001d030, "shim_dma_bd1_4"},
      {0x0001d03c, "shim_dma_bd1_7"},
      {0x0001d038, "shim_dma_bd1_6"},
      {0x0001f070, "mm_lock7_value"},
      {0x0001f0c0, "mm_lock12_value"},
      {0x000b0250, "mem_stream_switch_slave_dma_5_slot0"},
      {0x000b0254, "mem_stream_switch_slave_dma_5_slot1"},
      {0x000b0258, "mem_stream_switch_slave_dma_5_slot2"},
      {0x000c0330, "mem_lock51_value"},
      {0x0003f808, "cm_stream_switch_deterministic_merge_arb0_ctrl"},
      {0x00091008, "mem_performance_control2"},
      {0x00091000, "mem_performance_control0"},
      {0x00091004, "mem_performance_control1"},
      {0x00094090, "mem_event_broadcast_b_block_south_set"},
      {0x000c0300, "mem_lock48_value"},
      {0x000340d8, "cm_trace_status"},
      {0x00091080, "mem_performance_counter0_event_value"},
      {0x000140f8, "mm_timer_low"},
      {0x000c00e0, "mem_lock14_value"},
      {0x000940f4, "mem_timer_trig_event_high_value"},
      {0x000b0038, "mem_stream_switch_master_config_north3"},
      {0x000b0034, "mem_stream_switch_master_config_north2"},
      {0x0003f284, "cm_stream_switch_slave_south_3_slot1"},
      {0x0003f280, "cm_stream_switch_slave_south_3_slot0"},
      {0x000a0640, "mem_dma_mm2s_2_ctrl"},
      {0x00034008, "cm_event_generate"},
      {0x000c00b0, "mem_lock11_value"},
      {0x000c0090, "mem_lock9_value"},
      {0x00092120, "mem_ecc_failing_address"},
      {0x00012000, "mm_checkbit_error_generation"},
      {0x000a03a4, "mem_dma_bd29_1"},
      {0x000a03a0, "mem_dma_bd29_0"},
      {0x000a03ac, "mem_dma_bd29_3"},
      {0x000a03a8, "mem_dma_bd29_2"},
      {0x000a03b4, "mem_dma_bd29_5"},
      {0x000a03b0, "mem_dma_bd29_4"},
      {0x000a03bc, "mem_dma_bd29_7"},
      {0x000a03b8, "mem_dma_bd29_6"},
      {0x000fff10, "mem_module_reset_control"},
      {0x000a02a4, "mem_dma_bd21_1"},
      {0x000a02a0, "mem_dma_bd21_0"},
      {0x000a02ac, "mem_dma_bd21_3"},
      {0x000a02a8, "mem_dma_bd21_2"},
      {0x000a02b4, "mem_dma_bd21_5"},
      {0x000a02b0, "mem_dma_bd21_4"},
      {0x000a02bc, "mem_dma_bd21_7"},
      {0x000a02b8, "mem_dma_bd21_6"},
      {0x00036070, "cm_memory_control"},
      {0x000b02cc, "mem_stream_switch_slave_south_5_slot3"},
      {0x000b02c8, "mem_stream_switch_slave_south_5_slot2"},
      {0x000b02c4, "mem_stream_switch_slave_south_5_slot1"},
      {0x000b02c0, "mem_stream_switch_slave_south_5_slot0"},
      {0x00094404, "mem_combo_event_control"},
      {0x20, "shim_dma_bd_step_size"},
      {0x00032110, "cm_ecc_scrubbing_event"},
      {0x00034064, "shim_event_broadcast_a_block_west_clr"},
      {0x000a0658, "mem_dma_mm2s_5_ctrl"},
      {0x000b0138, "mem_stream_switch_slave_config_north_1"},
      {0x000b0134, "mem_stream_switch_slave_config_north_0"},
      {0x000b0140, "mem_stream_switch_slave_config_north_3"},
      {0x000b013c, "mem_stream_switch_slave_config_north_2"},
      {0x00031100, "cm_program_counter"},
      {0x0001d228, "shim_dma_mm2s_status_0"},
      {0x0001d22c, "shim_dma_mm2s_status_1"},
      {0x000a0634, "mem_dma_mm2s_0_start_queue"},
      {0x0001de04, "mm_dma_s2mm_0_start_queue"},
      {0x000340fc, "shim_timer_high"},
      {0x000a06a0, "mem_dma_event_channel_selection"},
      {0x00034520, "cm_event_group_user_event_enable"}
    };
  }
};

/*************************************************************************************
 AIE2ps Registers
 *************************************************************************************/
class AIE2psUsedRegisters : public UsedRegisters {
public:
  AIE2psUsedRegisters() {
    populateRegNameToValueMap();
    populateRegValueToNameMap();
  }
  ~AIE2psUsedRegisters() = default;

  void populateProfileRegisters() {
    // Core modules
    core_addresses.emplace(aie2ps::cm_performance_control0);
    core_addresses.emplace(aie2ps::cm_performance_control1);
    core_addresses.emplace(aie2ps::cm_performance_control2);
    core_addresses.emplace(aie2ps::cm_performance_counter0);
    core_addresses.emplace(aie2ps::cm_performance_counter1);
    core_addresses.emplace(aie2ps::cm_performance_counter2);
    core_addresses.emplace(aie2ps::cm_performance_counter3);
    core_addresses.emplace(aie2ps::cm_performance_counter0_event_value);
    core_addresses.emplace(aie2ps::cm_performance_counter1_event_value);
    core_addresses.emplace(aie2ps::cm_performance_counter2_event_value);
    core_addresses.emplace(aie2ps::cm_performance_counter3_event_value);
    core_addresses.emplace(aie2ps::cm_ecc_scrubbing_event);

    // Memory modules
    memory_addresses.emplace(aie2ps::mm_performance_control0);
    memory_addresses.emplace(aie2ps::mm_performance_control1);
    memory_addresses.emplace(aie2ps::mm_performance_control2);
    memory_addresses.emplace(aie2ps::mm_performance_control3);
    memory_addresses.emplace(aie2ps::mm_performance_counter0);
    memory_addresses.emplace(aie2ps::mm_performance_counter1);
    memory_addresses.emplace(aie2ps::mm_performance_counter2);
    memory_addresses.emplace(aie2ps::mm_performance_counter3);
    memory_addresses.emplace(aie2ps::mm_performance_counter0_event_value);
    memory_addresses.emplace(aie2ps::mm_performance_counter1_event_value);

    // Interface tiles
    interface_addresses.emplace(aie2ps::shim_performance_control0);
    interface_addresses.emplace(aie2ps::shim_performance_control1);
    interface_addresses.emplace(aie2ps::shim_performance_control2);
    interface_addresses.emplace(aie2ps::shim_performance_control3);
    interface_addresses.emplace(aie2ps::shim_performance_control4);
    interface_addresses.emplace(aie2ps::shim_performance_control5);
    interface_addresses.emplace(aie2ps::shim_performance_counter0);
    interface_addresses.emplace(aie2ps::shim_performance_counter1);
    interface_addresses.emplace(aie2ps::shim_performance_counter2);
    interface_addresses.emplace(aie2ps::shim_performance_counter3);
    interface_addresses.emplace(aie2ps::shim_performance_counter4);
    interface_addresses.emplace(aie2ps::shim_performance_counter5);
    interface_addresses.emplace(aie2ps::shim_performance_counter0_event_value);
    interface_addresses.emplace(aie2ps::shim_performance_counter1_event_value);
    interface_addresses.emplace(aie2ps::uc_memory_privileged);

    // Memory tiles
    memory_tile_addresses.emplace(aie2ps::mem_performance_control0);
    memory_tile_addresses.emplace(aie2ps::mem_performance_control1);
    memory_tile_addresses.emplace(aie2ps::mem_performance_control2);
    memory_tile_addresses.emplace(aie2ps::mem_performance_control3);
    memory_tile_addresses.emplace(aie2ps::mem_performance_control4);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter0);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter1);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter2);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter3);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter4);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter5);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter0_event_value);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter1_event_value);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter2_event_value);
    memory_tile_addresses.emplace(aie2ps::mem_performance_counter3_event_value);
  }

  void populateTraceRegisters() {
    // Core modules
    core_addresses.emplace(aie2ps::cm_core_status);
    core_addresses.emplace(aie2ps::cm_trace_control0);
    core_addresses.emplace(aie2ps::cm_trace_control1);
    core_addresses.emplace(aie2ps::cm_trace_status);
    core_addresses.emplace(aie2ps::cm_trace_event0);
    core_addresses.emplace(aie2ps::cm_trace_event1);
    core_addresses.emplace(aie2ps::cm_event_status0);
    core_addresses.emplace(aie2ps::cm_event_status1);
    core_addresses.emplace(aie2ps::cm_event_status2);
    core_addresses.emplace(aie2ps::cm_event_status3);
    core_addresses.emplace(aie2ps::cm_event_broadcast0);
    core_addresses.emplace(aie2ps::cm_event_broadcast1);
    core_addresses.emplace(aie2ps::cm_event_broadcast2);
    core_addresses.emplace(aie2ps::cm_event_broadcast3);
    core_addresses.emplace(aie2ps::cm_event_broadcast4);
    core_addresses.emplace(aie2ps::cm_event_broadcast5);
    core_addresses.emplace(aie2ps::cm_event_broadcast6);
    core_addresses.emplace(aie2ps::cm_event_broadcast7);
    core_addresses.emplace(aie2ps::cm_event_broadcast8);
    core_addresses.emplace(aie2ps::cm_event_broadcast9);
    core_addresses.emplace(aie2ps::cm_event_broadcast10);
    core_addresses.emplace(aie2ps::cm_event_broadcast11);
    core_addresses.emplace(aie2ps::cm_event_broadcast12);
    core_addresses.emplace(aie2ps::cm_event_broadcast13);
    core_addresses.emplace(aie2ps::cm_event_broadcast14);
    core_addresses.emplace(aie2ps::cm_event_broadcast15);
    core_addresses.emplace(aie2ps::cm_event_broadcast_block_south_value);
    core_addresses.emplace(aie2ps::cm_event_broadcast_block_west_value);
    core_addresses.emplace(aie2ps::cm_event_broadcast_block_north_value);
    core_addresses.emplace(aie2ps::cm_event_broadcast_block_east_value);
    core_addresses.emplace(aie2ps::cm_timer_trig_event_low_value);
    core_addresses.emplace(aie2ps::cm_timer_trig_event_high_value);
    core_addresses.emplace(aie2ps::cm_timer_low);
    core_addresses.emplace(aie2ps::cm_timer_high);
    core_addresses.emplace(aie2ps::cm_edge_detection_event_control);
    core_addresses.emplace(aie2ps::cm_stream_switch_event_port_selection_0);
    core_addresses.emplace(aie2ps::cm_stream_switch_event_port_selection_1);

    // Memory modules
    memory_addresses.emplace(aie2ps::mm_trace_control0);
    memory_addresses.emplace(aie2ps::mm_trace_control1);
    memory_addresses.emplace(aie2ps::mm_trace_status);
    memory_addresses.emplace(aie2ps::mm_trace_event0);
    memory_addresses.emplace(aie2ps::mm_trace_event1);
    memory_addresses.emplace(aie2ps::mm_event_status0);
    memory_addresses.emplace(aie2ps::mm_event_status1);
    memory_addresses.emplace(aie2ps::mm_event_status2);
    memory_addresses.emplace(aie2ps::mm_event_status3);
    memory_addresses.emplace(aie2ps::mm_event_broadcast0);
    memory_addresses.emplace(aie2ps::mm_event_broadcast1);
    memory_addresses.emplace(aie2ps::mm_event_broadcast2);
    memory_addresses.emplace(aie2ps::mm_event_broadcast3);
    memory_addresses.emplace(aie2ps::mm_event_broadcast4);
    memory_addresses.emplace(aie2ps::mm_event_broadcast5);
    memory_addresses.emplace(aie2ps::mm_event_broadcast6);
    memory_addresses.emplace(aie2ps::mm_event_broadcast7);
    memory_addresses.emplace(aie2ps::mm_event_broadcast8);
    memory_addresses.emplace(aie2ps::mm_event_broadcast9);
    memory_addresses.emplace(aie2ps::mm_event_broadcast10);
    memory_addresses.emplace(aie2ps::mm_event_broadcast11);
    memory_addresses.emplace(aie2ps::mm_event_broadcast12);
    memory_addresses.emplace(aie2ps::mm_event_broadcast13);
    memory_addresses.emplace(aie2ps::mm_event_broadcast14);
    memory_addresses.emplace(aie2ps::mm_event_broadcast15);

    // Interface tiles
    interface_addresses.emplace(aie2ps::shim_trace_control0);
    interface_addresses.emplace(aie2ps::shim_trace_control1);
    interface_addresses.emplace(aie2ps::shim_trace_status);
    interface_addresses.emplace(aie2ps::shim_trace_event0);
    interface_addresses.emplace(aie2ps::shim_trace_event1);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_0);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_1);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_2);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_3);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_4);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_5);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_6);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_7);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_8);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_9);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_10);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_11);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_12);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_13);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_14);
    interface_addresses.emplace(aie2ps::shim_event_broadcast_a_15);
    interface_addresses.emplace(aie2ps::shim_event_status0);
    interface_addresses.emplace(aie2ps::shim_event_status1);
    interface_addresses.emplace(aie2ps::shim_event_status2);
    interface_addresses.emplace(aie2ps::shim_event_status3);
    interface_addresses.emplace(aie2ps::shim_stream_switch_event_port_selection_0);
    interface_addresses.emplace(aie2ps::shim_stream_switch_event_port_selection_1);

    // Memory tiles
    memory_tile_addresses.emplace(aie2ps::mem_trace_control0);
    memory_tile_addresses.emplace(aie2ps::mem_trace_control1);
    memory_tile_addresses.emplace(aie2ps::mem_trace_status);
    memory_tile_addresses.emplace(aie2ps::mem_trace_event0);
    memory_tile_addresses.emplace(aie2ps::mem_trace_event1);
    memory_tile_addresses.emplace(aie2ps::mem_dma_event_channel_selection);
    memory_tile_addresses.emplace(aie2ps::mem_edge_detection_event_control);
    memory_tile_addresses.emplace(aie2ps::mem_stream_switch_event_port_selection_0);
    memory_tile_addresses.emplace(aie2ps::mem_stream_switch_event_port_selection_1);
    memory_tile_addresses.emplace(aie2ps::mem_event_broadcast0);
    memory_tile_addresses.emplace(aie2ps::mem_event_status0);
    memory_tile_addresses.emplace(aie2ps::mem_event_status1);
    memory_tile_addresses.emplace(aie2ps::mem_event_status2);
    memory_tile_addresses.emplace(aie2ps::mem_event_status3);
    memory_tile_addresses.emplace(aie2ps::mem_event_status4);
    memory_tile_addresses.emplace(aie2ps::mem_event_status5);
  }

  void populateRegNameToValueMap() {
    regNameToValues = {
      {"cm_core_bmll0_part1", aie2ps::cm_core_bmll0_part1},
      {"cm_core_bmll0_part2", aie2ps::cm_core_bmll0_part2},
      {"cm_core_bmll0_part3", aie2ps::cm_core_bmll0_part3},
      {"cm_core_bmll0_part4", aie2ps::cm_core_bmll0_part4},
      {"cm_core_bmlh0_part1", aie2ps::cm_core_bmlh0_part1},
      {"cm_core_bmlh0_part2", aie2ps::cm_core_bmlh0_part2},
      {"cm_core_bmlh0_part3", aie2ps::cm_core_bmlh0_part3},
      {"cm_core_bmlh0_part4", aie2ps::cm_core_bmlh0_part4},
      {"cm_core_bmhl0_part1", aie2ps::cm_core_bmhl0_part1},
      {"cm_core_bmhl0_part2", aie2ps::cm_core_bmhl0_part2},
      {"cm_core_bmhl0_part3", aie2ps::cm_core_bmhl0_part3},
      {"cm_core_bmhl0_part4", aie2ps::cm_core_bmhl0_part4},
      {"cm_core_bmhh0_part1", aie2ps::cm_core_bmhh0_part1},
      {"cm_core_bmhh0_part2", aie2ps::cm_core_bmhh0_part2},
      {"cm_core_bmhh0_part3", aie2ps::cm_core_bmhh0_part3},
      {"cm_core_bmhh0_part4", aie2ps::cm_core_bmhh0_part4},
      {"cm_core_bmll1_part1", aie2ps::cm_core_bmll1_part1},
      {"cm_core_bmll1_part2", aie2ps::cm_core_bmll1_part2},
      {"cm_core_bmll1_part3", aie2ps::cm_core_bmll1_part3},
      {"cm_core_bmll1_part4", aie2ps::cm_core_bmll1_part4},
      {"cm_core_bmlh1_part1", aie2ps::cm_core_bmlh1_part1},
      {"cm_core_bmlh1_part2", aie2ps::cm_core_bmlh1_part2},
      {"cm_core_bmlh1_part3", aie2ps::cm_core_bmlh1_part3},
      {"cm_core_bmlh1_part4", aie2ps::cm_core_bmlh1_part4},
      {"cm_core_bmhl1_part1", aie2ps::cm_core_bmhl1_part1},
      {"cm_core_bmhl1_part2", aie2ps::cm_core_bmhl1_part2},
      {"cm_core_bmhl1_part3", aie2ps::cm_core_bmhl1_part3},
      {"cm_core_bmhl1_part4", aie2ps::cm_core_bmhl1_part4},
      {"cm_core_bmhh1_part1", aie2ps::cm_core_bmhh1_part1},
      {"cm_core_bmhh1_part2", aie2ps::cm_core_bmhh1_part2},
      {"cm_core_bmhh1_part3", aie2ps::cm_core_bmhh1_part3},
      {"cm_core_bmhh1_part4", aie2ps::cm_core_bmhh1_part4},
      {"cm_core_bmll2_part1", aie2ps::cm_core_bmll2_part1},
      {"cm_core_bmll2_part2", aie2ps::cm_core_bmll2_part2},
      {"cm_core_bmll2_part3", aie2ps::cm_core_bmll2_part3},
      {"cm_core_bmll2_part4", aie2ps::cm_core_bmll2_part4},
      {"cm_core_bmlh2_part1", aie2ps::cm_core_bmlh2_part1},
      {"cm_core_bmlh2_part2", aie2ps::cm_core_bmlh2_part2},
      {"cm_core_bmlh2_part3", aie2ps::cm_core_bmlh2_part3},
      {"cm_core_bmlh2_part4", aie2ps::cm_core_bmlh2_part4},
      {"cm_core_bmhl2_part1", aie2ps::cm_core_bmhl2_part1},
      {"cm_core_bmhl2_part2", aie2ps::cm_core_bmhl2_part2},
      {"cm_core_bmhl2_part3", aie2ps::cm_core_bmhl2_part3},
      {"cm_core_bmhl2_part4", aie2ps::cm_core_bmhl2_part4},
      {"cm_core_bmhh2_part1", aie2ps::cm_core_bmhh2_part1},
      {"cm_core_bmhh2_part2", aie2ps::cm_core_bmhh2_part2},
      {"cm_core_bmhh2_part3", aie2ps::cm_core_bmhh2_part3},
      {"cm_core_bmhh2_part4", aie2ps::cm_core_bmhh2_part4},
      {"cm_core_bmll3_part1", aie2ps::cm_core_bmll3_part1},
      {"cm_core_bmll3_part2", aie2ps::cm_core_bmll3_part2},
      {"cm_core_bmll3_part3", aie2ps::cm_core_bmll3_part3},
      {"cm_core_bmll3_part4", aie2ps::cm_core_bmll3_part4},
      {"cm_core_bmlh3_part1", aie2ps::cm_core_bmlh3_part1},
      {"cm_core_bmlh3_part2", aie2ps::cm_core_bmlh3_part2},
      {"cm_core_bmlh3_part3", aie2ps::cm_core_bmlh3_part3},
      {"cm_core_bmlh3_part4", aie2ps::cm_core_bmlh3_part4},
      {"cm_core_bmhl3_part1", aie2ps::cm_core_bmhl3_part1},
      {"cm_core_bmhl3_part2", aie2ps::cm_core_bmhl3_part2},
      {"cm_core_bmhl3_part3", aie2ps::cm_core_bmhl3_part3},
      {"cm_core_bmhl3_part4", aie2ps::cm_core_bmhl3_part4},
      {"cm_core_bmhh3_part1", aie2ps::cm_core_bmhh3_part1},
      {"cm_core_bmhh3_part2", aie2ps::cm_core_bmhh3_part2},
      {"cm_core_bmhh3_part3", aie2ps::cm_core_bmhh3_part3},
      {"cm_core_bmhh3_part4", aie2ps::cm_core_bmhh3_part4},
      {"cm_core_bmll4_part1", aie2ps::cm_core_bmll4_part1},
      {"cm_core_bmll4_part2", aie2ps::cm_core_bmll4_part2},
      {"cm_core_bmll4_part3", aie2ps::cm_core_bmll4_part3},
      {"cm_core_bmll4_part4", aie2ps::cm_core_bmll4_part4},
      {"cm_core_bmlh4_part1", aie2ps::cm_core_bmlh4_part1},
      {"cm_core_bmlh4_part2", aie2ps::cm_core_bmlh4_part2},
      {"cm_core_bmlh4_part3", aie2ps::cm_core_bmlh4_part3},
      {"cm_core_bmlh4_part4", aie2ps::cm_core_bmlh4_part4},
      {"cm_core_bmhl4_part1", aie2ps::cm_core_bmhl4_part1},
      {"cm_core_bmhl4_part2", aie2ps::cm_core_bmhl4_part2},
      {"cm_core_bmhl4_part3", aie2ps::cm_core_bmhl4_part3},
      {"cm_core_bmhl4_part4", aie2ps::cm_core_bmhl4_part4},
      {"cm_core_bmhh4_part1", aie2ps::cm_core_bmhh4_part1},
      {"cm_core_bmhh4_part2", aie2ps::cm_core_bmhh4_part2},
      {"cm_core_bmhh4_part3", aie2ps::cm_core_bmhh4_part3},
      {"cm_core_bmhh4_part4", aie2ps::cm_core_bmhh4_part4},
      {"cm_core_bmll5_part1", aie2ps::cm_core_bmll5_part1},
      {"cm_core_bmll5_part2", aie2ps::cm_core_bmll5_part2},
      {"cm_core_bmll5_part3", aie2ps::cm_core_bmll5_part3},
      {"cm_core_bmll5_part4", aie2ps::cm_core_bmll5_part4},
      {"cm_core_bmlh5_part1", aie2ps::cm_core_bmlh5_part1},
      {"cm_core_bmlh5_part2", aie2ps::cm_core_bmlh5_part2},
      {"cm_core_bmlh5_part3", aie2ps::cm_core_bmlh5_part3},
      {"cm_core_bmlh5_part4", aie2ps::cm_core_bmlh5_part4},
      {"cm_core_bmhl5_part1", aie2ps::cm_core_bmhl5_part1},
      {"cm_core_bmhl5_part2", aie2ps::cm_core_bmhl5_part2},
      {"cm_core_bmhl5_part3", aie2ps::cm_core_bmhl5_part3},
      {"cm_core_bmhl5_part4", aie2ps::cm_core_bmhl5_part4},
      {"cm_core_bmhh5_part1", aie2ps::cm_core_bmhh5_part1},
      {"cm_core_bmhh5_part2", aie2ps::cm_core_bmhh5_part2},
      {"cm_core_bmhh5_part3", aie2ps::cm_core_bmhh5_part3},
      {"cm_core_bmhh5_part4", aie2ps::cm_core_bmhh5_part4},
      {"cm_core_bmll6_part1", aie2ps::cm_core_bmll6_part1},
      {"cm_core_bmll6_part2", aie2ps::cm_core_bmll6_part2},
      {"cm_core_bmll6_part3", aie2ps::cm_core_bmll6_part3},
      {"cm_core_bmll6_part4", aie2ps::cm_core_bmll6_part4},
      {"cm_core_bmlh6_part1", aie2ps::cm_core_bmlh6_part1},
      {"cm_core_bmlh6_part2", aie2ps::cm_core_bmlh6_part2},
      {"cm_core_bmlh6_part3", aie2ps::cm_core_bmlh6_part3},
      {"cm_core_bmlh6_part4", aie2ps::cm_core_bmlh6_part4},
      {"cm_core_bmhl6_part1", aie2ps::cm_core_bmhl6_part1},
      {"cm_core_bmhl6_part2", aie2ps::cm_core_bmhl6_part2},
      {"cm_core_bmhl6_part3", aie2ps::cm_core_bmhl6_part3},
      {"cm_core_bmhl6_part4", aie2ps::cm_core_bmhl6_part4},
      {"cm_core_bmhh6_part1", aie2ps::cm_core_bmhh6_part1},
      {"cm_core_bmhh6_part2", aie2ps::cm_core_bmhh6_part2},
      {"cm_core_bmhh6_part3", aie2ps::cm_core_bmhh6_part3},
      {"cm_core_bmhh6_part4", aie2ps::cm_core_bmhh6_part4},
      {"cm_core_bmll7_part1", aie2ps::cm_core_bmll7_part1},
      {"cm_core_bmll7_part2", aie2ps::cm_core_bmll7_part2},
      {"cm_core_bmll7_part3", aie2ps::cm_core_bmll7_part3},
      {"cm_core_bmll7_part4", aie2ps::cm_core_bmll7_part4},
      {"cm_core_bmlh7_part1", aie2ps::cm_core_bmlh7_part1},
      {"cm_core_bmlh7_part2", aie2ps::cm_core_bmlh7_part2},
      {"cm_core_bmlh7_part3", aie2ps::cm_core_bmlh7_part3},
      {"cm_core_bmlh7_part4", aie2ps::cm_core_bmlh7_part4},
      {"cm_core_bmhl7_part1", aie2ps::cm_core_bmhl7_part1},
      {"cm_core_bmhl7_part2", aie2ps::cm_core_bmhl7_part2},
      {"cm_core_bmhl7_part3", aie2ps::cm_core_bmhl7_part3},
      {"cm_core_bmhl7_part4", aie2ps::cm_core_bmhl7_part4},
      {"cm_core_bmhh7_part1", aie2ps::cm_core_bmhh7_part1},
      {"cm_core_bmhh7_part2", aie2ps::cm_core_bmhh7_part2},
      {"cm_core_bmhh7_part3", aie2ps::cm_core_bmhh7_part3},
      {"cm_core_bmhh7_part4", aie2ps::cm_core_bmhh7_part4},
      {"cm_core_x0_part1", aie2ps::cm_core_x0_part1},
      {"cm_core_x0_part2", aie2ps::cm_core_x0_part2},
      {"cm_core_x0_part3", aie2ps::cm_core_x0_part3},
      {"cm_core_x0_part4", aie2ps::cm_core_x0_part4},
      {"cm_core_x1_part1", aie2ps::cm_core_x1_part1},
      {"cm_core_x1_part2", aie2ps::cm_core_x1_part2},
      {"cm_core_x1_part3", aie2ps::cm_core_x1_part3},
      {"cm_core_x1_part4", aie2ps::cm_core_x1_part4},
      {"cm_core_x2_part1", aie2ps::cm_core_x2_part1},
      {"cm_core_x2_part2", aie2ps::cm_core_x2_part2},
      {"cm_core_x2_part3", aie2ps::cm_core_x2_part3},
      {"cm_core_x2_part4", aie2ps::cm_core_x2_part4},
      {"cm_core_x3_part1", aie2ps::cm_core_x3_part1},
      {"cm_core_x3_part2", aie2ps::cm_core_x3_part2},
      {"cm_core_x3_part3", aie2ps::cm_core_x3_part3},
      {"cm_core_x3_part4", aie2ps::cm_core_x3_part4},
      {"cm_core_x4_part1", aie2ps::cm_core_x4_part1},
      {"cm_core_x4_part2", aie2ps::cm_core_x4_part2},
      {"cm_core_x4_part3", aie2ps::cm_core_x4_part3},
      {"cm_core_x4_part4", aie2ps::cm_core_x4_part4},
      {"cm_core_x5_part1", aie2ps::cm_core_x5_part1},
      {"cm_core_x5_part2", aie2ps::cm_core_x5_part2},
      {"cm_core_x5_part3", aie2ps::cm_core_x5_part3},
      {"cm_core_x5_part4", aie2ps::cm_core_x5_part4},
      {"cm_core_x6_part1", aie2ps::cm_core_x6_part1},
      {"cm_core_x6_part2", aie2ps::cm_core_x6_part2},
      {"cm_core_x6_part3", aie2ps::cm_core_x6_part3},
      {"cm_core_x6_part4", aie2ps::cm_core_x6_part4},
      {"cm_core_x7_part1", aie2ps::cm_core_x7_part1},
      {"cm_core_x7_part2", aie2ps::cm_core_x7_part2},
      {"cm_core_x7_part3", aie2ps::cm_core_x7_part3},
      {"cm_core_x7_part4", aie2ps::cm_core_x7_part4},
      {"cm_core_x8_part1", aie2ps::cm_core_x8_part1},
      {"cm_core_x8_part2", aie2ps::cm_core_x8_part2},
      {"cm_core_x8_part3", aie2ps::cm_core_x8_part3},
      {"cm_core_x8_part4", aie2ps::cm_core_x8_part4},
      {"cm_core_x9_part1", aie2ps::cm_core_x9_part1},
      {"cm_core_x9_part2", aie2ps::cm_core_x9_part2},
      {"cm_core_x9_part3", aie2ps::cm_core_x9_part3},
      {"cm_core_x9_part4", aie2ps::cm_core_x9_part4},
      {"cm_core_x10_part1", aie2ps::cm_core_x10_part1},
      {"cm_core_x10_part2", aie2ps::cm_core_x10_part2},
      {"cm_core_x10_part3", aie2ps::cm_core_x10_part3},
      {"cm_core_x10_part4", aie2ps::cm_core_x10_part4},
      {"cm_core_x11_part1", aie2ps::cm_core_x11_part1},
      {"cm_core_x11_part2", aie2ps::cm_core_x11_part2},
      {"cm_core_x11_part3", aie2ps::cm_core_x11_part3},
      {"cm_core_x11_part4", aie2ps::cm_core_x11_part4},
      {"cm_core_ldfifol0_part1", aie2ps::cm_core_ldfifol0_part1},
      {"cm_core_ldfifol0_part2", aie2ps::cm_core_ldfifol0_part2},
      {"cm_core_ldfifol0_part3", aie2ps::cm_core_ldfifol0_part3},
      {"cm_core_ldfifol0_part4", aie2ps::cm_core_ldfifol0_part4},
      {"cm_core_ldfifoh0_part1", aie2ps::cm_core_ldfifoh0_part1},
      {"cm_core_ldfifoh0_part2", aie2ps::cm_core_ldfifoh0_part2},
      {"cm_core_ldfifoh0_part3", aie2ps::cm_core_ldfifoh0_part3},
      {"cm_core_ldfifoh0_part4", aie2ps::cm_core_ldfifoh0_part4},
      {"cm_core_ldfifol1_part1", aie2ps::cm_core_ldfifol1_part1},
      {"cm_core_ldfifol1_part2", aie2ps::cm_core_ldfifol1_part2},
      {"cm_core_ldfifol1_part3", aie2ps::cm_core_ldfifol1_part3},
      {"cm_core_ldfifol1_part4", aie2ps::cm_core_ldfifol1_part4},
      {"cm_core_ldfifoh1_part1", aie2ps::cm_core_ldfifoh1_part1},
      {"cm_core_ldfifoh1_part2", aie2ps::cm_core_ldfifoh1_part2},
      {"cm_core_ldfifoh1_part3", aie2ps::cm_core_ldfifoh1_part3},
      {"cm_core_ldfifoh1_part4", aie2ps::cm_core_ldfifoh1_part4},
      {"cm_core_stfifol_part1", aie2ps::cm_core_stfifol_part1},
      {"cm_core_stfifol_part2", aie2ps::cm_core_stfifol_part2},
      {"cm_core_stfifol_part3", aie2ps::cm_core_stfifol_part3},
      {"cm_core_stfifol_part4", aie2ps::cm_core_stfifol_part4},
      {"cm_core_stfifoh_part1", aie2ps::cm_core_stfifoh_part1},
      {"cm_core_stfifoh_part2", aie2ps::cm_core_stfifoh_part2},
      {"cm_core_stfifoh_part3", aie2ps::cm_core_stfifoh_part3},
      {"cm_core_stfifoh_part4", aie2ps::cm_core_stfifoh_part4},
      {"cm_core_fifoxtra_part1", aie2ps::cm_core_fifoxtra_part1},
      {"cm_core_fifoxtra_part2", aie2ps::cm_core_fifoxtra_part2},
      {"cm_core_fifoxtra_part3", aie2ps::cm_core_fifoxtra_part3},
      {"cm_core_fifoxtra_part4", aie2ps::cm_core_fifoxtra_part4},
      {"cm_core_eg0", aie2ps::cm_core_eg0},
      {"cm_core_eg1", aie2ps::cm_core_eg1},
      {"cm_core_eg2", aie2ps::cm_core_eg2},
      {"cm_core_eg3", aie2ps::cm_core_eg3},
      {"cm_core_eg4", aie2ps::cm_core_eg4},
      {"cm_core_eg5", aie2ps::cm_core_eg5},
      {"cm_core_eg6", aie2ps::cm_core_eg6},
      {"cm_core_eg7", aie2ps::cm_core_eg7},
      {"cm_core_eg8", aie2ps::cm_core_eg8},
      {"cm_core_eg9", aie2ps::cm_core_eg9},
      {"cm_core_eg10", aie2ps::cm_core_eg10},
      {"cm_core_eg11", aie2ps::cm_core_eg11},
      {"cm_core_f0", aie2ps::cm_core_f0},
      {"cm_core_f1", aie2ps::cm_core_f1},
      {"cm_core_f2", aie2ps::cm_core_f2},
      {"cm_core_f3", aie2ps::cm_core_f3},
      {"cm_core_f4", aie2ps::cm_core_f4},
      {"cm_core_f5", aie2ps::cm_core_f5},
      {"cm_core_f6", aie2ps::cm_core_f6},
      {"cm_core_f7", aie2ps::cm_core_f7},
      {"cm_core_f8", aie2ps::cm_core_f8},
      {"cm_core_f9", aie2ps::cm_core_f9},
      {"cm_core_f10", aie2ps::cm_core_f10},
      {"cm_core_f11", aie2ps::cm_core_f11},
      {"cm_core_r0", aie2ps::cm_core_r0},
      {"cm_core_r1", aie2ps::cm_core_r1},
      {"cm_core_r2", aie2ps::cm_core_r2},
      {"cm_core_r3", aie2ps::cm_core_r3},
      {"cm_core_r4", aie2ps::cm_core_r4},
      {"cm_core_r5", aie2ps::cm_core_r5},
      {"cm_core_r6", aie2ps::cm_core_r6},
      {"cm_core_r7", aie2ps::cm_core_r7},
      {"cm_core_r8", aie2ps::cm_core_r8},
      {"cm_core_r9", aie2ps::cm_core_r9},
      {"cm_core_r10", aie2ps::cm_core_r10},
      {"cm_core_r11", aie2ps::cm_core_r11},
      {"cm_core_r12", aie2ps::cm_core_r12},
      {"cm_core_r13", aie2ps::cm_core_r13},
      {"cm_core_r14", aie2ps::cm_core_r14},
      {"cm_core_r15", aie2ps::cm_core_r15},
      {"cm_core_r16", aie2ps::cm_core_r16},
      {"cm_core_r17", aie2ps::cm_core_r17},
      {"cm_core_r18", aie2ps::cm_core_r18},
      {"cm_core_r19", aie2ps::cm_core_r19},
      {"cm_core_r20", aie2ps::cm_core_r20},
      {"cm_core_r21", aie2ps::cm_core_r21},
      {"cm_core_r22", aie2ps::cm_core_r22},
      {"cm_core_r23", aie2ps::cm_core_r23},
      {"cm_core_r24", aie2ps::cm_core_r24},
      {"cm_core_r25", aie2ps::cm_core_r25},
      {"cm_core_r26", aie2ps::cm_core_r26},
      {"cm_core_r27", aie2ps::cm_core_r27},
      {"cm_core_r28", aie2ps::cm_core_r28},
      {"cm_core_r29", aie2ps::cm_core_r29},
      {"cm_core_r30", aie2ps::cm_core_r30},
      {"cm_core_r31", aie2ps::cm_core_r31},
      {"cm_core_m0", aie2ps::cm_core_m0},
      {"cm_core_m1", aie2ps::cm_core_m1},
      {"cm_core_m2", aie2ps::cm_core_m2},
      {"cm_core_m3", aie2ps::cm_core_m3},
      {"cm_core_m4", aie2ps::cm_core_m4},
      {"cm_core_m5", aie2ps::cm_core_m5},
      {"cm_core_m6", aie2ps::cm_core_m6},
      {"cm_core_m7", aie2ps::cm_core_m7},
      {"cm_core_dn0", aie2ps::cm_core_dn0},
      {"cm_core_dn1", aie2ps::cm_core_dn1},
      {"cm_core_dn2", aie2ps::cm_core_dn2},
      {"cm_core_dn3", aie2ps::cm_core_dn3},
      {"cm_core_dn4", aie2ps::cm_core_dn4},
      {"cm_core_dn5", aie2ps::cm_core_dn5},
      {"cm_core_dn6", aie2ps::cm_core_dn6},
      {"cm_core_dn7", aie2ps::cm_core_dn7},
      {"cm_core_dj0", aie2ps::cm_core_dj0},
      {"cm_core_dj1", aie2ps::cm_core_dj1},
      {"cm_core_dj2", aie2ps::cm_core_dj2},
      {"cm_core_dj3", aie2ps::cm_core_dj3},
      {"cm_core_dj4", aie2ps::cm_core_dj4},
      {"cm_core_dj5", aie2ps::cm_core_dj5},
      {"cm_core_dj6", aie2ps::cm_core_dj6},
      {"cm_core_dj7", aie2ps::cm_core_dj7},
      {"cm_core_dc0", aie2ps::cm_core_dc0},
      {"cm_core_dc1", aie2ps::cm_core_dc1},
      {"cm_core_dc2", aie2ps::cm_core_dc2},
      {"cm_core_dc3", aie2ps::cm_core_dc3},
      {"cm_core_dc4", aie2ps::cm_core_dc4},
      {"cm_core_dc5", aie2ps::cm_core_dc5},
      {"cm_core_dc6", aie2ps::cm_core_dc6},
      {"cm_core_dc7", aie2ps::cm_core_dc7},
      {"cm_core_p0", aie2ps::cm_core_p0},
      {"cm_core_p1", aie2ps::cm_core_p1},
      {"cm_core_p2", aie2ps::cm_core_p2},
      {"cm_core_p3", aie2ps::cm_core_p3},
      {"cm_core_p4", aie2ps::cm_core_p4},
      {"cm_core_p5", aie2ps::cm_core_p5},
      {"cm_core_p6", aie2ps::cm_core_p6},
      {"cm_core_p7", aie2ps::cm_core_p7},
      {"cm_core_s0", aie2ps::cm_core_s0},
      {"cm_core_s1", aie2ps::cm_core_s1},
      {"cm_core_s2", aie2ps::cm_core_s2},
      {"cm_core_s3", aie2ps::cm_core_s3},
      {"cm_core_pc", aie2ps::cm_core_pc},
      {"cm_program_counter", aie2ps::cm_program_counter},
      {"cm_core_fc", aie2ps::cm_core_fc},
      {"cm_core_sp", aie2ps::cm_core_sp},
      {"cm_core_lr", aie2ps::cm_core_lr},
      {"cm_core_ls", aie2ps::cm_core_ls},
      {"cm_core_le", aie2ps::cm_core_le},
      {"cm_core_lc", aie2ps::cm_core_lc},
      {"cm_core_lci", aie2ps::cm_core_lci},
      {"cm_core_cr1", aie2ps::cm_core_cr1},
      {"cm_core_cr2", aie2ps::cm_core_cr2},
      {"cm_core_cr3", aie2ps::cm_core_cr3},
      {"cm_core_sr1", aie2ps::cm_core_sr1},
      {"cm_core_sr2", aie2ps::cm_core_sr2},
      {"cm_timer_control", aie2ps::cm_timer_control},
      {"cm_event_generate", aie2ps::cm_event_generate},
      {"cm_event_broadcast0", aie2ps::cm_event_broadcast0},
      {"cm_event_broadcast1", aie2ps::cm_event_broadcast1},
      {"cm_event_broadcast2", aie2ps::cm_event_broadcast2},
      {"cm_event_broadcast3", aie2ps::cm_event_broadcast3},
      {"cm_event_broadcast4", aie2ps::cm_event_broadcast4},
      {"cm_event_broadcast5", aie2ps::cm_event_broadcast5},
      {"cm_event_broadcast6", aie2ps::cm_event_broadcast6},
      {"cm_event_broadcast7", aie2ps::cm_event_broadcast7},
      {"cm_event_broadcast8", aie2ps::cm_event_broadcast8},
      {"cm_event_broadcast9", aie2ps::cm_event_broadcast9},
      {"cm_event_broadcast10", aie2ps::cm_event_broadcast10},
      {"cm_event_broadcast11", aie2ps::cm_event_broadcast11},
      {"cm_event_broadcast12", aie2ps::cm_event_broadcast12},
      {"cm_event_broadcast13", aie2ps::cm_event_broadcast13},
      {"cm_event_broadcast14", aie2ps::cm_event_broadcast14},
      {"cm_event_broadcast15", aie2ps::cm_event_broadcast15},
      {"cm_event_broadcast_block_south_set", aie2ps::cm_event_broadcast_block_south_set},
      {"cm_event_broadcast_block_south_clr", aie2ps::cm_event_broadcast_block_south_clr},
      {"cm_event_broadcast_block_south_value", aie2ps::cm_event_broadcast_block_south_value},
      {"cm_event_broadcast_block_west_set", aie2ps::cm_event_broadcast_block_west_set},
      {"cm_event_broadcast_block_west_clr", aie2ps::cm_event_broadcast_block_west_clr},
      {"cm_event_broadcast_block_west_value", aie2ps::cm_event_broadcast_block_west_value},
      {"cm_event_broadcast_block_north_set", aie2ps::cm_event_broadcast_block_north_set},
      {"cm_event_broadcast_block_north_clr", aie2ps::cm_event_broadcast_block_north_clr},
      {"cm_event_broadcast_block_north_value", aie2ps::cm_event_broadcast_block_north_value},
      {"cm_event_broadcast_block_east_set", aie2ps::cm_event_broadcast_block_east_set},
      {"cm_event_broadcast_block_east_clr", aie2ps::cm_event_broadcast_block_east_clr},
      {"cm_event_broadcast_block_east_value", aie2ps::cm_event_broadcast_block_east_value},
      {"cm_trace_control0", aie2ps::cm_trace_control0},
      {"cm_trace_control1", aie2ps::cm_trace_control1},
      {"cm_trace_status", aie2ps::cm_trace_status},
      {"cm_trace_event0", aie2ps::cm_trace_event0},
      {"cm_trace_event1", aie2ps::cm_trace_event1},
      {"cm_timer_trig_event_low_value", aie2ps::cm_timer_trig_event_low_value},
      {"cm_timer_trig_event_high_value", aie2ps::cm_timer_trig_event_high_value},
      {"cm_timer_low", aie2ps::cm_timer_low},
      {"cm_timer_high", aie2ps::cm_timer_high},
      {"cm_event_status0", aie2ps::cm_event_status0},
      {"cm_event_status1", aie2ps::cm_event_status1},
      {"cm_event_status2", aie2ps::cm_event_status2},
      {"cm_event_status3", aie2ps::cm_event_status3},
      {"cm_combo_event_inputs", aie2ps::cm_combo_event_inputs},
      {"cm_combo_event_control", aie2ps::cm_combo_event_control},
      {"cm_edge_detection_event_control", aie2ps::cm_edge_detection_event_control},
      {"cm_event_group_0_enable", aie2ps::cm_event_group_0_enable},
      {"cm_event_group_pc_enable", aie2ps::cm_event_group_pc_enable},
      {"cm_event_group_core_stall_enable", aie2ps::cm_event_group_core_stall_enable},
      {"cm_event_group_core_program_flow_enable", aie2ps::cm_event_group_core_program_flow_enable},
      {"cm_event_group_errors0_enable", aie2ps::cm_event_group_errors0_enable},
      {"cm_event_group_errors1_enable", aie2ps::cm_event_group_errors1_enable},
      {"cm_event_group_stream_switch_enable", aie2ps::cm_event_group_stream_switch_enable},
      {"cm_event_group_broadcast_enable", aie2ps::cm_event_group_broadcast_enable},
      {"cm_event_group_user_event_enable", aie2ps::cm_event_group_user_event_enable},
      {"cm_accumulator_control", aie2ps::cm_accumulator_control},
      {"cm_memory_control", aie2ps::cm_memory_control},
      {"cm_performance_control0", aie2ps::cm_performance_control0},
      {"cm_performance_control1", aie2ps::cm_performance_control1},
      {"cm_performance_control2", aie2ps::cm_performance_control2},
      {"cm_performance_counter0", aie2ps::cm_performance_counter0},
      {"cm_performance_counter1", aie2ps::cm_performance_counter1},
      {"cm_performance_counter2", aie2ps::cm_performance_counter2},
      {"cm_performance_counter3", aie2ps::cm_performance_counter3},
      {"cm_performance_counter0_event_value", aie2ps::cm_performance_counter0_event_value},
      {"cm_performance_counter1_event_value", aie2ps::cm_performance_counter1_event_value},
      {"cm_performance_counter2_event_value", aie2ps::cm_performance_counter2_event_value},
      {"cm_performance_counter3_event_value", aie2ps::cm_performance_counter3_event_value},
      {"cm_core_control", aie2ps::cm_core_control},
      {"cm_core_status", aie2ps::cm_core_status},
      {"cm_enable_events", aie2ps::cm_enable_events},
      {"cm_reset_event", aie2ps::cm_reset_event},
      {"cm_debug_control0", aie2ps::cm_debug_control0},
      {"cm_debug_control1", aie2ps::cm_debug_control1},
      {"cm_debug_control2", aie2ps::cm_debug_control2},
      {"cm_debug_status", aie2ps::cm_debug_status},
      {"cm_pc_event0", aie2ps::cm_pc_event0},
      {"cm_pc_event1", aie2ps::cm_pc_event1},
      {"cm_pc_event2", aie2ps::cm_pc_event2},
      {"cm_pc_event3", aie2ps::cm_pc_event3},
      {"cm_error_halt_control", aie2ps::cm_error_halt_control},
      {"cm_error_halt_event", aie2ps::cm_error_halt_event},
      {"cm_core_processor_bus", aie2ps::cm_core_processor_bus},
      {"cm_ecc_scrubbing_event", aie2ps::cm_ecc_scrubbing_event},
      {"cm_stream_switch_master_config_aie_core0", aie2ps::cm_stream_switch_master_config_aie_core0},
      {"cm_stream_switch_master_config_dma0", aie2ps::cm_stream_switch_master_config_dma0},
      {"cm_stream_switch_master_config_dma1", aie2ps::cm_stream_switch_master_config_dma1},
      {"cm_stream_switch_master_config_tile_ctrl", aie2ps::cm_stream_switch_master_config_tile_ctrl},
      {"cm_stream_switch_master_config_fifo0", aie2ps::cm_stream_switch_master_config_fifo0},
      {"cm_stream_switch_master_config_south0", aie2ps::cm_stream_switch_master_config_south0},
      {"cm_stream_switch_master_config_south1", aie2ps::cm_stream_switch_master_config_south1},
      {"cm_stream_switch_master_config_south2", aie2ps::cm_stream_switch_master_config_south2},
      {"cm_stream_switch_master_config_south3", aie2ps::cm_stream_switch_master_config_south3},
      {"cm_stream_switch_master_config_west0", aie2ps::cm_stream_switch_master_config_west0},
      {"cm_stream_switch_master_config_west1", aie2ps::cm_stream_switch_master_config_west1},
      {"cm_stream_switch_master_config_west2", aie2ps::cm_stream_switch_master_config_west2},
      {"cm_stream_switch_master_config_west3", aie2ps::cm_stream_switch_master_config_west3},
      {"cm_stream_switch_master_config_north0", aie2ps::cm_stream_switch_master_config_north0},
      {"cm_stream_switch_master_config_north1", aie2ps::cm_stream_switch_master_config_north1},
      {"cm_stream_switch_master_config_north2", aie2ps::cm_stream_switch_master_config_north2},
      {"cm_stream_switch_master_config_north3", aie2ps::cm_stream_switch_master_config_north3},
      {"cm_stream_switch_master_config_north4", aie2ps::cm_stream_switch_master_config_north4},
      {"cm_stream_switch_master_config_north5", aie2ps::cm_stream_switch_master_config_north5},
      {"cm_stream_switch_master_config_east0", aie2ps::cm_stream_switch_master_config_east0},
      {"cm_stream_switch_master_config_east1", aie2ps::cm_stream_switch_master_config_east1},
      {"cm_stream_switch_master_config_east2", aie2ps::cm_stream_switch_master_config_east2},
      {"cm_stream_switch_master_config_east3", aie2ps::cm_stream_switch_master_config_east3},
      {"cm_stream_switch_slave_config_aie_core0", aie2ps::cm_stream_switch_slave_config_aie_core0},
      {"cm_stream_switch_slave_config_dma_0", aie2ps::cm_stream_switch_slave_config_dma_0},
      {"cm_stream_switch_slave_config_dma_1", aie2ps::cm_stream_switch_slave_config_dma_1},
      {"cm_stream_switch_slave_config_tile_ctrl", aie2ps::cm_stream_switch_slave_config_tile_ctrl},
      {"cm_stream_switch_slave_config_fifo_0", aie2ps::cm_stream_switch_slave_config_fifo_0},
      {"cm_stream_switch_slave_config_south_0", aie2ps::cm_stream_switch_slave_config_south_0},
      {"cm_stream_switch_slave_config_south_1", aie2ps::cm_stream_switch_slave_config_south_1},
      {"cm_stream_switch_slave_config_south_2", aie2ps::cm_stream_switch_slave_config_south_2},
      {"cm_stream_switch_slave_config_south_3", aie2ps::cm_stream_switch_slave_config_south_3},
      {"cm_stream_switch_slave_config_south_4", aie2ps::cm_stream_switch_slave_config_south_4},
      {"cm_stream_switch_slave_config_south_5", aie2ps::cm_stream_switch_slave_config_south_5},
      {"cm_stream_switch_slave_config_west_0", aie2ps::cm_stream_switch_slave_config_west_0},
      {"cm_stream_switch_slave_config_west_1", aie2ps::cm_stream_switch_slave_config_west_1},
      {"cm_stream_switch_slave_config_west_2", aie2ps::cm_stream_switch_slave_config_west_2},
      {"cm_stream_switch_slave_config_west_3", aie2ps::cm_stream_switch_slave_config_west_3},
      {"cm_stream_switch_slave_config_north_0", aie2ps::cm_stream_switch_slave_config_north_0},
      {"cm_stream_switch_slave_config_north_1", aie2ps::cm_stream_switch_slave_config_north_1},
      {"cm_stream_switch_slave_config_north_2", aie2ps::cm_stream_switch_slave_config_north_2},
      {"cm_stream_switch_slave_config_north_3", aie2ps::cm_stream_switch_slave_config_north_3},
      {"cm_stream_switch_slave_config_east_0", aie2ps::cm_stream_switch_slave_config_east_0},
      {"cm_stream_switch_slave_config_east_1", aie2ps::cm_stream_switch_slave_config_east_1},
      {"cm_stream_switch_slave_config_east_2", aie2ps::cm_stream_switch_slave_config_east_2},
      {"cm_stream_switch_slave_config_east_3", aie2ps::cm_stream_switch_slave_config_east_3},
      {"cm_stream_switch_slave_config_aie_trace", aie2ps::cm_stream_switch_slave_config_aie_trace},
      {"cm_stream_switch_slave_config_mem_trace", aie2ps::cm_stream_switch_slave_config_mem_trace},
      {"cm_stream_switch_slave_aie_core0_slot0", aie2ps::cm_stream_switch_slave_aie_core0_slot0},
      {"cm_stream_switch_slave_aie_core0_slot1", aie2ps::cm_stream_switch_slave_aie_core0_slot1},
      {"cm_stream_switch_slave_aie_core0_slot2", aie2ps::cm_stream_switch_slave_aie_core0_slot2},
      {"cm_stream_switch_slave_aie_core0_slot3", aie2ps::cm_stream_switch_slave_aie_core0_slot3},
      {"cm_stream_switch_slave_dma_0_slot0", aie2ps::cm_stream_switch_slave_dma_0_slot0},
      {"cm_stream_switch_slave_dma_0_slot1", aie2ps::cm_stream_switch_slave_dma_0_slot1},
      {"cm_stream_switch_slave_dma_0_slot2", aie2ps::cm_stream_switch_slave_dma_0_slot2},
      {"cm_stream_switch_slave_dma_0_slot3", aie2ps::cm_stream_switch_slave_dma_0_slot3},
      {"cm_stream_switch_slave_dma_1_slot0", aie2ps::cm_stream_switch_slave_dma_1_slot0},
      {"cm_stream_switch_slave_dma_1_slot1", aie2ps::cm_stream_switch_slave_dma_1_slot1},
      {"cm_stream_switch_slave_dma_1_slot2", aie2ps::cm_stream_switch_slave_dma_1_slot2},
      {"cm_stream_switch_slave_dma_1_slot3", aie2ps::cm_stream_switch_slave_dma_1_slot3},
      {"cm_stream_switch_slave_tile_ctrl_slot0", aie2ps::cm_stream_switch_slave_tile_ctrl_slot0},
      {"cm_stream_switch_slave_tile_ctrl_slot1", aie2ps::cm_stream_switch_slave_tile_ctrl_slot1},
      {"cm_stream_switch_slave_tile_ctrl_slot2", aie2ps::cm_stream_switch_slave_tile_ctrl_slot2},
      {"cm_stream_switch_slave_tile_ctrl_slot3", aie2ps::cm_stream_switch_slave_tile_ctrl_slot3},
      {"cm_stream_switch_slave_fifo_0_slot0", aie2ps::cm_stream_switch_slave_fifo_0_slot0},
      {"cm_stream_switch_slave_fifo_0_slot1", aie2ps::cm_stream_switch_slave_fifo_0_slot1},
      {"cm_stream_switch_slave_fifo_0_slot2", aie2ps::cm_stream_switch_slave_fifo_0_slot2},
      {"cm_stream_switch_slave_fifo_0_slot3", aie2ps::cm_stream_switch_slave_fifo_0_slot3},
      {"cm_stream_switch_slave_south_0_slot0", aie2ps::cm_stream_switch_slave_south_0_slot0},
      {"cm_stream_switch_slave_south_0_slot1", aie2ps::cm_stream_switch_slave_south_0_slot1},
      {"cm_stream_switch_slave_south_0_slot2", aie2ps::cm_stream_switch_slave_south_0_slot2},
      {"cm_stream_switch_slave_south_0_slot3", aie2ps::cm_stream_switch_slave_south_0_slot3},
      {"cm_stream_switch_slave_south_1_slot0", aie2ps::cm_stream_switch_slave_south_1_slot0},
      {"cm_stream_switch_slave_south_1_slot1", aie2ps::cm_stream_switch_slave_south_1_slot1},
      {"cm_stream_switch_slave_south_1_slot2", aie2ps::cm_stream_switch_slave_south_1_slot2},
      {"cm_stream_switch_slave_south_1_slot3", aie2ps::cm_stream_switch_slave_south_1_slot3},
      {"cm_stream_switch_slave_south_2_slot0", aie2ps::cm_stream_switch_slave_south_2_slot0},
      {"cm_stream_switch_slave_south_2_slot1", aie2ps::cm_stream_switch_slave_south_2_slot1},
      {"cm_stream_switch_slave_south_2_slot2", aie2ps::cm_stream_switch_slave_south_2_slot2},
      {"cm_stream_switch_slave_south_2_slot3", aie2ps::cm_stream_switch_slave_south_2_slot3},
      {"cm_stream_switch_slave_south_3_slot0", aie2ps::cm_stream_switch_slave_south_3_slot0},
      {"cm_stream_switch_slave_south_3_slot1", aie2ps::cm_stream_switch_slave_south_3_slot1},
      {"cm_stream_switch_slave_south_3_slot2", aie2ps::cm_stream_switch_slave_south_3_slot2},
      {"cm_stream_switch_slave_south_3_slot3", aie2ps::cm_stream_switch_slave_south_3_slot3},
      {"cm_stream_switch_slave_south_4_slot0", aie2ps::cm_stream_switch_slave_south_4_slot0},
      {"cm_stream_switch_slave_south_4_slot1", aie2ps::cm_stream_switch_slave_south_4_slot1},
      {"cm_stream_switch_slave_south_4_slot2", aie2ps::cm_stream_switch_slave_south_4_slot2},
      {"cm_stream_switch_slave_south_4_slot3", aie2ps::cm_stream_switch_slave_south_4_slot3},
      {"cm_stream_switch_slave_south_5_slot0", aie2ps::cm_stream_switch_slave_south_5_slot0},
      {"cm_stream_switch_slave_south_5_slot1", aie2ps::cm_stream_switch_slave_south_5_slot1},
      {"cm_stream_switch_slave_south_5_slot2", aie2ps::cm_stream_switch_slave_south_5_slot2},
      {"cm_stream_switch_slave_south_5_slot3", aie2ps::cm_stream_switch_slave_south_5_slot3},
      {"cm_stream_switch_slave_west_0_slot0", aie2ps::cm_stream_switch_slave_west_0_slot0},
      {"cm_stream_switch_slave_west_0_slot1", aie2ps::cm_stream_switch_slave_west_0_slot1},
      {"cm_stream_switch_slave_west_0_slot2", aie2ps::cm_stream_switch_slave_west_0_slot2},
      {"cm_stream_switch_slave_west_0_slot3", aie2ps::cm_stream_switch_slave_west_0_slot3},
      {"cm_stream_switch_slave_west_1_slot0", aie2ps::cm_stream_switch_slave_west_1_slot0},
      {"cm_stream_switch_slave_west_1_slot1", aie2ps::cm_stream_switch_slave_west_1_slot1},
      {"cm_stream_switch_slave_west_1_slot2", aie2ps::cm_stream_switch_slave_west_1_slot2},
      {"cm_stream_switch_slave_west_1_slot3", aie2ps::cm_stream_switch_slave_west_1_slot3},
      {"cm_stream_switch_slave_west_2_slot0", aie2ps::cm_stream_switch_slave_west_2_slot0},
      {"cm_stream_switch_slave_west_2_slot1", aie2ps::cm_stream_switch_slave_west_2_slot1},
      {"cm_stream_switch_slave_west_2_slot2", aie2ps::cm_stream_switch_slave_west_2_slot2},
      {"cm_stream_switch_slave_west_2_slot3", aie2ps::cm_stream_switch_slave_west_2_slot3},
      {"cm_stream_switch_slave_west_3_slot0", aie2ps::cm_stream_switch_slave_west_3_slot0},
      {"cm_stream_switch_slave_west_3_slot1", aie2ps::cm_stream_switch_slave_west_3_slot1},
      {"cm_stream_switch_slave_west_3_slot2", aie2ps::cm_stream_switch_slave_west_3_slot2},
      {"cm_stream_switch_slave_west_3_slot3", aie2ps::cm_stream_switch_slave_west_3_slot3},
      {"cm_stream_switch_slave_north_0_slot0", aie2ps::cm_stream_switch_slave_north_0_slot0},
      {"cm_stream_switch_slave_north_0_slot1", aie2ps::cm_stream_switch_slave_north_0_slot1},
      {"cm_stream_switch_slave_north_0_slot2", aie2ps::cm_stream_switch_slave_north_0_slot2},
      {"cm_stream_switch_slave_north_0_slot3", aie2ps::cm_stream_switch_slave_north_0_slot3},
      {"cm_stream_switch_slave_north_1_slot0", aie2ps::cm_stream_switch_slave_north_1_slot0},
      {"cm_stream_switch_slave_north_1_slot1", aie2ps::cm_stream_switch_slave_north_1_slot1},
      {"cm_stream_switch_slave_north_1_slot2", aie2ps::cm_stream_switch_slave_north_1_slot2},
      {"cm_stream_switch_slave_north_1_slot3", aie2ps::cm_stream_switch_slave_north_1_slot3},
      {"cm_stream_switch_slave_north_2_slot0", aie2ps::cm_stream_switch_slave_north_2_slot0},
      {"cm_stream_switch_slave_north_2_slot1", aie2ps::cm_stream_switch_slave_north_2_slot1},
      {"cm_stream_switch_slave_north_2_slot2", aie2ps::cm_stream_switch_slave_north_2_slot2},
      {"cm_stream_switch_slave_north_2_slot3", aie2ps::cm_stream_switch_slave_north_2_slot3},
      {"cm_stream_switch_slave_north_3_slot0", aie2ps::cm_stream_switch_slave_north_3_slot0},
      {"cm_stream_switch_slave_north_3_slot1", aie2ps::cm_stream_switch_slave_north_3_slot1},
      {"cm_stream_switch_slave_north_3_slot2", aie2ps::cm_stream_switch_slave_north_3_slot2},
      {"cm_stream_switch_slave_north_3_slot3", aie2ps::cm_stream_switch_slave_north_3_slot3},
      {"cm_stream_switch_slave_east_0_slot0", aie2ps::cm_stream_switch_slave_east_0_slot0},
      {"cm_stream_switch_slave_east_0_slot1", aie2ps::cm_stream_switch_slave_east_0_slot1},
      {"cm_stream_switch_slave_east_0_slot2", aie2ps::cm_stream_switch_slave_east_0_slot2},
      {"cm_stream_switch_slave_east_0_slot3", aie2ps::cm_stream_switch_slave_east_0_slot3},
      {"cm_stream_switch_slave_east_1_slot0", aie2ps::cm_stream_switch_slave_east_1_slot0},
      {"cm_stream_switch_slave_east_1_slot1", aie2ps::cm_stream_switch_slave_east_1_slot1},
      {"cm_stream_switch_slave_east_1_slot2", aie2ps::cm_stream_switch_slave_east_1_slot2},
      {"cm_stream_switch_slave_east_1_slot3", aie2ps::cm_stream_switch_slave_east_1_slot3},
      {"cm_stream_switch_slave_east_2_slot0", aie2ps::cm_stream_switch_slave_east_2_slot0},
      {"cm_stream_switch_slave_east_2_slot1", aie2ps::cm_stream_switch_slave_east_2_slot1},
      {"cm_stream_switch_slave_east_2_slot2", aie2ps::cm_stream_switch_slave_east_2_slot2},
      {"cm_stream_switch_slave_east_2_slot3", aie2ps::cm_stream_switch_slave_east_2_slot3},
      {"cm_stream_switch_slave_east_3_slot0", aie2ps::cm_stream_switch_slave_east_3_slot0},
      {"cm_stream_switch_slave_east_3_slot1", aie2ps::cm_stream_switch_slave_east_3_slot1},
      {"cm_stream_switch_slave_east_3_slot2", aie2ps::cm_stream_switch_slave_east_3_slot2},
      {"cm_stream_switch_slave_east_3_slot3", aie2ps::cm_stream_switch_slave_east_3_slot3},
      {"cm_stream_switch_slave_aie_trace_slot0", aie2ps::cm_stream_switch_slave_aie_trace_slot0},
      {"cm_stream_switch_slave_aie_trace_slot1", aie2ps::cm_stream_switch_slave_aie_trace_slot1},
      {"cm_stream_switch_slave_aie_trace_slot2", aie2ps::cm_stream_switch_slave_aie_trace_slot2},
      {"cm_stream_switch_slave_aie_trace_slot3", aie2ps::cm_stream_switch_slave_aie_trace_slot3},
      {"cm_stream_switch_slave_mem_trace_slot0", aie2ps::cm_stream_switch_slave_mem_trace_slot0},
      {"cm_stream_switch_slave_mem_trace_slot1", aie2ps::cm_stream_switch_slave_mem_trace_slot1},
      {"cm_stream_switch_slave_mem_trace_slot2", aie2ps::cm_stream_switch_slave_mem_trace_slot2},
      {"cm_stream_switch_slave_mem_trace_slot3", aie2ps::cm_stream_switch_slave_mem_trace_slot3},
      {"cm_stream_switch_deterministic_merge_arb0_slave0_1", aie2ps::cm_stream_switch_deterministic_merge_arb0_slave0_1},
      {"cm_stream_switch_deterministic_merge_arb0_slave2_3", aie2ps::cm_stream_switch_deterministic_merge_arb0_slave2_3},
      {"cm_stream_switch_deterministic_merge_arb0_ctrl", aie2ps::cm_stream_switch_deterministic_merge_arb0_ctrl},
      {"cm_stream_switch_deterministic_merge_arb1_slave0_1", aie2ps::cm_stream_switch_deterministic_merge_arb1_slave0_1},
      {"cm_stream_switch_deterministic_merge_arb1_slave2_3", aie2ps::cm_stream_switch_deterministic_merge_arb1_slave2_3},
      {"cm_stream_switch_deterministic_merge_arb1_ctrl", aie2ps::cm_stream_switch_deterministic_merge_arb1_ctrl},
      {"cm_stream_switch_event_port_selection_0", aie2ps::cm_stream_switch_event_port_selection_0},
      {"cm_stream_switch_event_port_selection_1", aie2ps::cm_stream_switch_event_port_selection_1},
      {"cm_stream_switch_parity_status", aie2ps::cm_stream_switch_parity_status},
      {"cm_stream_switch_parity_injection", aie2ps::cm_stream_switch_parity_injection},
      {"cm_tile_control_packet_handler_status", aie2ps::cm_tile_control_packet_handler_status},
      {"cm_stream_switch_adaptive_clock_gate_status", aie2ps::cm_stream_switch_adaptive_clock_gate_status},
      {"cm_stream_switch_adaptive_clock_gate_abort_period", aie2ps::cm_stream_switch_adaptive_clock_gate_abort_period},
      {"cm_module_clock_control", aie2ps::cm_module_clock_control},
      {"cm_tile_control", aie2ps::cm_tile_control},
      {"mm_performance_control0", aie2ps::mm_performance_control0},
      {"mm_performance_control1", aie2ps::mm_performance_control1},
      {"mm_performance_control2", aie2ps::mm_performance_control2},
      {"mm_performance_control3", aie2ps::mm_performance_control3},
      {"mm_performance_counter0", aie2ps::mm_performance_counter0},
      {"mm_performance_counter1", aie2ps::mm_performance_counter1},
      {"mm_performance_counter2", aie2ps::mm_performance_counter2},
      {"mm_performance_counter3", aie2ps::mm_performance_counter3},
      {"mm_performance_counter0_event_value", aie2ps::mm_performance_counter0_event_value},
      {"mm_performance_counter1_event_value", aie2ps::mm_performance_counter1_event_value},
      {"mm_timer_control", aie2ps::mm_timer_control},
      {"mm_event_generate", aie2ps::mm_event_generate},
      {"mm_event_broadcast0", aie2ps::mm_event_broadcast0},
      {"mm_event_broadcast1", aie2ps::mm_event_broadcast1},
      {"mm_event_broadcast2", aie2ps::mm_event_broadcast2},
      {"mm_event_broadcast3", aie2ps::mm_event_broadcast3},
      {"mm_event_broadcast4", aie2ps::mm_event_broadcast4},
      {"mm_event_broadcast5", aie2ps::mm_event_broadcast5},
      {"mm_event_broadcast6", aie2ps::mm_event_broadcast6},
      {"mm_event_broadcast7", aie2ps::mm_event_broadcast7},
      {"mm_event_broadcast8", aie2ps::mm_event_broadcast8},
      {"mm_event_broadcast9", aie2ps::mm_event_broadcast9},
      {"mm_event_broadcast10", aie2ps::mm_event_broadcast10},
      {"mm_event_broadcast11", aie2ps::mm_event_broadcast11},
      {"mm_event_broadcast12", aie2ps::mm_event_broadcast12},
      {"mm_event_broadcast13", aie2ps::mm_event_broadcast13},
      {"mm_event_broadcast14", aie2ps::mm_event_broadcast14},
      {"mm_event_broadcast15", aie2ps::mm_event_broadcast15},
      {"mm_event_broadcast_block_south_set", aie2ps::mm_event_broadcast_block_south_set},
      {"mm_event_broadcast_block_south_clr", aie2ps::mm_event_broadcast_block_south_clr},
      {"mm_event_broadcast_block_south_value", aie2ps::mm_event_broadcast_block_south_value},
      {"mm_event_broadcast_block_west_set", aie2ps::mm_event_broadcast_block_west_set},
      {"mm_event_broadcast_block_west_clr", aie2ps::mm_event_broadcast_block_west_clr},
      {"mm_event_broadcast_block_west_value", aie2ps::mm_event_broadcast_block_west_value},
      {"mm_event_broadcast_block_north_set", aie2ps::mm_event_broadcast_block_north_set},
      {"mm_event_broadcast_block_north_clr", aie2ps::mm_event_broadcast_block_north_clr},
      {"mm_event_broadcast_block_north_value", aie2ps::mm_event_broadcast_block_north_value},
      {"mm_event_broadcast_block_east_set", aie2ps::mm_event_broadcast_block_east_set},
      {"mm_event_broadcast_block_east_clr", aie2ps::mm_event_broadcast_block_east_clr},
      {"mm_event_broadcast_block_east_value", aie2ps::mm_event_broadcast_block_east_value},
      {"mm_trace_control0", aie2ps::mm_trace_control0},
      {"mm_trace_control1", aie2ps::mm_trace_control1},
      {"mm_trace_status", aie2ps::mm_trace_status},
      {"mm_trace_event0", aie2ps::mm_trace_event0},
      {"mm_trace_event1", aie2ps::mm_trace_event1},
      {"mm_timer_trig_event_low_value", aie2ps::mm_timer_trig_event_low_value},
      {"mm_timer_trig_event_high_value", aie2ps::mm_timer_trig_event_high_value},
      {"mm_timer_low", aie2ps::mm_timer_low},
      {"mm_timer_high", aie2ps::mm_timer_high},
      {"mm_watchpoint0", aie2ps::mm_watchpoint0},
      {"mm_watchpoint1", aie2ps::mm_watchpoint1},
      {"mm_event_status0", aie2ps::mm_event_status0},
      {"mm_event_status1", aie2ps::mm_event_status1},
      {"mm_event_status2", aie2ps::mm_event_status2},
      {"mm_event_status3", aie2ps::mm_event_status3},
      {"mm_combo_event_inputs", aie2ps::mm_combo_event_inputs},
      {"mm_combo_event_control", aie2ps::mm_combo_event_control},
      {"mm_edge_detection_event_control", aie2ps::mm_edge_detection_event_control},
      {"mm_event_group_0_enable", aie2ps::mm_event_group_0_enable},
      {"mm_event_group_watchpoint_enable", aie2ps::mm_event_group_watchpoint_enable},
      {"mm_event_group_dma_enable", aie2ps::mm_event_group_dma_enable},
      {"mm_event_group_lock_enable", aie2ps::mm_event_group_lock_enable},
      {"mm_event_group_memory_conflict_enable", aie2ps::mm_event_group_memory_conflict_enable},
      {"mm_event_group_error_enable", aie2ps::mm_event_group_error_enable},
      {"mm_event_group_broadcast_enable", aie2ps::mm_event_group_broadcast_enable},
      {"mm_event_group_user_event_enable", aie2ps::mm_event_group_user_event_enable},
      {"mm_memory_control", aie2ps::mm_memory_control},
      {"mm_dma_bd0_0", aie2ps::mm_dma_bd0_0},
      {"mm_dma_bd0_1", aie2ps::mm_dma_bd0_1},
      {"mm_dma_bd0_2", aie2ps::mm_dma_bd0_2},
      {"mm_dma_bd0_3", aie2ps::mm_dma_bd0_3},
      {"mm_dma_bd0_4", aie2ps::mm_dma_bd0_4},
      {"mm_dma_bd0_5", aie2ps::mm_dma_bd0_5},
      {"mm_dma_bd1_0", aie2ps::mm_dma_bd1_0},
      {"mm_dma_bd1_1", aie2ps::mm_dma_bd1_1},
      {"mm_dma_bd1_2", aie2ps::mm_dma_bd1_2},
      {"mm_dma_bd1_3", aie2ps::mm_dma_bd1_3},
      {"mm_dma_bd1_4", aie2ps::mm_dma_bd1_4},
      {"mm_dma_bd1_5", aie2ps::mm_dma_bd1_5},
      {"mm_dma_bd2_0", aie2ps::mm_dma_bd2_0},
      {"mm_dma_bd2_1", aie2ps::mm_dma_bd2_1},
      {"mm_dma_bd2_2", aie2ps::mm_dma_bd2_2},
      {"mm_dma_bd2_3", aie2ps::mm_dma_bd2_3},
      {"mm_dma_bd2_4", aie2ps::mm_dma_bd2_4},
      {"mm_dma_bd2_5", aie2ps::mm_dma_bd2_5},
      {"mm_dma_bd3_0", aie2ps::mm_dma_bd3_0},
      {"mm_dma_bd3_1", aie2ps::mm_dma_bd3_1},
      {"mm_dma_bd3_2", aie2ps::mm_dma_bd3_2},
      {"mm_dma_bd3_3", aie2ps::mm_dma_bd3_3},
      {"mm_dma_bd3_4", aie2ps::mm_dma_bd3_4},
      {"mm_dma_bd3_5", aie2ps::mm_dma_bd3_5},
      {"mm_dma_bd4_0", aie2ps::mm_dma_bd4_0},
      {"mm_dma_bd4_1", aie2ps::mm_dma_bd4_1},
      {"mm_dma_bd4_2", aie2ps::mm_dma_bd4_2},
      {"mm_dma_bd4_3", aie2ps::mm_dma_bd4_3},
      {"mm_dma_bd4_4", aie2ps::mm_dma_bd4_4},
      {"mm_dma_bd4_5", aie2ps::mm_dma_bd4_5},
      {"mm_dma_bd5_0", aie2ps::mm_dma_bd5_0},
      {"mm_dma_bd5_1", aie2ps::mm_dma_bd5_1},
      {"mm_dma_bd5_2", aie2ps::mm_dma_bd5_2},
      {"mm_dma_bd5_3", aie2ps::mm_dma_bd5_3},
      {"mm_dma_bd5_4", aie2ps::mm_dma_bd5_4},
      {"mm_dma_bd5_5", aie2ps::mm_dma_bd5_5},
      {"mm_dma_bd6_0", aie2ps::mm_dma_bd6_0},
      {"mm_dma_bd6_1", aie2ps::mm_dma_bd6_1},
      {"mm_dma_bd6_2", aie2ps::mm_dma_bd6_2},
      {"mm_dma_bd6_3", aie2ps::mm_dma_bd6_3},
      {"mm_dma_bd6_4", aie2ps::mm_dma_bd6_4},
      {"mm_dma_bd6_5", aie2ps::mm_dma_bd6_5},
      {"mm_dma_bd7_0", aie2ps::mm_dma_bd7_0},
      {"mm_dma_bd7_1", aie2ps::mm_dma_bd7_1},
      {"mm_dma_bd7_2", aie2ps::mm_dma_bd7_2},
      {"mm_dma_bd7_3", aie2ps::mm_dma_bd7_3},
      {"mm_dma_bd7_4", aie2ps::mm_dma_bd7_4},
      {"mm_dma_bd7_5", aie2ps::mm_dma_bd7_5},
      {"mm_dma_bd8_0", aie2ps::mm_dma_bd8_0},
      {"mm_dma_bd8_1", aie2ps::mm_dma_bd8_1},
      {"mm_dma_bd8_2", aie2ps::mm_dma_bd8_2},
      {"mm_dma_bd8_3", aie2ps::mm_dma_bd8_3},
      {"mm_dma_bd8_4", aie2ps::mm_dma_bd8_4},
      {"mm_dma_bd8_5", aie2ps::mm_dma_bd8_5},
      {"mm_dma_bd9_0", aie2ps::mm_dma_bd9_0},
      {"mm_dma_bd9_1", aie2ps::mm_dma_bd9_1},
      {"mm_dma_bd9_2", aie2ps::mm_dma_bd9_2},
      {"mm_dma_bd9_3", aie2ps::mm_dma_bd9_3},
      {"mm_dma_bd9_4", aie2ps::mm_dma_bd9_4},
      {"mm_dma_bd9_5", aie2ps::mm_dma_bd9_5},
      {"mm_dma_bd10_0", aie2ps::mm_dma_bd10_0},
      {"mm_dma_bd10_1", aie2ps::mm_dma_bd10_1},
      {"mm_dma_bd10_2", aie2ps::mm_dma_bd10_2},
      {"mm_dma_bd10_3", aie2ps::mm_dma_bd10_3},
      {"mm_dma_bd10_4", aie2ps::mm_dma_bd10_4},
      {"mm_dma_bd10_5", aie2ps::mm_dma_bd10_5},
      {"mm_dma_bd11_0", aie2ps::mm_dma_bd11_0},
      {"mm_dma_bd11_1", aie2ps::mm_dma_bd11_1},
      {"mm_dma_bd11_2", aie2ps::mm_dma_bd11_2},
      {"mm_dma_bd11_3", aie2ps::mm_dma_bd11_3},
      {"mm_dma_bd11_4", aie2ps::mm_dma_bd11_4},
      {"mm_dma_bd11_5", aie2ps::mm_dma_bd11_5},
      {"mm_dma_bd12_0", aie2ps::mm_dma_bd12_0},
      {"mm_dma_bd12_1", aie2ps::mm_dma_bd12_1},
      {"mm_dma_bd12_2", aie2ps::mm_dma_bd12_2},
      {"mm_dma_bd12_3", aie2ps::mm_dma_bd12_3},
      {"mm_dma_bd12_4", aie2ps::mm_dma_bd12_4},
      {"mm_dma_bd12_5", aie2ps::mm_dma_bd12_5},
      {"mm_dma_bd13_0", aie2ps::mm_dma_bd13_0},
      {"mm_dma_bd13_1", aie2ps::mm_dma_bd13_1},
      {"mm_dma_bd13_2", aie2ps::mm_dma_bd13_2},
      {"mm_dma_bd13_3", aie2ps::mm_dma_bd13_3},
      {"mm_dma_bd13_4", aie2ps::mm_dma_bd13_4},
      {"mm_dma_bd13_5", aie2ps::mm_dma_bd13_5},
      {"mm_dma_bd14_0", aie2ps::mm_dma_bd14_0},
      {"mm_dma_bd14_1", aie2ps::mm_dma_bd14_1},
      {"mm_dma_bd14_2", aie2ps::mm_dma_bd14_2},
      {"mm_dma_bd14_3", aie2ps::mm_dma_bd14_3},
      {"mm_dma_bd14_4", aie2ps::mm_dma_bd14_4},
      {"mm_dma_bd14_5", aie2ps::mm_dma_bd14_5},
      {"mm_dma_bd15_0", aie2ps::mm_dma_bd15_0},
      {"mm_dma_bd15_1", aie2ps::mm_dma_bd15_1},
      {"mm_dma_bd15_2", aie2ps::mm_dma_bd15_2},
      {"mm_dma_bd15_3", aie2ps::mm_dma_bd15_3},
      {"mm_dma_bd15_4", aie2ps::mm_dma_bd15_4},
      {"mm_dma_bd15_5", aie2ps::mm_dma_bd15_5},
      {"mm_dma_s2mm_0_ctrl", aie2ps::mm_dma_s2mm_0_ctrl},
      {"mm_dma_s2mm_0_start_queue", aie2ps::mm_dma_s2mm_0_start_queue},
      {"mm_dma_s2mm_1_ctrl", aie2ps::mm_dma_s2mm_1_ctrl},
      {"mm_dma_s2mm_1_start_queue", aie2ps::mm_dma_s2mm_1_start_queue},
      {"mm_dma_mm2s_0_ctrl", aie2ps::mm_dma_mm2s_0_ctrl},
      {"mm_dma_mm2s_0_start_queue", aie2ps::mm_dma_mm2s_0_start_queue},
      {"mm_dma_mm2s_1_ctrl", aie2ps::mm_dma_mm2s_1_ctrl},
      {"mm_dma_mm2s_1_start_queue", aie2ps::mm_dma_mm2s_1_start_queue},
      {"mm_dma_s2mm_status_0", aie2ps::mm_dma_s2mm_status_0},
      {"mm_dma_s2mm_status_1", aie2ps::mm_dma_s2mm_status_1},
      {"mm_dma_mm2s_status_0", aie2ps::mm_dma_mm2s_status_0},
      {"mm_dma_mm2s_status_1", aie2ps::mm_dma_mm2s_status_1},
      {"mm_dma_s2mm_current_write_count_0", aie2ps::mm_dma_s2mm_current_write_count_0},
      {"mm_dma_s2mm_current_write_count_1", aie2ps::mm_dma_s2mm_current_write_count_1},
      {"mm_dma_s2mm_fot_count_fifo_pop_0", aie2ps::mm_dma_s2mm_fot_count_fifo_pop_0},
      {"mm_dma_s2mm_fot_count_fifo_pop_1", aie2ps::mm_dma_s2mm_fot_count_fifo_pop_1},
      {"mm_lock0_value", aie2ps::mm_lock0_value},
      {"mm_lock1_value", aie2ps::mm_lock1_value},
      {"mm_lock2_value", aie2ps::mm_lock2_value},
      {"mm_lock3_value", aie2ps::mm_lock3_value},
      {"mm_lock4_value", aie2ps::mm_lock4_value},
      {"mm_lock5_value", aie2ps::mm_lock5_value},
      {"mm_lock6_value", aie2ps::mm_lock6_value},
      {"mm_lock7_value", aie2ps::mm_lock7_value},
      {"mm_lock8_value", aie2ps::mm_lock8_value},
      {"mm_lock9_value", aie2ps::mm_lock9_value},
      {"mm_lock10_value", aie2ps::mm_lock10_value},
      {"mm_lock11_value", aie2ps::mm_lock11_value},
      {"mm_lock12_value", aie2ps::mm_lock12_value},
      {"mm_lock13_value", aie2ps::mm_lock13_value},
      {"mm_lock14_value", aie2ps::mm_lock14_value},
      {"mm_lock15_value", aie2ps::mm_lock15_value},
      {"mm_locks_event_selection_0", aie2ps::mm_locks_event_selection_0},
      {"mm_locks_event_selection_1", aie2ps::mm_locks_event_selection_1},
      {"mm_locks_event_selection_2", aie2ps::mm_locks_event_selection_2},
      {"mm_locks_event_selection_3", aie2ps::mm_locks_event_selection_3},
      {"mm_locks_event_selection_4", aie2ps::mm_locks_event_selection_4},
      {"mm_locks_event_selection_5", aie2ps::mm_locks_event_selection_5},
      {"mm_locks_event_selection_6", aie2ps::mm_locks_event_selection_6},
      {"mm_locks_event_selection_7", aie2ps::mm_locks_event_selection_7},
      {"mm_locks_overflow", aie2ps::mm_locks_overflow},
      {"mm_locks_underflow", aie2ps::mm_locks_underflow},
      {"mm_lock_request", aie2ps::mm_lock_request},
      {"mem_performance_control0", aie2ps::mem_performance_control0},
      {"mem_performance_control1", aie2ps::mem_performance_control1},
      {"mem_performance_control2", aie2ps::mem_performance_control2},
      {"mem_performance_control3", aie2ps::mem_performance_control3},
      {"mem_performance_control4", aie2ps::mem_performance_control4},
      {"mem_performance_counter0", aie2ps::mem_performance_counter0},
      {"mem_performance_counter1", aie2ps::mem_performance_counter1},
      {"mem_performance_counter2", aie2ps::mem_performance_counter2},
      {"mem_performance_counter3", aie2ps::mem_performance_counter3},
      {"mem_performance_counter4", aie2ps::mem_performance_counter4},
      {"mem_performance_counter5", aie2ps::mem_performance_counter5},
      {"mem_performance_counter0_event_value", aie2ps::mem_performance_counter0_event_value},
      {"mem_performance_counter1_event_value", aie2ps::mem_performance_counter1_event_value},
      {"mem_performance_counter2_event_value", aie2ps::mem_performance_counter2_event_value},
      {"mem_performance_counter3_event_value", aie2ps::mem_performance_counter3_event_value},
      {"mem_timer_control", aie2ps::mem_timer_control},
      {"mem_event_generate", aie2ps::mem_event_generate},
      {"mem_event_broadcast0", aie2ps::mem_event_broadcast0},
      {"mem_event_broadcast1", aie2ps::mem_event_broadcast1},
      {"mem_event_broadcast2", aie2ps::mem_event_broadcast2},
      {"mem_event_broadcast3", aie2ps::mem_event_broadcast3},
      {"mem_event_broadcast4", aie2ps::mem_event_broadcast4},
      {"mem_event_broadcast5", aie2ps::mem_event_broadcast5},
      {"mem_event_broadcast6", aie2ps::mem_event_broadcast6},
      {"mem_event_broadcast7", aie2ps::mem_event_broadcast7},
      {"mem_event_broadcast8", aie2ps::mem_event_broadcast8},
      {"mem_event_broadcast9", aie2ps::mem_event_broadcast9},
      {"mem_event_broadcast10", aie2ps::mem_event_broadcast10},
      {"mem_event_broadcast11", aie2ps::mem_event_broadcast11},
      {"mem_event_broadcast12", aie2ps::mem_event_broadcast12},
      {"mem_event_broadcast13", aie2ps::mem_event_broadcast13},
      {"mem_event_broadcast14", aie2ps::mem_event_broadcast14},
      {"mem_event_broadcast15", aie2ps::mem_event_broadcast15},
      {"mem_event_broadcast_a_block_south_set", aie2ps::mem_event_broadcast_a_block_south_set},
      {"mem_event_broadcast_a_block_south_clr", aie2ps::mem_event_broadcast_a_block_south_clr},
      {"mem_event_broadcast_a_block_south_value", aie2ps::mem_event_broadcast_a_block_south_value},
      {"mem_event_broadcast_a_block_west_set", aie2ps::mem_event_broadcast_a_block_west_set},
      {"mem_event_broadcast_a_block_west_clr", aie2ps::mem_event_broadcast_a_block_west_clr},
      {"mem_event_broadcast_a_block_west_value", aie2ps::mem_event_broadcast_a_block_west_value},
      {"mem_event_broadcast_a_block_north_set", aie2ps::mem_event_broadcast_a_block_north_set},
      {"mem_event_broadcast_a_block_north_clr", aie2ps::mem_event_broadcast_a_block_north_clr},
      {"mem_event_broadcast_a_block_north_value", aie2ps::mem_event_broadcast_a_block_north_value},
      {"mem_event_broadcast_a_block_east_set", aie2ps::mem_event_broadcast_a_block_east_set},
      {"mem_event_broadcast_a_block_east_clr", aie2ps::mem_event_broadcast_a_block_east_clr},
      {"mem_event_broadcast_a_block_east_value", aie2ps::mem_event_broadcast_a_block_east_value},
      {"mem_event_broadcast_b_block_south_set", aie2ps::mem_event_broadcast_b_block_south_set},
      {"mem_event_broadcast_b_block_south_clr", aie2ps::mem_event_broadcast_b_block_south_clr},
      {"mem_event_broadcast_b_block_south_value", aie2ps::mem_event_broadcast_b_block_south_value},
      {"mem_event_broadcast_b_block_west_set", aie2ps::mem_event_broadcast_b_block_west_set},
      {"mem_event_broadcast_b_block_west_clr", aie2ps::mem_event_broadcast_b_block_west_clr},
      {"mem_event_broadcast_b_block_west_value", aie2ps::mem_event_broadcast_b_block_west_value},
      {"mem_event_broadcast_b_block_north_set", aie2ps::mem_event_broadcast_b_block_north_set},
      {"mem_event_broadcast_b_block_north_clr", aie2ps::mem_event_broadcast_b_block_north_clr},
      {"mem_event_broadcast_b_block_north_value", aie2ps::mem_event_broadcast_b_block_north_value},
      {"mem_event_broadcast_b_block_east_set", aie2ps::mem_event_broadcast_b_block_east_set},
      {"mem_event_broadcast_b_block_east_clr", aie2ps::mem_event_broadcast_b_block_east_clr},
      {"mem_event_broadcast_b_block_east_value", aie2ps::mem_event_broadcast_b_block_east_value},
      {"mem_trace_control0", aie2ps::mem_trace_control0},
      {"mem_trace_control1", aie2ps::mem_trace_control1},
      {"mem_trace_status", aie2ps::mem_trace_status},
      {"mem_trace_event0", aie2ps::mem_trace_event0},
      {"mem_trace_event1", aie2ps::mem_trace_event1},
      {"mem_timer_trig_event_low_value", aie2ps::mem_timer_trig_event_low_value},
      {"mem_timer_trig_event_high_value", aie2ps::mem_timer_trig_event_high_value},
      {"mem_timer_low", aie2ps::mem_timer_low},
      {"mem_timer_high", aie2ps::mem_timer_high},
      {"mem_watchpoint0", aie2ps::mem_watchpoint0},
      {"mem_watchpoint1", aie2ps::mem_watchpoint1},
      {"mem_watchpoint2", aie2ps::mem_watchpoint2},
      {"mem_watchpoint3", aie2ps::mem_watchpoint3},
      {"mem_event_status0", aie2ps::mem_event_status0},
      {"mem_event_status1", aie2ps::mem_event_status1},
      {"mem_event_status2", aie2ps::mem_event_status2},
      {"mem_event_status3", aie2ps::mem_event_status3},
      {"mem_event_status4", aie2ps::mem_event_status4},
      {"mem_event_status5", aie2ps::mem_event_status5},
      {"mem_combo_event_inputs", aie2ps::mem_combo_event_inputs},
      {"mem_combo_event_control", aie2ps::mem_combo_event_control},
      {"mem_edge_detection_event_control", aie2ps::mem_edge_detection_event_control},
      {"mem_event_group_0_enable", aie2ps::mem_event_group_0_enable},
      {"mem_event_group_watchpoint_enable", aie2ps::mem_event_group_watchpoint_enable},
      {"mem_event_group_dma_enable", aie2ps::mem_event_group_dma_enable},
      {"mem_event_group_lock_enable", aie2ps::mem_event_group_lock_enable},
      {"mem_event_group_stream_switch_enable", aie2ps::mem_event_group_stream_switch_enable},
      {"mem_event_group_memory_conflict_enable", aie2ps::mem_event_group_memory_conflict_enable},
      {"mem_event_group_error_enable", aie2ps::mem_event_group_error_enable},
      {"mem_event_group_broadcast_enable", aie2ps::mem_event_group_broadcast_enable},
      {"mem_event_group_user_event_enable", aie2ps::mem_event_group_user_event_enable},
      {"mem_memory_control", aie2ps::mem_memory_control},
      {"mem_dma_bd0_0", aie2ps::mem_dma_bd0_0},
      {"mem_dma_bd0_1", aie2ps::mem_dma_bd0_1},
      {"mem_dma_bd0_2", aie2ps::mem_dma_bd0_2},
      {"mem_dma_bd0_3", aie2ps::mem_dma_bd0_3},
      {"mem_dma_bd0_4", aie2ps::mem_dma_bd0_4},
      {"mem_dma_bd0_5", aie2ps::mem_dma_bd0_5},
      {"mem_dma_bd0_6", aie2ps::mem_dma_bd0_6},
      {"mem_dma_bd0_7", aie2ps::mem_dma_bd0_7},
      {"mem_dma_bd1_0", aie2ps::mem_dma_bd1_0},
      {"mem_dma_bd1_1", aie2ps::mem_dma_bd1_1},
      {"mem_dma_bd1_2", aie2ps::mem_dma_bd1_2},
      {"mem_dma_bd1_3", aie2ps::mem_dma_bd1_3},
      {"mem_dma_bd1_4", aie2ps::mem_dma_bd1_4},
      {"mem_dma_bd1_5", aie2ps::mem_dma_bd1_5},
      {"mem_dma_bd1_6", aie2ps::mem_dma_bd1_6},
      {"mem_dma_bd1_7", aie2ps::mem_dma_bd1_7},
      {"mem_dma_bd2_0", aie2ps::mem_dma_bd2_0},
      {"mem_dma_bd2_1", aie2ps::mem_dma_bd2_1},
      {"mem_dma_bd2_2", aie2ps::mem_dma_bd2_2},
      {"mem_dma_bd2_3", aie2ps::mem_dma_bd2_3},
      {"mem_dma_bd2_4", aie2ps::mem_dma_bd2_4},
      {"mem_dma_bd2_5", aie2ps::mem_dma_bd2_5},
      {"mem_dma_bd2_6", aie2ps::mem_dma_bd2_6},
      {"mem_dma_bd2_7", aie2ps::mem_dma_bd2_7},
      {"mem_dma_bd3_0", aie2ps::mem_dma_bd3_0},
      {"mem_dma_bd3_1", aie2ps::mem_dma_bd3_1},
      {"mem_dma_bd3_2", aie2ps::mem_dma_bd3_2},
      {"mem_dma_bd3_3", aie2ps::mem_dma_bd3_3},
      {"mem_dma_bd3_4", aie2ps::mem_dma_bd3_4},
      {"mem_dma_bd3_5", aie2ps::mem_dma_bd3_5},
      {"mem_dma_bd3_6", aie2ps::mem_dma_bd3_6},
      {"mem_dma_bd3_7", aie2ps::mem_dma_bd3_7},
      {"mem_dma_bd4_0", aie2ps::mem_dma_bd4_0},
      {"mem_dma_bd4_1", aie2ps::mem_dma_bd4_1},
      {"mem_dma_bd4_2", aie2ps::mem_dma_bd4_2},
      {"mem_dma_bd4_3", aie2ps::mem_dma_bd4_3},
      {"mem_dma_bd4_4", aie2ps::mem_dma_bd4_4},
      {"mem_dma_bd4_5", aie2ps::mem_dma_bd4_5},
      {"mem_dma_bd4_6", aie2ps::mem_dma_bd4_6},
      {"mem_dma_bd4_7", aie2ps::mem_dma_bd4_7},
      {"mem_dma_bd5_0", aie2ps::mem_dma_bd5_0},
      {"mem_dma_bd5_1", aie2ps::mem_dma_bd5_1},
      {"mem_dma_bd5_2", aie2ps::mem_dma_bd5_2},
      {"mem_dma_bd5_3", aie2ps::mem_dma_bd5_3},
      {"mem_dma_bd5_4", aie2ps::mem_dma_bd5_4},
      {"mem_dma_bd5_5", aie2ps::mem_dma_bd5_5},
      {"mem_dma_bd5_6", aie2ps::mem_dma_bd5_6},
      {"mem_dma_bd5_7", aie2ps::mem_dma_bd5_7},
      {"mem_dma_bd6_0", aie2ps::mem_dma_bd6_0},
      {"mem_dma_bd6_1", aie2ps::mem_dma_bd6_1},
      {"mem_dma_bd6_2", aie2ps::mem_dma_bd6_2},
      {"mem_dma_bd6_3", aie2ps::mem_dma_bd6_3},
      {"mem_dma_bd6_4", aie2ps::mem_dma_bd6_4},
      {"mem_dma_bd6_5", aie2ps::mem_dma_bd6_5},
      {"mem_dma_bd6_6", aie2ps::mem_dma_bd6_6},
      {"mem_dma_bd6_7", aie2ps::mem_dma_bd6_7},
      {"mem_dma_bd7_0", aie2ps::mem_dma_bd7_0},
      {"mem_dma_bd7_1", aie2ps::mem_dma_bd7_1},
      {"mem_dma_bd7_2", aie2ps::mem_dma_bd7_2},
      {"mem_dma_bd7_3", aie2ps::mem_dma_bd7_3},
      {"mem_dma_bd7_4", aie2ps::mem_dma_bd7_4},
      {"mem_dma_bd7_5", aie2ps::mem_dma_bd7_5},
      {"mem_dma_bd7_6", aie2ps::mem_dma_bd7_6},
      {"mem_dma_bd7_7", aie2ps::mem_dma_bd7_7},
      {"mem_dma_bd8_0", aie2ps::mem_dma_bd8_0},
      {"mem_dma_bd8_1", aie2ps::mem_dma_bd8_1},
      {"mem_dma_bd8_2", aie2ps::mem_dma_bd8_2},
      {"mem_dma_bd8_3", aie2ps::mem_dma_bd8_3},
      {"mem_dma_bd8_4", aie2ps::mem_dma_bd8_4},
      {"mem_dma_bd8_5", aie2ps::mem_dma_bd8_5},
      {"mem_dma_bd8_6", aie2ps::mem_dma_bd8_6},
      {"mem_dma_bd8_7", aie2ps::mem_dma_bd8_7},
      {"mem_dma_bd9_0", aie2ps::mem_dma_bd9_0},
      {"mem_dma_bd9_1", aie2ps::mem_dma_bd9_1},
      {"mem_dma_bd9_2", aie2ps::mem_dma_bd9_2},
      {"mem_dma_bd9_3", aie2ps::mem_dma_bd9_3},
      {"mem_dma_bd9_4", aie2ps::mem_dma_bd9_4},
      {"mem_dma_bd9_5", aie2ps::mem_dma_bd9_5},
      {"mem_dma_bd9_6", aie2ps::mem_dma_bd9_6},
      {"mem_dma_bd9_7", aie2ps::mem_dma_bd9_7},
      {"mem_dma_bd10_0", aie2ps::mem_dma_bd10_0},
      {"mem_dma_bd10_1", aie2ps::mem_dma_bd10_1},
      {"mem_dma_bd10_2", aie2ps::mem_dma_bd10_2},
      {"mem_dma_bd10_3", aie2ps::mem_dma_bd10_3},
      {"mem_dma_bd10_4", aie2ps::mem_dma_bd10_4},
      {"mem_dma_bd10_5", aie2ps::mem_dma_bd10_5},
      {"mem_dma_bd10_6", aie2ps::mem_dma_bd10_6},
      {"mem_dma_bd10_7", aie2ps::mem_dma_bd10_7},
      {"mem_dma_bd11_0", aie2ps::mem_dma_bd11_0},
      {"mem_dma_bd11_1", aie2ps::mem_dma_bd11_1},
      {"mem_dma_bd11_2", aie2ps::mem_dma_bd11_2},
      {"mem_dma_bd11_3", aie2ps::mem_dma_bd11_3},
      {"mem_dma_bd11_4", aie2ps::mem_dma_bd11_4},
      {"mem_dma_bd11_5", aie2ps::mem_dma_bd11_5},
      {"mem_dma_bd11_6", aie2ps::mem_dma_bd11_6},
      {"mem_dma_bd11_7", aie2ps::mem_dma_bd11_7},
      {"mem_dma_bd12_0", aie2ps::mem_dma_bd12_0},
      {"mem_dma_bd12_1", aie2ps::mem_dma_bd12_1},
      {"mem_dma_bd12_2", aie2ps::mem_dma_bd12_2},
      {"mem_dma_bd12_3", aie2ps::mem_dma_bd12_3},
      {"mem_dma_bd12_4", aie2ps::mem_dma_bd12_4},
      {"mem_dma_bd12_5", aie2ps::mem_dma_bd12_5},
      {"mem_dma_bd12_6", aie2ps::mem_dma_bd12_6},
      {"mem_dma_bd12_7", aie2ps::mem_dma_bd12_7},
      {"mem_dma_bd13_0", aie2ps::mem_dma_bd13_0},
      {"mem_dma_bd13_1", aie2ps::mem_dma_bd13_1},
      {"mem_dma_bd13_2", aie2ps::mem_dma_bd13_2},
      {"mem_dma_bd13_3", aie2ps::mem_dma_bd13_3},
      {"mem_dma_bd13_4", aie2ps::mem_dma_bd13_4},
      {"mem_dma_bd13_5", aie2ps::mem_dma_bd13_5},
      {"mem_dma_bd13_6", aie2ps::mem_dma_bd13_6},
      {"mem_dma_bd13_7", aie2ps::mem_dma_bd13_7},
      {"mem_dma_bd14_0", aie2ps::mem_dma_bd14_0},
      {"mem_dma_bd14_1", aie2ps::mem_dma_bd14_1},
      {"mem_dma_bd14_2", aie2ps::mem_dma_bd14_2},
      {"mem_dma_bd14_3", aie2ps::mem_dma_bd14_3},
      {"mem_dma_bd14_4", aie2ps::mem_dma_bd14_4},
      {"mem_dma_bd14_5", aie2ps::mem_dma_bd14_5},
      {"mem_dma_bd14_6", aie2ps::mem_dma_bd14_6},
      {"mem_dma_bd14_7", aie2ps::mem_dma_bd14_7},
      {"mem_dma_bd15_0", aie2ps::mem_dma_bd15_0},
      {"mem_dma_bd15_1", aie2ps::mem_dma_bd15_1},
      {"mem_dma_bd15_2", aie2ps::mem_dma_bd15_2},
      {"mem_dma_bd15_3", aie2ps::mem_dma_bd15_3},
      {"mem_dma_bd15_4", aie2ps::mem_dma_bd15_4},
      {"mem_dma_bd15_5", aie2ps::mem_dma_bd15_5},
      {"mem_dma_bd15_6", aie2ps::mem_dma_bd15_6},
      {"mem_dma_bd15_7", aie2ps::mem_dma_bd15_7},
      {"mem_dma_bd16_0", aie2ps::mem_dma_bd16_0},
      {"mem_dma_bd16_1", aie2ps::mem_dma_bd16_1},
      {"mem_dma_bd16_2", aie2ps::mem_dma_bd16_2},
      {"mem_dma_bd16_3", aie2ps::mem_dma_bd16_3},
      {"mem_dma_bd16_4", aie2ps::mem_dma_bd16_4},
      {"mem_dma_bd16_5", aie2ps::mem_dma_bd16_5},
      {"mem_dma_bd16_6", aie2ps::mem_dma_bd16_6},
      {"mem_dma_bd16_7", aie2ps::mem_dma_bd16_7},
      {"mem_dma_bd17_0", aie2ps::mem_dma_bd17_0},
      {"mem_dma_bd17_1", aie2ps::mem_dma_bd17_1},
      {"mem_dma_bd17_2", aie2ps::mem_dma_bd17_2},
      {"mem_dma_bd17_3", aie2ps::mem_dma_bd17_3},
      {"mem_dma_bd17_4", aie2ps::mem_dma_bd17_4},
      {"mem_dma_bd17_5", aie2ps::mem_dma_bd17_5},
      {"mem_dma_bd17_6", aie2ps::mem_dma_bd17_6},
      {"mem_dma_bd17_7", aie2ps::mem_dma_bd17_7},
      {"mem_dma_bd18_0", aie2ps::mem_dma_bd18_0},
      {"mem_dma_bd18_1", aie2ps::mem_dma_bd18_1},
      {"mem_dma_bd18_2", aie2ps::mem_dma_bd18_2},
      {"mem_dma_bd18_3", aie2ps::mem_dma_bd18_3},
      {"mem_dma_bd18_4", aie2ps::mem_dma_bd18_4},
      {"mem_dma_bd18_5", aie2ps::mem_dma_bd18_5},
      {"mem_dma_bd18_6", aie2ps::mem_dma_bd18_6},
      {"mem_dma_bd18_7", aie2ps::mem_dma_bd18_7},
      {"mem_dma_bd19_0", aie2ps::mem_dma_bd19_0},
      {"mem_dma_bd19_1", aie2ps::mem_dma_bd19_1},
      {"mem_dma_bd19_2", aie2ps::mem_dma_bd19_2},
      {"mem_dma_bd19_3", aie2ps::mem_dma_bd19_3},
      {"mem_dma_bd19_4", aie2ps::mem_dma_bd19_4},
      {"mem_dma_bd19_5", aie2ps::mem_dma_bd19_5},
      {"mem_dma_bd19_6", aie2ps::mem_dma_bd19_6},
      {"mem_dma_bd19_7", aie2ps::mem_dma_bd19_7},
      {"mem_dma_bd20_0", aie2ps::mem_dma_bd20_0},
      {"mem_dma_bd20_1", aie2ps::mem_dma_bd20_1},
      {"mem_dma_bd20_2", aie2ps::mem_dma_bd20_2},
      {"mem_dma_bd20_3", aie2ps::mem_dma_bd20_3},
      {"mem_dma_bd20_4", aie2ps::mem_dma_bd20_4},
      {"mem_dma_bd20_5", aie2ps::mem_dma_bd20_5},
      {"mem_dma_bd20_6", aie2ps::mem_dma_bd20_6},
      {"mem_dma_bd20_7", aie2ps::mem_dma_bd20_7},
      {"mem_dma_bd21_0", aie2ps::mem_dma_bd21_0},
      {"mem_dma_bd21_1", aie2ps::mem_dma_bd21_1},
      {"mem_dma_bd21_2", aie2ps::mem_dma_bd21_2},
      {"mem_dma_bd21_3", aie2ps::mem_dma_bd21_3},
      {"mem_dma_bd21_4", aie2ps::mem_dma_bd21_4},
      {"mem_dma_bd21_5", aie2ps::mem_dma_bd21_5},
      {"mem_dma_bd21_6", aie2ps::mem_dma_bd21_6},
      {"mem_dma_bd21_7", aie2ps::mem_dma_bd21_7},
      {"mem_dma_bd22_0", aie2ps::mem_dma_bd22_0},
      {"mem_dma_bd22_1", aie2ps::mem_dma_bd22_1},
      {"mem_dma_bd22_2", aie2ps::mem_dma_bd22_2},
      {"mem_dma_bd22_3", aie2ps::mem_dma_bd22_3},
      {"mem_dma_bd22_4", aie2ps::mem_dma_bd22_4},
      {"mem_dma_bd22_5", aie2ps::mem_dma_bd22_5},
      {"mem_dma_bd22_6", aie2ps::mem_dma_bd22_6},
      {"mem_dma_bd22_7", aie2ps::mem_dma_bd22_7},
      {"mem_dma_bd23_0", aie2ps::mem_dma_bd23_0},
      {"mem_dma_bd23_1", aie2ps::mem_dma_bd23_1},
      {"mem_dma_bd23_2", aie2ps::mem_dma_bd23_2},
      {"mem_dma_bd23_3", aie2ps::mem_dma_bd23_3},
      {"mem_dma_bd23_4", aie2ps::mem_dma_bd23_4},
      {"mem_dma_bd23_5", aie2ps::mem_dma_bd23_5},
      {"mem_dma_bd23_6", aie2ps::mem_dma_bd23_6},
      {"mem_dma_bd23_7", aie2ps::mem_dma_bd23_7},
      {"mem_dma_bd24_0", aie2ps::mem_dma_bd24_0},
      {"mem_dma_bd24_1", aie2ps::mem_dma_bd24_1},
      {"mem_dma_bd24_2", aie2ps::mem_dma_bd24_2},
      {"mem_dma_bd24_3", aie2ps::mem_dma_bd24_3},
      {"mem_dma_bd24_4", aie2ps::mem_dma_bd24_4},
      {"mem_dma_bd24_5", aie2ps::mem_dma_bd24_5},
      {"mem_dma_bd24_6", aie2ps::mem_dma_bd24_6},
      {"mem_dma_bd24_7", aie2ps::mem_dma_bd24_7},
      {"mem_dma_bd25_0", aie2ps::mem_dma_bd25_0},
      {"mem_dma_bd25_1", aie2ps::mem_dma_bd25_1},
      {"mem_dma_bd25_2", aie2ps::mem_dma_bd25_2},
      {"mem_dma_bd25_3", aie2ps::mem_dma_bd25_3},
      {"mem_dma_bd25_4", aie2ps::mem_dma_bd25_4},
      {"mem_dma_bd25_5", aie2ps::mem_dma_bd25_5},
      {"mem_dma_bd25_6", aie2ps::mem_dma_bd25_6},
      {"mem_dma_bd25_7", aie2ps::mem_dma_bd25_7},
      {"mem_dma_bd26_0", aie2ps::mem_dma_bd26_0},
      {"mem_dma_bd26_1", aie2ps::mem_dma_bd26_1},
      {"mem_dma_bd26_2", aie2ps::mem_dma_bd26_2},
      {"mem_dma_bd26_3", aie2ps::mem_dma_bd26_3},
      {"mem_dma_bd26_4", aie2ps::mem_dma_bd26_4},
      {"mem_dma_bd26_5", aie2ps::mem_dma_bd26_5},
      {"mem_dma_bd26_6", aie2ps::mem_dma_bd26_6},
      {"mem_dma_bd26_7", aie2ps::mem_dma_bd26_7},
      {"mem_dma_bd27_0", aie2ps::mem_dma_bd27_0},
      {"mem_dma_bd27_1", aie2ps::mem_dma_bd27_1},
      {"mem_dma_bd27_2", aie2ps::mem_dma_bd27_2},
      {"mem_dma_bd27_3", aie2ps::mem_dma_bd27_3},
      {"mem_dma_bd27_4", aie2ps::mem_dma_bd27_4},
      {"mem_dma_bd27_5", aie2ps::mem_dma_bd27_5},
      {"mem_dma_bd27_6", aie2ps::mem_dma_bd27_6},
      {"mem_dma_bd27_7", aie2ps::mem_dma_bd27_7},
      {"mem_dma_bd28_0", aie2ps::mem_dma_bd28_0},
      {"mem_dma_bd28_1", aie2ps::mem_dma_bd28_1},
      {"mem_dma_bd28_2", aie2ps::mem_dma_bd28_2},
      {"mem_dma_bd28_3", aie2ps::mem_dma_bd28_3},
      {"mem_dma_bd28_4", aie2ps::mem_dma_bd28_4},
      {"mem_dma_bd28_5", aie2ps::mem_dma_bd28_5},
      {"mem_dma_bd28_6", aie2ps::mem_dma_bd28_6},
      {"mem_dma_bd28_7", aie2ps::mem_dma_bd28_7},
      {"mem_dma_bd29_0", aie2ps::mem_dma_bd29_0},
      {"mem_dma_bd29_1", aie2ps::mem_dma_bd29_1},
      {"mem_dma_bd29_2", aie2ps::mem_dma_bd29_2},
      {"mem_dma_bd29_3", aie2ps::mem_dma_bd29_3},
      {"mem_dma_bd29_4", aie2ps::mem_dma_bd29_4},
      {"mem_dma_bd29_5", aie2ps::mem_dma_bd29_5},
      {"mem_dma_bd29_6", aie2ps::mem_dma_bd29_6},
      {"mem_dma_bd29_7", aie2ps::mem_dma_bd29_7},
      {"mem_dma_bd30_0", aie2ps::mem_dma_bd30_0},
      {"mem_dma_bd30_1", aie2ps::mem_dma_bd30_1},
      {"mem_dma_bd30_2", aie2ps::mem_dma_bd30_2},
      {"mem_dma_bd30_3", aie2ps::mem_dma_bd30_3},
      {"mem_dma_bd30_4", aie2ps::mem_dma_bd30_4},
      {"mem_dma_bd30_5", aie2ps::mem_dma_bd30_5},
      {"mem_dma_bd30_6", aie2ps::mem_dma_bd30_6},
      {"mem_dma_bd30_7", aie2ps::mem_dma_bd30_7},
      {"mem_dma_bd31_0", aie2ps::mem_dma_bd31_0},
      {"mem_dma_bd31_1", aie2ps::mem_dma_bd31_1},
      {"mem_dma_bd31_2", aie2ps::mem_dma_bd31_2},
      {"mem_dma_bd31_3", aie2ps::mem_dma_bd31_3},
      {"mem_dma_bd31_4", aie2ps::mem_dma_bd31_4},
      {"mem_dma_bd31_5", aie2ps::mem_dma_bd31_5},
      {"mem_dma_bd31_6", aie2ps::mem_dma_bd31_6},
      {"mem_dma_bd31_7", aie2ps::mem_dma_bd31_7},
      {"mem_dma_bd32_0", aie2ps::mem_dma_bd32_0},
      {"mem_dma_bd32_1", aie2ps::mem_dma_bd32_1},
      {"mem_dma_bd32_2", aie2ps::mem_dma_bd32_2},
      {"mem_dma_bd32_3", aie2ps::mem_dma_bd32_3},
      {"mem_dma_bd32_4", aie2ps::mem_dma_bd32_4},
      {"mem_dma_bd32_5", aie2ps::mem_dma_bd32_5},
      {"mem_dma_bd32_6", aie2ps::mem_dma_bd32_6},
      {"mem_dma_bd32_7", aie2ps::mem_dma_bd32_7},
      {"mem_dma_bd33_0", aie2ps::mem_dma_bd33_0},
      {"mem_dma_bd33_1", aie2ps::mem_dma_bd33_1},
      {"mem_dma_bd33_2", aie2ps::mem_dma_bd33_2},
      {"mem_dma_bd33_3", aie2ps::mem_dma_bd33_3},
      {"mem_dma_bd33_4", aie2ps::mem_dma_bd33_4},
      {"mem_dma_bd33_5", aie2ps::mem_dma_bd33_5},
      {"mem_dma_bd33_6", aie2ps::mem_dma_bd33_6},
      {"mem_dma_bd33_7", aie2ps::mem_dma_bd33_7},
      {"mem_dma_bd34_0", aie2ps::mem_dma_bd34_0},
      {"mem_dma_bd34_1", aie2ps::mem_dma_bd34_1},
      {"mem_dma_bd34_2", aie2ps::mem_dma_bd34_2},
      {"mem_dma_bd34_3", aie2ps::mem_dma_bd34_3},
      {"mem_dma_bd34_4", aie2ps::mem_dma_bd34_4},
      {"mem_dma_bd34_5", aie2ps::mem_dma_bd34_5},
      {"mem_dma_bd34_6", aie2ps::mem_dma_bd34_6},
      {"mem_dma_bd34_7", aie2ps::mem_dma_bd34_7},
      {"mem_dma_bd35_0", aie2ps::mem_dma_bd35_0},
      {"mem_dma_bd35_1", aie2ps::mem_dma_bd35_1},
      {"mem_dma_bd35_2", aie2ps::mem_dma_bd35_2},
      {"mem_dma_bd35_3", aie2ps::mem_dma_bd35_3},
      {"mem_dma_bd35_4", aie2ps::mem_dma_bd35_4},
      {"mem_dma_bd35_5", aie2ps::mem_dma_bd35_5},
      {"mem_dma_bd35_6", aie2ps::mem_dma_bd35_6},
      {"mem_dma_bd35_7", aie2ps::mem_dma_bd35_7},
      {"mem_dma_bd36_0", aie2ps::mem_dma_bd36_0},
      {"mem_dma_bd36_1", aie2ps::mem_dma_bd36_1},
      {"mem_dma_bd36_2", aie2ps::mem_dma_bd36_2},
      {"mem_dma_bd36_3", aie2ps::mem_dma_bd36_3},
      {"mem_dma_bd36_4", aie2ps::mem_dma_bd36_4},
      {"mem_dma_bd36_5", aie2ps::mem_dma_bd36_5},
      {"mem_dma_bd36_6", aie2ps::mem_dma_bd36_6},
      {"mem_dma_bd36_7", aie2ps::mem_dma_bd36_7},
      {"mem_dma_bd37_0", aie2ps::mem_dma_bd37_0},
      {"mem_dma_bd37_1", aie2ps::mem_dma_bd37_1},
      {"mem_dma_bd37_2", aie2ps::mem_dma_bd37_2},
      {"mem_dma_bd37_3", aie2ps::mem_dma_bd37_3},
      {"mem_dma_bd37_4", aie2ps::mem_dma_bd37_4},
      {"mem_dma_bd37_5", aie2ps::mem_dma_bd37_5},
      {"mem_dma_bd37_6", aie2ps::mem_dma_bd37_6},
      {"mem_dma_bd37_7", aie2ps::mem_dma_bd37_7},
      {"mem_dma_bd38_0", aie2ps::mem_dma_bd38_0},
      {"mem_dma_bd38_1", aie2ps::mem_dma_bd38_1},
      {"mem_dma_bd38_2", aie2ps::mem_dma_bd38_2},
      {"mem_dma_bd38_3", aie2ps::mem_dma_bd38_3},
      {"mem_dma_bd38_4", aie2ps::mem_dma_bd38_4},
      {"mem_dma_bd38_5", aie2ps::mem_dma_bd38_5},
      {"mem_dma_bd38_6", aie2ps::mem_dma_bd38_6},
      {"mem_dma_bd38_7", aie2ps::mem_dma_bd38_7},
      {"mem_dma_bd39_0", aie2ps::mem_dma_bd39_0},
      {"mem_dma_bd39_1", aie2ps::mem_dma_bd39_1},
      {"mem_dma_bd39_2", aie2ps::mem_dma_bd39_2},
      {"mem_dma_bd39_3", aie2ps::mem_dma_bd39_3},
      {"mem_dma_bd39_4", aie2ps::mem_dma_bd39_4},
      {"mem_dma_bd39_5", aie2ps::mem_dma_bd39_5},
      {"mem_dma_bd39_6", aie2ps::mem_dma_bd39_6},
      {"mem_dma_bd39_7", aie2ps::mem_dma_bd39_7},
      {"mem_dma_bd40_0", aie2ps::mem_dma_bd40_0},
      {"mem_dma_bd40_1", aie2ps::mem_dma_bd40_1},
      {"mem_dma_bd40_2", aie2ps::mem_dma_bd40_2},
      {"mem_dma_bd40_3", aie2ps::mem_dma_bd40_3},
      {"mem_dma_bd40_4", aie2ps::mem_dma_bd40_4},
      {"mem_dma_bd40_5", aie2ps::mem_dma_bd40_5},
      {"mem_dma_bd40_6", aie2ps::mem_dma_bd40_6},
      {"mem_dma_bd40_7", aie2ps::mem_dma_bd40_7},
      {"mem_dma_bd41_0", aie2ps::mem_dma_bd41_0},
      {"mem_dma_bd41_1", aie2ps::mem_dma_bd41_1},
      {"mem_dma_bd41_2", aie2ps::mem_dma_bd41_2},
      {"mem_dma_bd41_3", aie2ps::mem_dma_bd41_3},
      {"mem_dma_bd41_4", aie2ps::mem_dma_bd41_4},
      {"mem_dma_bd41_5", aie2ps::mem_dma_bd41_5},
      {"mem_dma_bd41_6", aie2ps::mem_dma_bd41_6},
      {"mem_dma_bd41_7", aie2ps::mem_dma_bd41_7},
      {"mem_dma_bd42_0", aie2ps::mem_dma_bd42_0},
      {"mem_dma_bd42_1", aie2ps::mem_dma_bd42_1},
      {"mem_dma_bd42_2", aie2ps::mem_dma_bd42_2},
      {"mem_dma_bd42_3", aie2ps::mem_dma_bd42_3},
      {"mem_dma_bd42_4", aie2ps::mem_dma_bd42_4},
      {"mem_dma_bd42_5", aie2ps::mem_dma_bd42_5},
      {"mem_dma_bd42_6", aie2ps::mem_dma_bd42_6},
      {"mem_dma_bd42_7", aie2ps::mem_dma_bd42_7},
      {"mem_dma_bd43_0", aie2ps::mem_dma_bd43_0},
      {"mem_dma_bd43_1", aie2ps::mem_dma_bd43_1},
      {"mem_dma_bd43_2", aie2ps::mem_dma_bd43_2},
      {"mem_dma_bd43_3", aie2ps::mem_dma_bd43_3},
      {"mem_dma_bd43_4", aie2ps::mem_dma_bd43_4},
      {"mem_dma_bd43_5", aie2ps::mem_dma_bd43_5},
      {"mem_dma_bd43_6", aie2ps::mem_dma_bd43_6},
      {"mem_dma_bd43_7", aie2ps::mem_dma_bd43_7},
      {"mem_dma_bd44_0", aie2ps::mem_dma_bd44_0},
      {"mem_dma_bd44_1", aie2ps::mem_dma_bd44_1},
      {"mem_dma_bd44_2", aie2ps::mem_dma_bd44_2},
      {"mem_dma_bd44_3", aie2ps::mem_dma_bd44_3},
      {"mem_dma_bd44_4", aie2ps::mem_dma_bd44_4},
      {"mem_dma_bd44_5", aie2ps::mem_dma_bd44_5},
      {"mem_dma_bd44_6", aie2ps::mem_dma_bd44_6},
      {"mem_dma_bd44_7", aie2ps::mem_dma_bd44_7},
      {"mem_dma_bd45_0", aie2ps::mem_dma_bd45_0},
      {"mem_dma_bd45_1", aie2ps::mem_dma_bd45_1},
      {"mem_dma_bd45_2", aie2ps::mem_dma_bd45_2},
      {"mem_dma_bd45_3", aie2ps::mem_dma_bd45_3},
      {"mem_dma_bd45_4", aie2ps::mem_dma_bd45_4},
      {"mem_dma_bd45_5", aie2ps::mem_dma_bd45_5},
      {"mem_dma_bd45_6", aie2ps::mem_dma_bd45_6},
      {"mem_dma_bd45_7", aie2ps::mem_dma_bd45_7},
      {"mem_dma_bd46_0", aie2ps::mem_dma_bd46_0},
      {"mem_dma_bd46_1", aie2ps::mem_dma_bd46_1},
      {"mem_dma_bd46_2", aie2ps::mem_dma_bd46_2},
      {"mem_dma_bd46_3", aie2ps::mem_dma_bd46_3},
      {"mem_dma_bd46_4", aie2ps::mem_dma_bd46_4},
      {"mem_dma_bd46_5", aie2ps::mem_dma_bd46_5},
      {"mem_dma_bd46_6", aie2ps::mem_dma_bd46_6},
      {"mem_dma_bd46_7", aie2ps::mem_dma_bd46_7},
      {"mem_dma_bd47_0", aie2ps::mem_dma_bd47_0},
      {"mem_dma_bd47_1", aie2ps::mem_dma_bd47_1},
      {"mem_dma_bd47_2", aie2ps::mem_dma_bd47_2},
      {"mem_dma_bd47_3", aie2ps::mem_dma_bd47_3},
      {"mem_dma_bd47_4", aie2ps::mem_dma_bd47_4},
      {"mem_dma_bd47_5", aie2ps::mem_dma_bd47_5},
      {"mem_dma_bd47_6", aie2ps::mem_dma_bd47_6},
      {"mem_dma_bd47_7", aie2ps::mem_dma_bd47_7},
      {"mem_dma_s2mm_0_ctrl", aie2ps::mem_dma_s2mm_0_ctrl},
      {"mem_dma_s2mm_0_start_queue", aie2ps::mem_dma_s2mm_0_start_queue},
      {"mem_dma_s2mm_1_ctrl", aie2ps::mem_dma_s2mm_1_ctrl},
      {"mem_dma_s2mm_1_start_queue", aie2ps::mem_dma_s2mm_1_start_queue},
      {"mem_dma_s2mm_2_ctrl", aie2ps::mem_dma_s2mm_2_ctrl},
      {"mem_dma_s2mm_2_start_queue", aie2ps::mem_dma_s2mm_2_start_queue},
      {"mem_dma_s2mm_3_ctrl", aie2ps::mem_dma_s2mm_3_ctrl},
      {"mem_dma_s2mm_3_start_queue", aie2ps::mem_dma_s2mm_3_start_queue},
      {"mem_dma_s2mm_4_ctrl", aie2ps::mem_dma_s2mm_4_ctrl},
      {"mem_dma_s2mm_4_start_queue", aie2ps::mem_dma_s2mm_4_start_queue},
      {"mem_dma_s2mm_5_ctrl", aie2ps::mem_dma_s2mm_5_ctrl},
      {"mem_dma_s2mm_5_start_queue", aie2ps::mem_dma_s2mm_5_start_queue},
      {"mem_dma_mm2s_0_ctrl", aie2ps::mem_dma_mm2s_0_ctrl},
      {"mem_dma_mm2s_0_start_queue", aie2ps::mem_dma_mm2s_0_start_queue},
      {"mem_dma_mm2s_1_ctrl", aie2ps::mem_dma_mm2s_1_ctrl},
      {"mem_dma_mm2s_1_start_queue", aie2ps::mem_dma_mm2s_1_start_queue},
      {"mem_dma_mm2s_2_ctrl", aie2ps::mem_dma_mm2s_2_ctrl},
      {"mem_dma_mm2s_2_start_queue", aie2ps::mem_dma_mm2s_2_start_queue},
      {"mem_dma_mm2s_3_ctrl", aie2ps::mem_dma_mm2s_3_ctrl},
      {"mem_dma_mm2s_3_start_queue", aie2ps::mem_dma_mm2s_3_start_queue},
      {"mem_dma_mm2s_4_ctrl", aie2ps::mem_dma_mm2s_4_ctrl},
      {"mem_dma_mm2s_4_start_queue", aie2ps::mem_dma_mm2s_4_start_queue},
      {"mem_dma_mm2s_5_ctrl", aie2ps::mem_dma_mm2s_5_ctrl},
      {"mem_dma_mm2s_5_start_queue", aie2ps::mem_dma_mm2s_5_start_queue},
      {"mem_dma_s2mm_status_0", aie2ps::mem_dma_s2mm_status_0},
      {"mem_dma_s2mm_status_1", aie2ps::mem_dma_s2mm_status_1},
      {"mem_dma_s2mm_status_2", aie2ps::mem_dma_s2mm_status_2},
      {"mem_dma_s2mm_status_3", aie2ps::mem_dma_s2mm_status_3},
      {"mem_dma_s2mm_status_4", aie2ps::mem_dma_s2mm_status_4},
      {"mem_dma_s2mm_status_5", aie2ps::mem_dma_s2mm_status_5},
      {"mem_dma_mm2s_status_0", aie2ps::mem_dma_mm2s_status_0},
      {"mem_dma_mm2s_status_1", aie2ps::mem_dma_mm2s_status_1},
      {"mem_dma_mm2s_status_2", aie2ps::mem_dma_mm2s_status_2},
      {"mem_dma_mm2s_status_3", aie2ps::mem_dma_mm2s_status_3},
      {"mem_dma_mm2s_status_4", aie2ps::mem_dma_mm2s_status_4},
      {"mem_dma_mm2s_status_5", aie2ps::mem_dma_mm2s_status_5},
      {"mem_dma_event_channel_selection", aie2ps::mem_dma_event_channel_selection},
      {"mem_dma_s2mm_current_write_count_0", aie2ps::mem_dma_s2mm_current_write_count_0},
      {"mem_dma_s2mm_current_write_count_1", aie2ps::mem_dma_s2mm_current_write_count_1},
      {"mem_dma_s2mm_current_write_count_2", aie2ps::mem_dma_s2mm_current_write_count_2},
      {"mem_dma_s2mm_current_write_count_3", aie2ps::mem_dma_s2mm_current_write_count_3},
      {"mem_dma_s2mm_current_write_count_4", aie2ps::mem_dma_s2mm_current_write_count_4},
      {"mem_dma_s2mm_current_write_count_5", aie2ps::mem_dma_s2mm_current_write_count_5},
      {"mem_dma_s2mm_fot_count_fifo_pop_0", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_0},
      {"mem_dma_s2mm_fot_count_fifo_pop_1", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_1},
      {"mem_dma_s2mm_fot_count_fifo_pop_2", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_2},
      {"mem_dma_s2mm_fot_count_fifo_pop_3", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_3},
      {"mem_dma_s2mm_fot_count_fifo_pop_4", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_4},
      {"mem_dma_s2mm_fot_count_fifo_pop_5", aie2ps::mem_dma_s2mm_fot_count_fifo_pop_5},
      {"mem_dma_mm2s_0_constant_pad_value", aie2ps::mem_dma_mm2s_0_constant_pad_value},
      {"mem_dma_mm2s_1_constant_pad_value", aie2ps::mem_dma_mm2s_1_constant_pad_value},
      {"mem_dma_mm2s_2_constant_pad_value", aie2ps::mem_dma_mm2s_2_constant_pad_value},
      {"mem_dma_mm2s_3_constant_pad_value", aie2ps::mem_dma_mm2s_3_constant_pad_value},
      {"mem_dma_mm2s_4_constant_pad_value", aie2ps::mem_dma_mm2s_4_constant_pad_value},
      {"mem_dma_mm2s_5_constant_pad_value", aie2ps::mem_dma_mm2s_5_constant_pad_value},
      {"mem_stream_switch_master_config_dma0", aie2ps::mem_stream_switch_master_config_dma0},
      {"mem_stream_switch_master_config_dma1", aie2ps::mem_stream_switch_master_config_dma1},
      {"mem_stream_switch_master_config_dma2", aie2ps::mem_stream_switch_master_config_dma2},
      {"mem_stream_switch_master_config_dma3", aie2ps::mem_stream_switch_master_config_dma3},
      {"mem_stream_switch_master_config_dma4", aie2ps::mem_stream_switch_master_config_dma4},
      {"mem_stream_switch_master_config_dma5", aie2ps::mem_stream_switch_master_config_dma5},
      {"mem_stream_switch_master_config_tile_ctrl", aie2ps::mem_stream_switch_master_config_tile_ctrl},
      {"mem_stream_switch_master_config_south0", aie2ps::mem_stream_switch_master_config_south0},
      {"mem_stream_switch_master_config_south1", aie2ps::mem_stream_switch_master_config_south1},
      {"mem_stream_switch_master_config_south2", aie2ps::mem_stream_switch_master_config_south2},
      {"mem_stream_switch_master_config_south3", aie2ps::mem_stream_switch_master_config_south3},
      {"mem_stream_switch_master_config_north0", aie2ps::mem_stream_switch_master_config_north0},
      {"mem_stream_switch_master_config_north1", aie2ps::mem_stream_switch_master_config_north1},
      {"mem_stream_switch_master_config_north2", aie2ps::mem_stream_switch_master_config_north2},
      {"mem_stream_switch_master_config_north3", aie2ps::mem_stream_switch_master_config_north3},
      {"mem_stream_switch_master_config_north4", aie2ps::mem_stream_switch_master_config_north4},
      {"mem_stream_switch_master_config_north5", aie2ps::mem_stream_switch_master_config_north5},
      {"mem_stream_switch_slave_config_dma_0", aie2ps::mem_stream_switch_slave_config_dma_0},
      {"mem_stream_switch_slave_config_dma_1", aie2ps::mem_stream_switch_slave_config_dma_1},
      {"mem_stream_switch_slave_config_dma_2", aie2ps::mem_stream_switch_slave_config_dma_2},
      {"mem_stream_switch_slave_config_dma_3", aie2ps::mem_stream_switch_slave_config_dma_3},
      {"mem_stream_switch_slave_config_dma_4", aie2ps::mem_stream_switch_slave_config_dma_4},
      {"mem_stream_switch_slave_config_dma_5", aie2ps::mem_stream_switch_slave_config_dma_5},
      {"mem_stream_switch_slave_config_tile_ctrl", aie2ps::mem_stream_switch_slave_config_tile_ctrl},
      {"mem_stream_switch_slave_config_south_0", aie2ps::mem_stream_switch_slave_config_south_0},
      {"mem_stream_switch_slave_config_south_1", aie2ps::mem_stream_switch_slave_config_south_1},
      {"mem_stream_switch_slave_config_south_2", aie2ps::mem_stream_switch_slave_config_south_2},
      {"mem_stream_switch_slave_config_south_3", aie2ps::mem_stream_switch_slave_config_south_3},
      {"mem_stream_switch_slave_config_south_4", aie2ps::mem_stream_switch_slave_config_south_4},
      {"mem_stream_switch_slave_config_south_5", aie2ps::mem_stream_switch_slave_config_south_5},
      {"mem_stream_switch_slave_config_north_0", aie2ps::mem_stream_switch_slave_config_north_0},
      {"mem_stream_switch_slave_config_north_1", aie2ps::mem_stream_switch_slave_config_north_1},
      {"mem_stream_switch_slave_config_north_2", aie2ps::mem_stream_switch_slave_config_north_2},
      {"mem_stream_switch_slave_config_north_3", aie2ps::mem_stream_switch_slave_config_north_3},
      {"mem_stream_switch_slave_config_trace", aie2ps::mem_stream_switch_slave_config_trace},
      {"mem_stream_switch_slave_dma_0_slot0", aie2ps::mem_stream_switch_slave_dma_0_slot0},
      {"mem_stream_switch_slave_dma_0_slot1", aie2ps::mem_stream_switch_slave_dma_0_slot1},
      {"mem_stream_switch_slave_dma_0_slot2", aie2ps::mem_stream_switch_slave_dma_0_slot2},
      {"mem_stream_switch_slave_dma_0_slot3", aie2ps::mem_stream_switch_slave_dma_0_slot3},
      {"mem_stream_switch_slave_dma_1_slot0", aie2ps::mem_stream_switch_slave_dma_1_slot0},
      {"mem_stream_switch_slave_dma_1_slot1", aie2ps::mem_stream_switch_slave_dma_1_slot1},
      {"mem_stream_switch_slave_dma_1_slot2", aie2ps::mem_stream_switch_slave_dma_1_slot2},
      {"mem_stream_switch_slave_dma_1_slot3", aie2ps::mem_stream_switch_slave_dma_1_slot3},
      {"mem_stream_switch_slave_dma_2_slot0", aie2ps::mem_stream_switch_slave_dma_2_slot0},
      {"mem_stream_switch_slave_dma_2_slot1", aie2ps::mem_stream_switch_slave_dma_2_slot1},
      {"mem_stream_switch_slave_dma_2_slot2", aie2ps::mem_stream_switch_slave_dma_2_slot2},
      {"mem_stream_switch_slave_dma_2_slot3", aie2ps::mem_stream_switch_slave_dma_2_slot3},
      {"mem_stream_switch_slave_dma_3_slot0", aie2ps::mem_stream_switch_slave_dma_3_slot0},
      {"mem_stream_switch_slave_dma_3_slot1", aie2ps::mem_stream_switch_slave_dma_3_slot1},
      {"mem_stream_switch_slave_dma_3_slot2", aie2ps::mem_stream_switch_slave_dma_3_slot2},
      {"mem_stream_switch_slave_dma_3_slot3", aie2ps::mem_stream_switch_slave_dma_3_slot3},
      {"mem_stream_switch_slave_dma_4_slot0", aie2ps::mem_stream_switch_slave_dma_4_slot0},
      {"mem_stream_switch_slave_dma_4_slot1", aie2ps::mem_stream_switch_slave_dma_4_slot1},
      {"mem_stream_switch_slave_dma_4_slot2", aie2ps::mem_stream_switch_slave_dma_4_slot2},
      {"mem_stream_switch_slave_dma_4_slot3", aie2ps::mem_stream_switch_slave_dma_4_slot3},
      {"mem_stream_switch_slave_dma_5_slot0", aie2ps::mem_stream_switch_slave_dma_5_slot0},
      {"mem_stream_switch_slave_dma_5_slot1", aie2ps::mem_stream_switch_slave_dma_5_slot1},
      {"mem_stream_switch_slave_dma_5_slot2", aie2ps::mem_stream_switch_slave_dma_5_slot2},
      {"mem_stream_switch_slave_dma_5_slot3", aie2ps::mem_stream_switch_slave_dma_5_slot3},
      {"mem_stream_switch_slave_tile_ctrl_slot0", aie2ps::mem_stream_switch_slave_tile_ctrl_slot0},
      {"mem_stream_switch_slave_tile_ctrl_slot1", aie2ps::mem_stream_switch_slave_tile_ctrl_slot1},
      {"mem_stream_switch_slave_tile_ctrl_slot2", aie2ps::mem_stream_switch_slave_tile_ctrl_slot2},
      {"mem_stream_switch_slave_tile_ctrl_slot3", aie2ps::mem_stream_switch_slave_tile_ctrl_slot3},
      {"mem_stream_switch_slave_south_0_slot0", aie2ps::mem_stream_switch_slave_south_0_slot0},
      {"mem_stream_switch_slave_south_0_slot1", aie2ps::mem_stream_switch_slave_south_0_slot1},
      {"mem_stream_switch_slave_south_0_slot2", aie2ps::mem_stream_switch_slave_south_0_slot2},
      {"mem_stream_switch_slave_south_0_slot3", aie2ps::mem_stream_switch_slave_south_0_slot3},
      {"mem_stream_switch_slave_south_1_slot0", aie2ps::mem_stream_switch_slave_south_1_slot0},
      {"mem_stream_switch_slave_south_1_slot1", aie2ps::mem_stream_switch_slave_south_1_slot1},
      {"mem_stream_switch_slave_south_1_slot2", aie2ps::mem_stream_switch_slave_south_1_slot2},
      {"mem_stream_switch_slave_south_1_slot3", aie2ps::mem_stream_switch_slave_south_1_slot3},
      {"mem_stream_switch_slave_south_2_slot0", aie2ps::mem_stream_switch_slave_south_2_slot0},
      {"mem_stream_switch_slave_south_2_slot1", aie2ps::mem_stream_switch_slave_south_2_slot1},
      {"mem_stream_switch_slave_south_2_slot2", aie2ps::mem_stream_switch_slave_south_2_slot2},
      {"mem_stream_switch_slave_south_2_slot3", aie2ps::mem_stream_switch_slave_south_2_slot3},
      {"mem_stream_switch_slave_south_3_slot0", aie2ps::mem_stream_switch_slave_south_3_slot0},
      {"mem_stream_switch_slave_south_3_slot1", aie2ps::mem_stream_switch_slave_south_3_slot1},
      {"mem_stream_switch_slave_south_3_slot2", aie2ps::mem_stream_switch_slave_south_3_slot2},
      {"mem_stream_switch_slave_south_3_slot3", aie2ps::mem_stream_switch_slave_south_3_slot3},
      {"mem_stream_switch_slave_south_4_slot0", aie2ps::mem_stream_switch_slave_south_4_slot0},
      {"mem_stream_switch_slave_south_4_slot1", aie2ps::mem_stream_switch_slave_south_4_slot1},
      {"mem_stream_switch_slave_south_4_slot2", aie2ps::mem_stream_switch_slave_south_4_slot2},
      {"mem_stream_switch_slave_south_4_slot3", aie2ps::mem_stream_switch_slave_south_4_slot3},
      {"mem_stream_switch_slave_south_5_slot0", aie2ps::mem_stream_switch_slave_south_5_slot0},
      {"mem_stream_switch_slave_south_5_slot1", aie2ps::mem_stream_switch_slave_south_5_slot1},
      {"mem_stream_switch_slave_south_5_slot2", aie2ps::mem_stream_switch_slave_south_5_slot2},
      {"mem_stream_switch_slave_south_5_slot3", aie2ps::mem_stream_switch_slave_south_5_slot3},
      {"mem_stream_switch_slave_north_0_slot0", aie2ps::mem_stream_switch_slave_north_0_slot0},
      {"mem_stream_switch_slave_north_0_slot1", aie2ps::mem_stream_switch_slave_north_0_slot1},
      {"mem_stream_switch_slave_north_0_slot2", aie2ps::mem_stream_switch_slave_north_0_slot2},
      {"mem_stream_switch_slave_north_0_slot3", aie2ps::mem_stream_switch_slave_north_0_slot3},
      {"mem_stream_switch_slave_north_1_slot0", aie2ps::mem_stream_switch_slave_north_1_slot0},
      {"mem_stream_switch_slave_north_1_slot1", aie2ps::mem_stream_switch_slave_north_1_slot1},
      {"mem_stream_switch_slave_north_1_slot2", aie2ps::mem_stream_switch_slave_north_1_slot2},
      {"mem_stream_switch_slave_north_1_slot3", aie2ps::mem_stream_switch_slave_north_1_slot3},
      {"mem_stream_switch_slave_north_2_slot0", aie2ps::mem_stream_switch_slave_north_2_slot0},
      {"mem_stream_switch_slave_north_2_slot1", aie2ps::mem_stream_switch_slave_north_2_slot1},
      {"mem_stream_switch_slave_north_2_slot2", aie2ps::mem_stream_switch_slave_north_2_slot2},
      {"mem_stream_switch_slave_north_2_slot3", aie2ps::mem_stream_switch_slave_north_2_slot3},
      {"mem_stream_switch_slave_north_3_slot0", aie2ps::mem_stream_switch_slave_north_3_slot0},
      {"mem_stream_switch_slave_north_3_slot1", aie2ps::mem_stream_switch_slave_north_3_slot1},
      {"mem_stream_switch_slave_north_3_slot2", aie2ps::mem_stream_switch_slave_north_3_slot2},
      {"mem_stream_switch_slave_north_3_slot3", aie2ps::mem_stream_switch_slave_north_3_slot3},
      {"mem_stream_switch_slave_trace_slot0", aie2ps::mem_stream_switch_slave_trace_slot0},
      {"mem_stream_switch_slave_trace_slot1", aie2ps::mem_stream_switch_slave_trace_slot1},
      {"mem_stream_switch_slave_trace_slot2", aie2ps::mem_stream_switch_slave_trace_slot2},
      {"mem_stream_switch_slave_trace_slot3", aie2ps::mem_stream_switch_slave_trace_slot3},
      {"mem_stream_switch_deterministic_merge_arb0_slave0_1", aie2ps::mem_stream_switch_deterministic_merge_arb0_slave0_1},
      {"mem_stream_switch_deterministic_merge_arb0_slave2_3", aie2ps::mem_stream_switch_deterministic_merge_arb0_slave2_3},
      {"mem_stream_switch_deterministic_merge_arb0_ctrl", aie2ps::mem_stream_switch_deterministic_merge_arb0_ctrl},
      {"mem_stream_switch_deterministic_merge_arb1_slave0_1", aie2ps::mem_stream_switch_deterministic_merge_arb1_slave0_1},
      {"mem_stream_switch_deterministic_merge_arb1_slave2_3", aie2ps::mem_stream_switch_deterministic_merge_arb1_slave2_3},
      {"mem_stream_switch_deterministic_merge_arb1_ctrl", aie2ps::mem_stream_switch_deterministic_merge_arb1_ctrl},
      {"mem_stream_switch_event_port_selection_0", aie2ps::mem_stream_switch_event_port_selection_0},
      {"mem_stream_switch_event_port_selection_1", aie2ps::mem_stream_switch_event_port_selection_1},
      {"mem_stream_switch_parity_status", aie2ps::mem_stream_switch_parity_status},
      {"mem_stream_switch_parity_injection", aie2ps::mem_stream_switch_parity_injection},
      {"mem_tile_control_packet_handler_status", aie2ps::mem_tile_control_packet_handler_status},
      {"mem_stream_switch_adaptive_clock_gate_status", aie2ps::mem_stream_switch_adaptive_clock_gate_status},
      {"mem_stream_switch_adaptive_clock_gate_abort_period", aie2ps::mem_stream_switch_adaptive_clock_gate_abort_period},
      {"mem_lock0_value", aie2ps::mem_lock0_value},
      {"mem_lock1_value", aie2ps::mem_lock1_value},
      {"mem_lock2_value", aie2ps::mem_lock2_value},
      {"mem_lock3_value", aie2ps::mem_lock3_value},
      {"mem_lock4_value", aie2ps::mem_lock4_value},
      {"mem_lock5_value", aie2ps::mem_lock5_value},
      {"mem_lock6_value", aie2ps::mem_lock6_value},
      {"mem_lock7_value", aie2ps::mem_lock7_value},
      {"mem_lock8_value", aie2ps::mem_lock8_value},
      {"mem_lock9_value", aie2ps::mem_lock9_value},
      {"mem_lock10_value", aie2ps::mem_lock10_value},
      {"mem_lock11_value", aie2ps::mem_lock11_value},
      {"mem_lock12_value", aie2ps::mem_lock12_value},
      {"mem_lock13_value", aie2ps::mem_lock13_value},
      {"mem_lock14_value", aie2ps::mem_lock14_value},
      {"mem_lock15_value", aie2ps::mem_lock15_value},
      {"mem_lock16_value", aie2ps::mem_lock16_value},
      {"mem_lock17_value", aie2ps::mem_lock17_value},
      {"mem_lock18_value", aie2ps::mem_lock18_value},
      {"mem_lock19_value", aie2ps::mem_lock19_value},
      {"mem_lock20_value", aie2ps::mem_lock20_value},
      {"mem_lock21_value", aie2ps::mem_lock21_value},
      {"mem_lock22_value", aie2ps::mem_lock22_value},
      {"mem_lock23_value", aie2ps::mem_lock23_value},
      {"mem_lock24_value", aie2ps::mem_lock24_value},
      {"mem_lock25_value", aie2ps::mem_lock25_value},
      {"mem_lock26_value", aie2ps::mem_lock26_value},
      {"mem_lock27_value", aie2ps::mem_lock27_value},
      {"mem_lock28_value", aie2ps::mem_lock28_value},
      {"mem_lock29_value", aie2ps::mem_lock29_value},
      {"mem_lock30_value", aie2ps::mem_lock30_value},
      {"mem_lock31_value", aie2ps::mem_lock31_value},
      {"mem_lock32_value", aie2ps::mem_lock32_value},
      {"mem_lock33_value", aie2ps::mem_lock33_value},
      {"mem_lock34_value", aie2ps::mem_lock34_value},
      {"mem_lock35_value", aie2ps::mem_lock35_value},
      {"mem_lock36_value", aie2ps::mem_lock36_value},
      {"mem_lock37_value", aie2ps::mem_lock37_value},
      {"mem_lock38_value", aie2ps::mem_lock38_value},
      {"mem_lock39_value", aie2ps::mem_lock39_value},
      {"mem_lock40_value", aie2ps::mem_lock40_value},
      {"mem_lock41_value", aie2ps::mem_lock41_value},
      {"mem_lock42_value", aie2ps::mem_lock42_value},
      {"mem_lock43_value", aie2ps::mem_lock43_value},
      {"mem_lock44_value", aie2ps::mem_lock44_value},
      {"mem_lock45_value", aie2ps::mem_lock45_value},
      {"mem_lock46_value", aie2ps::mem_lock46_value},
      {"mem_lock47_value", aie2ps::mem_lock47_value},
      {"mem_lock48_value", aie2ps::mem_lock48_value},
      {"mem_lock49_value", aie2ps::mem_lock49_value},
      {"mem_lock50_value", aie2ps::mem_lock50_value},
      {"mem_lock51_value", aie2ps::mem_lock51_value},
      {"mem_lock52_value", aie2ps::mem_lock52_value},
      {"mem_lock53_value", aie2ps::mem_lock53_value},
      {"mem_lock54_value", aie2ps::mem_lock54_value},
      {"mem_lock55_value", aie2ps::mem_lock55_value},
      {"mem_lock56_value", aie2ps::mem_lock56_value},
      {"mem_lock57_value", aie2ps::mem_lock57_value},
      {"mem_lock58_value", aie2ps::mem_lock58_value},
      {"mem_lock59_value", aie2ps::mem_lock59_value},
      {"mem_lock60_value", aie2ps::mem_lock60_value},
      {"mem_lock61_value", aie2ps::mem_lock61_value},
      {"mem_lock62_value", aie2ps::mem_lock62_value},
      {"mem_lock63_value", aie2ps::mem_lock63_value},
      {"mem_locks_event_selection_0", aie2ps::mem_locks_event_selection_0},
      {"mem_locks_event_selection_1", aie2ps::mem_locks_event_selection_1},
      {"mem_locks_event_selection_2", aie2ps::mem_locks_event_selection_2},
      {"mem_locks_event_selection_3", aie2ps::mem_locks_event_selection_3},
      {"mem_locks_event_selection_4", aie2ps::mem_locks_event_selection_4},
      {"mem_locks_event_selection_5", aie2ps::mem_locks_event_selection_5},
      {"mem_locks_event_selection_6", aie2ps::mem_locks_event_selection_6},
      {"mem_locks_event_selection_7", aie2ps::mem_locks_event_selection_7},
      {"mem_locks_overflow_0", aie2ps::mem_locks_overflow_0},
      {"mem_locks_overflow_1", aie2ps::mem_locks_overflow_1},
      {"mem_locks_underflow_0", aie2ps::mem_locks_underflow_0},
      {"mem_locks_underflow_1", aie2ps::mem_locks_underflow_1},
      {"mem_lock_request", aie2ps::mem_lock_request},
      {"mem_module_clock_control", aie2ps::mem_module_clock_control},
      {"mem_tile_control", aie2ps::mem_tile_control},
      {"shim_lock_step_size", aie2ps::shim_lock_step_size},
      {"shim_dma_bd_step_size", aie2ps::shim_dma_bd_step_size},
      {"shim_dma_s2mm_step_size", aie2ps::shim_dma_s2mm_step_size},
      {"shim_lock0_value", aie2ps::shim_lock0_value},
      {"shim_lock1_value", aie2ps::shim_lock1_value},
      {"shim_lock2_value", aie2ps::shim_lock2_value},
      {"shim_lock3_value", aie2ps::shim_lock3_value},
      {"shim_lock4_value", aie2ps::shim_lock4_value},
      {"shim_lock5_value", aie2ps::shim_lock5_value},
      {"shim_lock6_value", aie2ps::shim_lock6_value},
      {"shim_lock7_value", aie2ps::shim_lock7_value},
      {"shim_lock8_value", aie2ps::shim_lock8_value},
      {"shim_lock9_value", aie2ps::shim_lock9_value},
      {"shim_lock10_value", aie2ps::shim_lock10_value},
      {"shim_lock11_value", aie2ps::shim_lock11_value},
      {"shim_lock12_value", aie2ps::shim_lock12_value},
      {"shim_lock13_value", aie2ps::shim_lock13_value},
      {"shim_lock14_value", aie2ps::shim_lock14_value},
      {"shim_lock15_value", aie2ps::shim_lock15_value},
      {"shim_locks_event_selection_0", aie2ps::shim_locks_event_selection_0},
      {"shim_locks_event_selection_1", aie2ps::shim_locks_event_selection_1},
      {"shim_locks_event_selection_2", aie2ps::shim_locks_event_selection_2},
      {"shim_locks_event_selection_3", aie2ps::shim_locks_event_selection_3},
      {"shim_locks_event_selection_4", aie2ps::shim_locks_event_selection_4},
      {"shim_locks_event_selection_5", aie2ps::shim_locks_event_selection_5},
      {"shim_locks_overflow", aie2ps::shim_locks_overflow},
      {"shim_locks_underflow", aie2ps::shim_locks_underflow},
      {"shim_interrupt_controller_2nd_level_mask", aie2ps::shim_interrupt_controller_2nd_level_mask},
      {"shim_interrupt_controller_2nd_level_enable", aie2ps::shim_interrupt_controller_2nd_level_enable},
      {"shim_interrupt_controller_2nd_level_disable", aie2ps::shim_interrupt_controller_2nd_level_disable},
      {"shim_interrupt_controller_2nd_level_status", aie2ps::shim_interrupt_controller_2nd_level_status},
      {"shim_interrupt_controller_2nd_level_interrupt", aie2ps::shim_interrupt_controller_2nd_level_interrupt},
      {"shim_me_aximm_config", aie2ps::shim_me_aximm_config},
      {"shim_mux_config", aie2ps::shim_mux_config},
      {"shim_demux_config", aie2ps::shim_demux_config},
      {"shim_axi_mm_outstanding_transactions", aie2ps::shim_axi_mm_outstanding_transactions},
      {"shim_smid", aie2ps::shim_smid},
      {"shim_dma_bd0_0", aie2ps::shim_dma_bd0_0},
      {"shim_dma_bd0_1", aie2ps::shim_dma_bd0_1},
      {"shim_dma_bd0_2", aie2ps::shim_dma_bd0_2},
      {"shim_dma_bd0_3", aie2ps::shim_dma_bd0_3},
      {"shim_dma_bd0_4", aie2ps::shim_dma_bd0_4},
      {"shim_dma_bd0_5", aie2ps::shim_dma_bd0_5},
      {"shim_dma_bd0_6", aie2ps::shim_dma_bd0_6},
      {"shim_dma_bd0_7", aie2ps::shim_dma_bd0_7},
      {"shim_dma_bd0_8", aie2ps::shim_dma_bd0_8},
      {"shim_dma_bd1_0", aie2ps::shim_dma_bd1_0},
      {"shim_dma_bd1_1", aie2ps::shim_dma_bd1_1},
      {"shim_dma_bd1_2", aie2ps::shim_dma_bd1_2},
      {"shim_dma_bd1_3", aie2ps::shim_dma_bd1_3},
      {"shim_dma_bd1_4", aie2ps::shim_dma_bd1_4},
      {"shim_dma_bd1_5", aie2ps::shim_dma_bd1_5},
      {"shim_dma_bd1_6", aie2ps::shim_dma_bd1_6},
      {"shim_dma_bd1_7", aie2ps::shim_dma_bd1_7},
      {"shim_dma_bd1_8", aie2ps::shim_dma_bd1_8},
      {"shim_dma_bd2_0", aie2ps::shim_dma_bd2_0},
      {"shim_dma_bd2_1", aie2ps::shim_dma_bd2_1},
      {"shim_dma_bd2_2", aie2ps::shim_dma_bd2_2},
      {"shim_dma_bd2_3", aie2ps::shim_dma_bd2_3},
      {"shim_dma_bd2_4", aie2ps::shim_dma_bd2_4},
      {"shim_dma_bd2_5", aie2ps::shim_dma_bd2_5},
      {"shim_dma_bd2_6", aie2ps::shim_dma_bd2_6},
      {"shim_dma_bd2_7", aie2ps::shim_dma_bd2_7},
      {"shim_dma_bd2_8", aie2ps::shim_dma_bd2_8},
      {"shim_dma_bd3_0", aie2ps::shim_dma_bd3_0},
      {"shim_dma_bd3_1", aie2ps::shim_dma_bd3_1},
      {"shim_dma_bd3_2", aie2ps::shim_dma_bd3_2},
      {"shim_dma_bd3_3", aie2ps::shim_dma_bd3_3},
      {"shim_dma_bd3_4", aie2ps::shim_dma_bd3_4},
      {"shim_dma_bd3_5", aie2ps::shim_dma_bd3_5},
      {"shim_dma_bd3_6", aie2ps::shim_dma_bd3_6},
      {"shim_dma_bd3_7", aie2ps::shim_dma_bd3_7},
      {"shim_dma_bd3_8", aie2ps::shim_dma_bd3_8},
      {"shim_dma_bd4_0", aie2ps::shim_dma_bd4_0},
      {"shim_dma_bd4_1", aie2ps::shim_dma_bd4_1},
      {"shim_dma_bd4_2", aie2ps::shim_dma_bd4_2},
      {"shim_dma_bd4_3", aie2ps::shim_dma_bd4_3},
      {"shim_dma_bd4_4", aie2ps::shim_dma_bd4_4},
      {"shim_dma_bd4_5", aie2ps::shim_dma_bd4_5},
      {"shim_dma_bd4_6", aie2ps::shim_dma_bd4_6},
      {"shim_dma_bd4_7", aie2ps::shim_dma_bd4_7},
      {"shim_dma_bd4_8", aie2ps::shim_dma_bd4_8},
      {"shim_dma_bd5_0", aie2ps::shim_dma_bd5_0},
      {"shim_dma_bd5_1", aie2ps::shim_dma_bd5_1},
      {"shim_dma_bd5_2", aie2ps::shim_dma_bd5_2},
      {"shim_dma_bd5_3", aie2ps::shim_dma_bd5_3},
      {"shim_dma_bd5_4", aie2ps::shim_dma_bd5_4},
      {"shim_dma_bd5_5", aie2ps::shim_dma_bd5_5},
      {"shim_dma_bd5_6", aie2ps::shim_dma_bd5_6},
      {"shim_dma_bd5_7", aie2ps::shim_dma_bd5_7},
      {"shim_dma_bd5_8", aie2ps::shim_dma_bd5_8},
      {"shim_dma_bd6_0", aie2ps::shim_dma_bd6_0},
      {"shim_dma_bd6_1", aie2ps::shim_dma_bd6_1},
      {"shim_dma_bd6_2", aie2ps::shim_dma_bd6_2},
      {"shim_dma_bd6_3", aie2ps::shim_dma_bd6_3},
      {"shim_dma_bd6_4", aie2ps::shim_dma_bd6_4},
      {"shim_dma_bd6_5", aie2ps::shim_dma_bd6_5},
      {"shim_dma_bd6_6", aie2ps::shim_dma_bd6_6},
      {"shim_dma_bd6_7", aie2ps::shim_dma_bd6_7},
      {"shim_dma_bd6_8", aie2ps::shim_dma_bd6_8},
      {"shim_dma_bd7_0", aie2ps::shim_dma_bd7_0},
      {"shim_dma_bd7_1", aie2ps::shim_dma_bd7_1},
      {"shim_dma_bd7_2", aie2ps::shim_dma_bd7_2},
      {"shim_dma_bd7_3", aie2ps::shim_dma_bd7_3},
      {"shim_dma_bd7_4", aie2ps::shim_dma_bd7_4},
      {"shim_dma_bd7_5", aie2ps::shim_dma_bd7_5},
      {"shim_dma_bd7_6", aie2ps::shim_dma_bd7_6},
      {"shim_dma_bd7_7", aie2ps::shim_dma_bd7_7},
      {"shim_dma_bd7_8", aie2ps::shim_dma_bd7_8},
      {"shim_dma_bd8_0", aie2ps::shim_dma_bd8_0},
      {"shim_dma_bd8_1", aie2ps::shim_dma_bd8_1},
      {"shim_dma_bd8_2", aie2ps::shim_dma_bd8_2},
      {"shim_dma_bd8_3", aie2ps::shim_dma_bd8_3},
      {"shim_dma_bd8_4", aie2ps::shim_dma_bd8_4},
      {"shim_dma_bd8_5", aie2ps::shim_dma_bd8_5},
      {"shim_dma_bd8_6", aie2ps::shim_dma_bd8_6},
      {"shim_dma_bd8_7", aie2ps::shim_dma_bd8_7},
      {"shim_dma_bd8_8", aie2ps::shim_dma_bd8_8},
      {"shim_dma_bd9_0", aie2ps::shim_dma_bd9_0},
      {"shim_dma_bd9_1", aie2ps::shim_dma_bd9_1},
      {"shim_dma_bd9_2", aie2ps::shim_dma_bd9_2},
      {"shim_dma_bd9_3", aie2ps::shim_dma_bd9_3},
      {"shim_dma_bd9_4", aie2ps::shim_dma_bd9_4},
      {"shim_dma_bd9_5", aie2ps::shim_dma_bd9_5},
      {"shim_dma_bd9_6", aie2ps::shim_dma_bd9_6},
      {"shim_dma_bd9_7", aie2ps::shim_dma_bd9_7},
      {"shim_dma_bd9_8", aie2ps::shim_dma_bd9_8},
      {"shim_dma_bd10_0", aie2ps::shim_dma_bd10_0},
      {"shim_dma_bd10_1", aie2ps::shim_dma_bd10_1},
      {"shim_dma_bd10_2", aie2ps::shim_dma_bd10_2},
      {"shim_dma_bd10_3", aie2ps::shim_dma_bd10_3},
      {"shim_dma_bd10_4", aie2ps::shim_dma_bd10_4},
      {"shim_dma_bd10_5", aie2ps::shim_dma_bd10_5},
      {"shim_dma_bd10_6", aie2ps::shim_dma_bd10_6},
      {"shim_dma_bd10_7", aie2ps::shim_dma_bd10_7},
      {"shim_dma_bd10_8", aie2ps::shim_dma_bd10_8},
      {"shim_dma_bd11_0", aie2ps::shim_dma_bd11_0},
      {"shim_dma_bd11_1", aie2ps::shim_dma_bd11_1},
      {"shim_dma_bd11_2", aie2ps::shim_dma_bd11_2},
      {"shim_dma_bd11_3", aie2ps::shim_dma_bd11_3},
      {"shim_dma_bd11_4", aie2ps::shim_dma_bd11_4},
      {"shim_dma_bd11_5", aie2ps::shim_dma_bd11_5},
      {"shim_dma_bd11_6", aie2ps::shim_dma_bd11_6},
      {"shim_dma_bd11_7", aie2ps::shim_dma_bd11_7},
      {"shim_dma_bd11_8", aie2ps::shim_dma_bd11_8},
      {"shim_dma_bd12_0", aie2ps::shim_dma_bd12_0},
      {"shim_dma_bd12_1", aie2ps::shim_dma_bd12_1},
      {"shim_dma_bd12_2", aie2ps::shim_dma_bd12_2},
      {"shim_dma_bd12_3", aie2ps::shim_dma_bd12_3},
      {"shim_dma_bd12_4", aie2ps::shim_dma_bd12_4},
      {"shim_dma_bd12_5", aie2ps::shim_dma_bd12_5},
      {"shim_dma_bd12_6", aie2ps::shim_dma_bd12_6},
      {"shim_dma_bd12_7", aie2ps::shim_dma_bd12_7},
      {"shim_dma_bd12_8", aie2ps::shim_dma_bd12_8},
      {"shim_dma_bd13_0", aie2ps::shim_dma_bd13_0},
      {"shim_dma_bd13_1", aie2ps::shim_dma_bd13_1},
      {"shim_dma_bd13_2", aie2ps::shim_dma_bd13_2},
      {"shim_dma_bd13_3", aie2ps::shim_dma_bd13_3},
      {"shim_dma_bd13_4", aie2ps::shim_dma_bd13_4},
      {"shim_dma_bd13_5", aie2ps::shim_dma_bd13_5},
      {"shim_dma_bd13_6", aie2ps::shim_dma_bd13_6},
      {"shim_dma_bd13_7", aie2ps::shim_dma_bd13_7},
      {"shim_dma_bd13_8", aie2ps::shim_dma_bd13_8},
      {"shim_dma_bd14_0", aie2ps::shim_dma_bd14_0},
      {"shim_dma_bd14_1", aie2ps::shim_dma_bd14_1},
      {"shim_dma_bd14_2", aie2ps::shim_dma_bd14_2},
      {"shim_dma_bd14_3", aie2ps::shim_dma_bd14_3},
      {"shim_dma_bd14_4", aie2ps::shim_dma_bd14_4},
      {"shim_dma_bd14_5", aie2ps::shim_dma_bd14_5},
      {"shim_dma_bd14_6", aie2ps::shim_dma_bd14_6},
      {"shim_dma_bd14_7", aie2ps::shim_dma_bd14_7},
      {"shim_dma_bd14_8", aie2ps::shim_dma_bd14_8},
      {"shim_dma_bd15_0", aie2ps::shim_dma_bd15_0},
      {"shim_dma_bd15_1", aie2ps::shim_dma_bd15_1},
      {"shim_dma_bd15_2", aie2ps::shim_dma_bd15_2},
      {"shim_dma_bd15_3", aie2ps::shim_dma_bd15_3},
      {"shim_dma_bd15_4", aie2ps::shim_dma_bd15_4},
      {"shim_dma_bd15_5", aie2ps::shim_dma_bd15_5},
      {"shim_dma_bd15_6", aie2ps::shim_dma_bd15_6},
      {"shim_dma_bd15_7", aie2ps::shim_dma_bd15_7},
      {"shim_dma_bd15_8", aie2ps::shim_dma_bd15_8},
      {"shim_dma_s2mm_0_ctrl", aie2ps::shim_dma_s2mm_0_ctrl},
      {"shim_dma_s2mm_0_task_queue", aie2ps::shim_dma_s2mm_0_task_queue},
      {"shim_dma_s2mm_1_ctrl", aie2ps::shim_dma_s2mm_1_ctrl},
      {"shim_dma_s2mm_1_task_queue", aie2ps::shim_dma_s2mm_1_task_queue},
      {"shim_dma_mm2s_0_ctrl", aie2ps::shim_dma_mm2s_0_ctrl},
      {"shim_dma_mm2s_0_task_queue", aie2ps::shim_dma_mm2s_0_task_queue},
      {"shim_dma_mm2s_1_ctrl", aie2ps::shim_dma_mm2s_1_ctrl},
      {"shim_dma_mm2s_1_task_queue", aie2ps::shim_dma_mm2s_1_task_queue},
      {"shim_dma_s2mm_status_0", aie2ps::shim_dma_s2mm_status_0},
      {"shim_dma_s2mm_status_1", aie2ps::shim_dma_s2mm_status_1},
      {"shim_dma_mm2s_status_0", aie2ps::shim_dma_mm2s_status_0},
      {"shim_dma_mm2s_status_1", aie2ps::shim_dma_mm2s_status_1},
      {"shim_dma_s2mm_current_write_count_0", aie2ps::shim_dma_s2mm_current_write_count_0},
      {"shim_dma_s2mm_current_write_count_1", aie2ps::shim_dma_s2mm_current_write_count_1},
      {"shim_dma_s2mm_fot_count_fifo_pop_0", aie2ps::shim_dma_s2mm_fot_count_fifo_pop_0},
      {"shim_dma_s2mm_fot_count_fifo_pop_1", aie2ps::shim_dma_s2mm_fot_count_fifo_pop_1},
      {"shim_dma_mm2s_0_response_fifo_parity_error_injection", aie2ps::shim_dma_mm2s_0_response_fifo_parity_error_injection},
      {"shim_dma_mm2s_1_response_fifo_parity_error_injection", aie2ps::shim_dma_mm2s_1_response_fifo_parity_error_injection},
      {"shim_dma_pause", aie2ps::shim_dma_pause},
      {"shim_lock_request", aie2ps::shim_lock_request},
      {"shim_performance_control0", aie2ps::shim_performance_control0},
      {"shim_performance_control1", aie2ps::shim_performance_control1},
      {"shim_performance_control2", aie2ps::shim_performance_control2},
      {"shim_performance_control3", aie2ps::shim_performance_control3},
      {"shim_performance_control4", aie2ps::shim_performance_control4},
      {"shim_performance_control5", aie2ps::shim_performance_control5},
      {"shim_performance_counter0", aie2ps::shim_performance_counter0},
      {"shim_performance_counter1", aie2ps::shim_performance_counter1},
      {"shim_performance_counter2", aie2ps::shim_performance_counter2},
      {"shim_performance_counter3", aie2ps::shim_performance_counter3},
      {"shim_performance_counter4", aie2ps::shim_performance_counter4},
      {"shim_performance_counter5", aie2ps::shim_performance_counter5},
      {"shim_performance_counter0_event_value", aie2ps::shim_performance_counter0_event_value},
      {"shim_performance_counter1_event_value", aie2ps::shim_performance_counter1_event_value},
      {"shim_event_generate", aie2ps::shim_event_generate},
      {"shim_event_broadcast_a_0", aie2ps::shim_event_broadcast_a_0},
      {"shim_event_broadcast_a_1", aie2ps::shim_event_broadcast_a_1},
      {"shim_event_broadcast_a_2", aie2ps::shim_event_broadcast_a_2},
      {"shim_event_broadcast_a_3", aie2ps::shim_event_broadcast_a_3},
      {"shim_event_broadcast_a_4", aie2ps::shim_event_broadcast_a_4},
      {"shim_event_broadcast_a_5", aie2ps::shim_event_broadcast_a_5},
      {"shim_event_broadcast_a_6", aie2ps::shim_event_broadcast_a_6},
      {"shim_event_broadcast_a_7", aie2ps::shim_event_broadcast_a_7},
      {"shim_event_broadcast_a_8", aie2ps::shim_event_broadcast_a_8},
      {"shim_event_broadcast_a_9", aie2ps::shim_event_broadcast_a_9},
      {"shim_event_broadcast_a_10", aie2ps::shim_event_broadcast_a_10},
      {"shim_event_broadcast_a_11", aie2ps::shim_event_broadcast_a_11},
      {"shim_event_broadcast_a_12", aie2ps::shim_event_broadcast_a_12},
      {"shim_event_broadcast_a_13", aie2ps::shim_event_broadcast_a_13},
      {"shim_event_broadcast_a_14", aie2ps::shim_event_broadcast_a_14},
      {"shim_event_broadcast_a_15", aie2ps::shim_event_broadcast_a_15},
      {"shim_event_broadcast_a_block_south_set", aie2ps::shim_event_broadcast_a_block_south_set},
      {"shim_event_broadcast_a_block_south_clr", aie2ps::shim_event_broadcast_a_block_south_clr},
      {"shim_event_broadcast_a_block_south_value", aie2ps::shim_event_broadcast_a_block_south_value},
      {"shim_event_broadcast_a_block_west_set", aie2ps::shim_event_broadcast_a_block_west_set},
      {"shim_event_broadcast_a_block_west_clr", aie2ps::shim_event_broadcast_a_block_west_clr},
      {"shim_event_broadcast_a_block_west_value", aie2ps::shim_event_broadcast_a_block_west_value},
      {"shim_event_broadcast_a_block_north_set", aie2ps::shim_event_broadcast_a_block_north_set},
      {"shim_event_broadcast_a_block_north_clr", aie2ps::shim_event_broadcast_a_block_north_clr},
      {"shim_event_broadcast_a_block_north_value", aie2ps::shim_event_broadcast_a_block_north_value},
      {"shim_event_broadcast_a_block_east_set", aie2ps::shim_event_broadcast_a_block_east_set},
      {"shim_event_broadcast_a_block_east_clr", aie2ps::shim_event_broadcast_a_block_east_clr},
      {"shim_event_broadcast_a_block_east_value", aie2ps::shim_event_broadcast_a_block_east_value},
      {"shim_trace_control0", aie2ps::shim_trace_control0},
      {"shim_trace_control1", aie2ps::shim_trace_control1},
      {"shim_trace_status", aie2ps::shim_trace_status},
      {"shim_trace_event0", aie2ps::shim_trace_event0},
      {"shim_trace_event1", aie2ps::shim_trace_event1},
      {"shim_timer_trig_event_low_value", aie2ps::shim_timer_trig_event_low_value},
      {"shim_timer_trig_event_high_value", aie2ps::shim_timer_trig_event_high_value},
      {"shim_timer_low", aie2ps::shim_timer_low},
      {"shim_timer_high", aie2ps::shim_timer_high},
      {"shim_event_status0", aie2ps::shim_event_status0},
      {"shim_event_status1", aie2ps::shim_event_status1},
      {"shim_event_status2", aie2ps::shim_event_status2},
      {"shim_event_status3", aie2ps::shim_event_status3},
      {"shim_event_group_dma_enable", aie2ps::shim_event_group_dma_enable},
      {"shim_stream_switch_event_port_selection_0", aie2ps::shim_stream_switch_event_port_selection_0},
      {"shim_stream_switch_event_port_selection_1", aie2ps::shim_stream_switch_event_port_selection_1},
      {"npi_me_isr", aie2ps::npi_me_isr},
      {"npi_me_itr", aie2ps::npi_me_itr},
      {"npi_me_imr0", aie2ps::npi_me_imr0},
      {"npi_me_ier0", aie2ps::npi_me_ier0},
      {"npi_me_idr0", aie2ps::npi_me_idr0},
      {"npi_me_imr1", aie2ps::npi_me_imr1},
      {"npi_me_ier1", aie2ps::npi_me_ier1},
      {"npi_me_idr1", aie2ps::npi_me_idr1},
      {"npi_me_imr2", aie2ps::npi_me_imr2},
      {"npi_me_ier2", aie2ps::npi_me_ier2},
      {"npi_me_idr2", aie2ps::npi_me_idr2},
      {"npi_me_imr3", aie2ps::npi_me_imr3},
      {"npi_me_ier3", aie2ps::npi_me_ier3},
      {"npi_me_idr3", aie2ps::npi_me_idr3},
      {"npi_me_ior", aie2ps::npi_me_ior},
      {"npi_me_pll_status", aie2ps::npi_me_pll_status},
      {"npi_me_secure_reg", aie2ps::npi_me_secure_reg},
      {"uc_base_address", aie2ps::uc_base_address},
      {"uc_mdm_dbg_ctrl_status", aie2ps::uc_mdm_dbg_ctrl_status},
      {"uc_mdm_dbg_data", aie2ps::uc_mdm_dbg_data},
      {"uc_mdm_dbg_lock", aie2ps::uc_mdm_dbg_lock},
      {"uc_mdm_pcctrlr", aie2ps::uc_mdm_pcctrlr},
      {"uc_mdm_pccmdr", aie2ps::uc_mdm_pccmdr},
      {"uc_mdm_pcsr", aie2ps::uc_mdm_pcsr},
      {"uc_mdm_pcdrr", aie2ps::uc_mdm_pcdrr},
      {"uc_mdm_pcwr", aie2ps::uc_mdm_pcwr},
      {"uc_core_status", aie2ps::uc_core_status},
      {"uc_core_control", aie2ps::uc_core_control},
      {"uc_core_interrupt_status", aie2ps::uc_core_interrupt_status},
      {"uc_module_aximm_offset", aie2ps::uc_module_aximm_offset},
      {"uc_module_axi_mm_outstanding_transactions", aie2ps::uc_module_axi_mm_outstanding_transactions},
      {"uc_memory_zeroization", aie2ps::uc_memory_zeroization},
      {"uc_memory_privileged", aie2ps::uc_memory_privileged},
      {"uc_memory_dm_ecc_scrubbing_period", aie2ps::uc_memory_dm_ecc_scrubbing_period},
      {"uc_memory_dm_ecc_error_generation", aie2ps::uc_memory_dm_ecc_error_generation},
      {"uc_dma_dm2mm_status", aie2ps::uc_dma_dm2mm_status},
      {"uc_dma_dm2mm_control", aie2ps::uc_dma_dm2mm_control},
      {"uc_dma_dm2mm_axi_control", aie2ps::uc_dma_dm2mm_axi_control},
      {"uc_dma_mm2dm_status", aie2ps::uc_dma_mm2dm_status},
      {"uc_dma_mm2dm_control", aie2ps::uc_dma_mm2dm_control},
      {"uc_dma_mm2dm_axi_control", aie2ps::uc_dma_mm2dm_axi_control},
      {"uc_dma_pause", aie2ps::uc_dma_pause}
    };
  }

  void populateRegValueToNameMap() {
    regValueToName=  {
      {0x000940a0, "mem_event_broadcast_b_block_west_set"},
      {0x00009320, "shim_dma_s2mm_status_0"},
      {0x00009324, "shim_dma_s2mm_status_1"},
      {0x000a061c, "mem_dma_s2mm_3_start_queue"},
      {0x00030330, "cm_core_bmll3_part4"},
      {0x00030310, "cm_core_bmll3_part2"},
      {0x00030320, "cm_core_bmll3_part3"},
      {0x00030300, "cm_core_bmll3_part1"},
      {0x000b5580, "uc_mdm_pcdrr"},
      {0x000c0230, "mem_lock35_value"},
      {0x00002104, "shim_mux_config"},
      {0x000a0614, "mem_dma_s2mm_2_start_queue"},
      {0x000340f4, "shim_timer_trig_event_high_value"},
      {0x00034068, "shim_event_broadcast_a_block_west_value"},
      {0x0001d10c, "mm_dma_bd8_3"},
      {0x000b0228, "mem_stream_switch_slave_dma_2_slot2"},
      {0x0001d104, "mm_dma_bd8_1"},
      {0x000000e0, "shim_lock14_value"},
      {0x0001d100, "mm_dma_bd8_0"},
      {0x0003f2d4, "cm_stream_switch_slave_west_2_slot1"},
      {0x0003f2d0, "cm_stream_switch_slave_west_2_slot0"},
      {0x0003f2dc, "cm_stream_switch_slave_west_2_slot3"},
      {0x0003f2d8, "cm_stream_switch_slave_west_2_slot2"},
      {0x000c0000, "mem_lock0_value"},
      {0x000c0130, "mem_lock19_value"},
      {0x000b55c0, "uc_mdm_pcwr"},
      {0x000a0254, "mem_dma_bd18_5"},
      {0x000a0250, "mem_dma_bd18_4"},
      {0x000a025c, "mem_dma_bd18_7"},
      {0x000a0258, "mem_dma_bd18_6"},
      {0x000a0244, "mem_dma_bd18_1"},
      {0x000a0240, "mem_dma_bd18_0"},
      {0x000a024c, "mem_dma_bd18_3"},
      {0x000a0248, "mem_dma_bd18_2"},
      {0x00032bf0, "cm_core_dc7"},
      {0x0000930c, "shim_dma_s2mm_1_task_queue"},
      {0x00030750, "cm_core_bmlh7_part2"},
      {0x00032b80, "cm_core_dc0"},
      {0x000c0110, "uc_dma_mm2dm_status"},
      {0x000c02c0, "mem_lock44_value"},
      {0x00009310, "shim_dma_mm2s_0_ctrl"},
      {0x0001de0c, "mm_dma_s2mm_1_start_queue"},
      {0x00030740, "cm_core_bmlh7_part1"},
      {0x000c0038, "uc_memory_dm_ecc_scrubbing_period"},
      {0x00034508, "cm_event_group_core_stall_enable"},
      {0x00094214, "mem_event_status5"},
      {0x00094200, "mem_event_status0"},
      {0x00094204, "mem_event_status1"},
      {0x00094208, "mem_event_status2"},
      {0x0009420c, "mem_event_status3"},
      {0x0003ff38, "cm_stream_switch_adaptive_clock_gate_abort_period"},
      {0x00094500, "mem_event_group_0_enable"},
      {0x0003f104, "cm_stream_switch_slave_config_dma_0"},
      {0x000c01c0, "mem_lock28_value"},
      {0x00030480, "cm_core_bmhl4_part1"},
      {0x00014514, "mm_event_group_error_enable"},
      {0x000304a0, "cm_core_bmhl4_part3"},
      {0x00030490, "cm_core_bmhl4_part2"},
      {0x000c00c0, "mem_lock12_value"},
      {0x000304b0, "cm_core_bmhl4_part4"},
      {0x00009300, "shim_dma_s2mm_0_ctrl"},
      {0x00014074, "mm_event_broadcast_block_north_clr"},
      {0x00034068, "cm_event_broadcast_block_west_value"},
      {0x000340f0, "cm_timer_trig_event_low_value"},
      {0x00000000, "shim_lock0_value"},
      {0x000307d0, "cm_core_bmhh7_part2"},
      {0x000307e0, "cm_core_bmhh7_part3"},
      {0x000307c0, "cm_core_bmhh7_part1"},
      {0x000307f0, "cm_core_bmhh7_part4"},
      {0x000b02d0, "mem_stream_switch_slave_north_0_slot0"},
      {0x000b02d4, "mem_stream_switch_slave_north_0_slot1"},
      {0x000b02d8, "mem_stream_switch_slave_north_0_slot2"},
      {0x000b02dc, "mem_stream_switch_slave_north_0_slot3"},
      {0x0003f278, "cm_stream_switch_slave_south_2_slot2"},
      {0x0003f27c, "cm_stream_switch_slave_south_2_slot3"},
      {0x0003f270, "cm_stream_switch_slave_south_2_slot0"},
      {0x0003f274, "cm_stream_switch_slave_south_2_slot1"},
      {0x0001f050, "mm_lock5_value"},
      {0x00031a80, "cm_core_x10_part1"},
      {0x00031aa0, "cm_core_x10_part3"},
      {0x00031a90, "cm_core_x10_part2"},
      {0x00031ab0, "cm_core_x10_part4"},
      {0x00002108, "shim_demux_config"},
      {0x000940f8, "mem_timer_low"},
      {0x0001d1f0, "mm_dma_bd15_4"},
      {0x0001d1f4, "mm_dma_bd15_5"},
      {0x000a063c, "mem_dma_mm2s_1_start_queue"},
      {0x0001d1e0, "mm_dma_bd15_0"},
      {0x0001d1e4, "mm_dma_bd15_1"},
      {0x0001d1e8, "mm_dma_bd15_2"},
      {0x0001d1ec, "mm_dma_bd15_3"},
      {0x00094400, "mem_combo_event_inputs"},
      {0x000b5440, "uc_mdm_pcctrlr"},
      {0x0003f100, "cm_stream_switch_slave_config_aie_core0"},
      {0x000c0200, "mem_lock32_value"},
      {0x000340d8, "shim_trace_status"},
      {0x0000004c, "npi_me_idr1"},
      {0x00009340, "shim_dma_mm2s_0_response_fifo_parity_error_injection"},
      {0x00014070, "mm_event_broadcast_block_north_set"},
      {0x0001d160, "mm_dma_bd11_0"},
      {0x0001d164, "mm_dma_bd11_1"},
      {0x0001d168, "mm_dma_bd11_2"},
      {0x0001d16c, "mm_dma_bd11_3"},
      {0x0001d170, "mm_dma_bd11_4"},
      {0x0001d174, "mm_dma_bd11_5"},
      {0x0001d1b0, "mm_dma_bd13_4"},
      {0x0001d1b4, "mm_dma_bd13_5"},
      {0x0001d1a8, "mm_dma_bd13_2"},
      {0x0001d1ac, "mm_dma_bd13_3"},
      {0x0001d1a0, "mm_dma_bd13_0"},
      {0x0001d1a4, "mm_dma_bd13_1"},
      {0x00094514, "mem_event_group_memory_conflict_enable"},
      {0x000140e0, "mm_trace_event0"},
      {0x000140e4, "mm_trace_event1"},
      {0x000a0230, "mem_dma_bd17_4"},
      {0x000a0234, "mem_dma_bd17_5"},
      {0x000a0228, "mem_dma_bd17_2"},
      {0x000a022c, "mem_dma_bd17_3"},
      {0x000a0220, "mem_dma_bd17_0"},
      {0x000a0224, "mem_dma_bd17_1"},
      {0x000a01e0, "mem_dma_bd15_0"},
      {0x000a01e4, "mem_dma_bd15_1"},
      {0x000a01e8, "mem_dma_bd15_2"},
      {0x000a01ec, "mem_dma_bd15_3"},
      {0x000a01f0, "mem_dma_bd15_4"},
      {0x000a01f4, "mem_dma_bd15_5"},
      {0x000a01f8, "mem_dma_bd15_6"},
      {0x000a01fc, "mem_dma_bd15_7"},
      {0x000a06dc, "mem_dma_s2mm_fot_count_fifo_pop_5"},
      {0x000a06d8, "mem_dma_s2mm_fot_count_fifo_pop_4"},
      {0x000a06cc, "mem_dma_s2mm_fot_count_fifo_pop_1"},
      {0x0001f0d0, "mm_lock13_value"},
      {0x000a06d4, "mem_dma_s2mm_fot_count_fifo_pop_3"},
      {0x000a06d0, "mem_dma_s2mm_fot_count_fifo_pop_2"},
      {0x0003f230, "cm_stream_switch_slave_tile_ctrl_slot0"},
      {0x0003f234, "cm_stream_switch_slave_tile_ctrl_slot1"},
      {0x0003f238, "cm_stream_switch_slave_tile_ctrl_slot2"},
      {0x0003f23c, "cm_stream_switch_slave_tile_ctrl_slot3"},
      {0x0003f14c, "cm_stream_switch_slave_config_east_0"},
      {0x0003f150, "cm_stream_switch_slave_config_east_1"},
      {0x0003f154, "cm_stream_switch_slave_config_east_2"},
      {0x0003f158, "cm_stream_switch_slave_config_east_3"},
      {0x000a0504, "mem_dma_bd40_1"},
      {0x00094070, "mem_event_broadcast_a_block_north_set"},
      {0x000a01ac, "mem_dma_bd13_3"},
      {0x000a01a0, "mem_dma_bd13_0"},
      {0x000a01a4, "mem_dma_bd13_1"},
      {0x000a01b8, "mem_dma_bd13_6"},
      {0x000a01bc, "mem_dma_bd13_7"},
      {0x000a01b0, "mem_dma_bd13_4"},
      {0x000a01b4, "mem_dma_bd13_5"},
      {0x000a0170, "mem_dma_bd11_4"},
      {0x000a0174, "mem_dma_bd11_5"},
      {0x000a017c, "mem_dma_bd11_7"},
      {0x000a0160, "mem_dma_bd11_0"},
      {0x000a0164, "mem_dma_bd11_1"},
      {0x000a0168, "mem_dma_bd11_2"},
      {0x000a016c, "mem_dma_bd11_3"},
      {0x000940fc, "mem_timer_high"},
      {0x0003f374, "cm_stream_switch_slave_aie_trace_slot1"},
      {0x0003f370, "cm_stream_switch_slave_aie_trace_slot0"},
      {0x0003f37c, "cm_stream_switch_slave_aie_trace_slot3"},
      {0x0003f378, "cm_stream_switch_slave_aie_trace_slot2"},
      {0x0001d050, "mm_dma_bd2_4"},
      {0x00037524, "cm_performance_counter1"},
      {0x00037520, "cm_performance_counter0"},
      {0x0003752c, "cm_performance_counter3"},
      {0x00037528, "cm_performance_counter2"},
      {0x000b0018, "mem_stream_switch_master_config_tile_ctrl"},
      {0x000940c8, "mem_event_broadcast_b_block_east_value"},
      {0x0003f2bc, "cm_stream_switch_slave_west_0_slot3"},
      {0x0003f2b8, "cm_stream_switch_slave_west_0_slot2"},
      {0x0003f2b4, "cm_stream_switch_slave_west_0_slot1"},
      {0x0003f2b0, "cm_stream_switch_slave_west_0_slot0"},
      {0x000940d4, "mem_trace_control1"},
      {0x000940d0, "mem_trace_control0"},
      {0x0001f030, "mm_lock3_value"},
      {0x00014028, "mm_event_broadcast6"},
      {0x0001402c, "mm_event_broadcast7"},
      {0x00030660, "cm_core_bmlh6_part3"},
      {0x00030650, "cm_core_bmlh6_part2"},
      {0x00030640, "cm_core_bmlh6_part1"},
      {0x000301a0, "cm_core_bmhl1_part3"},
      {0x000301b0, "cm_core_bmhl1_part4"},
      {0x00030670, "cm_core_bmlh6_part4"},
      {0x00014018, "mm_event_broadcast2"},
      {0x00000040, "shim_lock4_value"},
      {0x0001401c, "mm_event_broadcast3"},
      {0x00014010, "mm_event_broadcast0"},
      {0x00014014, "mm_event_broadcast1"},
      {0x00000090, "shim_lock9_value"},
      {0x00009250, "shim_dma_bd12_4"},
      {0x0003ff30, "cm_tile_control_packet_handler_status"},
      {0x00034030, "cm_event_broadcast8"},
      {0x00034034, "cm_event_broadcast9"},
      {0x0003f368, "cm_stream_switch_slave_east_3_slot2"},
      {0x0003f36c, "cm_stream_switch_slave_east_3_slot3"},
      {0x0003f360, "cm_stream_switch_slave_east_3_slot0"},
      {0x0003f364, "cm_stream_switch_slave_east_3_slot1"},
      {0x00034010, "cm_event_broadcast0"},
      {0x00034014, "cm_event_broadcast1"},
      {0x00034018, "cm_event_broadcast2"},
      {0x0003401c, "cm_event_broadcast3"},
      {0x00091084, "mem_performance_counter1_event_value"},
      {0x000305d0, "cm_core_bmhh5_part2"},
      {0x000305e0, "cm_core_bmhh5_part3"},
      {0x000305f0, "cm_core_bmhh5_part4"},
      {0x00000040, "npi_me_idr0"},
      {0x00000064, "npi_me_idr3"},
      {0x000091f0, "shim_dma_bd10_4"},
      {0x00000058, "npi_me_idr2"},
      {0x000091f4, "shim_dma_bd10_5"},
      {0x000091f8, "shim_dma_bd10_6"},
      {0x0003f02c, "cm_stream_switch_master_config_west2"},
      {0x0003f030, "cm_stream_switch_master_config_west3"},
      {0x0003f024, "cm_stream_switch_master_config_west0"},
      {0x0003f028, "cm_stream_switch_master_config_west1"},
      {0x0003f144, "cm_stream_switch_slave_config_north_2"},
      {0x00000030, "npi_me_isr"},
      {0x000a05c0, "mem_dma_bd46_0"},
      {0x000a05c4, "mem_dma_bd46_1"},
      {0x000a05c8, "mem_dma_bd46_2"},
      {0x000a05cc, "mem_dma_bd46_3"},
      {0x000a05d0, "mem_dma_bd46_4"},
      {0x000a05d4, "mem_dma_bd46_5"},
      {0x000a05d8, "mem_dma_bd46_6"},
      {0x000a05dc, "mem_dma_bd46_7"},
      {0x00031a40, "cm_core_x9_part1"},
      {0x00014100, "mm_watchpoint0"},
      {0x00014104, "mm_watchpoint1"},
      {0x00094068, "mem_event_broadcast_a_block_west_value"},
      {0x0001de18, "mm_dma_mm2s_1_ctrl"},
      {0x000140f0, "mm_timer_trig_event_low_value"},
      {0x000b0808, "mem_stream_switch_deterministic_merge_arb0_ctrl"},
      {0x0003f28c, "cm_stream_switch_slave_south_3_slot3"},
      {0x0003f288, "cm_stream_switch_slave_south_3_slot2"},
      {0x000b0030, "mem_stream_switch_master_config_north1"},
      {0x000b002c, "mem_stream_switch_master_config_north0"},
      {0x000b0040, "mem_stream_switch_master_config_north5"},
      {0x000b003c, "mem_stream_switch_master_config_north4"},
      {0x000140d4, "mm_trace_control1"},
      {0x000140d0, "mm_trace_control0"},
      {0x00034078, "shim_event_broadcast_a_block_north_value"},
      {0x00034054, "shim_event_broadcast_a_block_south_clr"},
      {0x00030220, "cm_core_bmll2_part3"},
      {0x00030210, "cm_core_bmll2_part2"},
      {0x00030200, "cm_core_bmll2_part1"},
      {0x00030230, "cm_core_bmll2_part4"},
      {0x00009338, "shim_dma_s2mm_fot_count_fifo_pop_0"},
      {0x0000933c, "shim_dma_s2mm_fot_count_fifo_pop_1"},
      {0x000c0114, "uc_dma_mm2dm_control"},
      {0x00014064, "mm_event_broadcast_block_west_clr"},
      {0x000c02e0, "mem_lock46_value"},
      {0x0001d114, "mm_dma_bd8_5"},
      {0x0001d110, "mm_dma_bd8_4"},
      {0x000b022c, "mem_stream_switch_slave_dma_2_slot3"},
      {0x0001d108, "mm_dma_bd8_2"},
      {0x000b0224, "mem_stream_switch_slave_dma_2_slot1"},
      {0x000b0220, "mem_stream_switch_slave_dma_2_slot0"},
      {0x00001004, "shim_interrupt_controller_2nd_level_enable"},
      {0x0003404c, "shim_event_broadcast_a_15"},
      {0x0003758c, "cm_performance_counter3_event_value"},
      {0x00030570, "cm_core_bmlh5_part4"},
      {0x000b0018, "uc_mdm_dbg_lock"},
      {0x00030540, "cm_core_bmlh5_part1"},
      {0x00030550, "cm_core_bmlh5_part2"},
      {0x00030560, "cm_core_bmlh5_part3"},
      {0x00091088, "mem_performance_counter2_event_value"},
      {0x0003f108, "cm_stream_switch_slave_config_dma_1"},
      {0x000a052c, "mem_dma_bd41_3"},
      {0x000a0528, "mem_dma_bd41_2"},
      {0x000a0524, "mem_dma_bd41_1"},
      {0x000a0520, "mem_dma_bd41_0"},
      {0x000a053c, "mem_dma_bd41_7"},
      {0x000a0538, "mem_dma_bd41_6"},
      {0x000a0534, "mem_dma_bd41_5"},
      {0x000a0530, "mem_dma_bd41_4"},
      {0x000a0574, "mem_dma_bd43_5"},
      {0x000a0570, "mem_dma_bd43_4"},
      {0x000a057c, "mem_dma_bd43_7"},
      {0x000a0578, "mem_dma_bd43_6"},
      {0x000a0564, "mem_dma_bd43_1"},
      {0x000a0560, "mem_dma_bd43_0"},
      {0x000a056c, "mem_dma_bd43_3"},
      {0x000a0568, "mem_dma_bd43_2"},
      {0x000c0350, "mem_lock53_value"},
      {0x000000b0, "shim_lock11_value"},
      {0x000b0244, "mem_stream_switch_slave_dma_4_slot1"},
      {0x000b0240, "mem_stream_switch_slave_dma_4_slot0"},
      {0x000b024c, "mem_stream_switch_slave_dma_4_slot3"},
      {0x000b0248, "mem_stream_switch_slave_dma_4_slot2"},
      {0x000a05bc, "mem_dma_bd45_7"},
      {0x000a05b8, "mem_dma_bd45_6"},
      {0x000a05b4, "mem_dma_bd45_5"},
      {0x000a05b0, "mem_dma_bd45_4"},
      {0x000a05ac, "mem_dma_bd45_3"},
      {0x000a05a8, "mem_dma_bd45_2"},
      {0x000a05a4, "mem_dma_bd45_1"},
      {0x000a05a0, "mem_dma_bd45_0"},
      {0x000c03f0, "mem_lock63_value"},
      {0x00000050, "shim_lock5_value"},
      {0x00034084, "shim_event_broadcast_a_block_east_clr"},
      {0x0003f290, "cm_stream_switch_slave_south_4_slot0"},
      {0x0003f294, "cm_stream_switch_slave_south_4_slot1"},
      {0x0003f298, "cm_stream_switch_slave_south_4_slot2"},
      {0x0003f29c, "cm_stream_switch_slave_south_4_slot3"},
      {0x00094520, "mem_event_group_user_event_enable"},
      {0x00000034, "npi_me_itr"},
      {0x00030500, "cm_core_bmll5_part1"},
      {0x00030510, "cm_core_bmll5_part2"},
      {0x00030520, "cm_core_bmll5_part3"},
      {0x00030530, "cm_core_bmll5_part4"},
      {0x00000104, "shim_locks_event_selection_1"},
      {0x00000100, "shim_locks_event_selection_0"},
      {0x0000010c, "shim_locks_event_selection_3"},
      {0x000b02b4, "mem_stream_switch_slave_south_4_slot1"},
      {0x00000114, "shim_locks_event_selection_5"},
      {0x00000110, "shim_locks_event_selection_4"},
      {0x000319a0, "cm_core_x6_part3"},
      {0x00031990, "cm_core_x6_part2"},
      {0x00031980, "cm_core_x6_part1"},
      {0x000327b0, "cm_core_f11"},
      {0x000327a0, "cm_core_f10"},
      {0x000319b0, "cm_core_x6_part4"},
      {0x0001100c, "mm_performance_control2"},
      {0x00011010, "mm_performance_control3"},
      {0x00011000, "mm_performance_control0"},
      {0x00011008, "mm_performance_control1"},
      {0x0003f134, "cm_stream_switch_slave_config_west_2"},
      {0x0003f138, "cm_stream_switch_slave_config_west_3"},
      {0x0003f12c, "cm_stream_switch_slave_config_west_0"},
      {0x0003f130, "cm_stream_switch_slave_config_west_1"},
      {0x00014518, "mm_event_group_broadcast_enable"},
      {0x00034070, "shim_event_broadcast_a_block_north_set"},
      {0x00032480, "cm_core_ldfifol1_part1"},
      {0x00032490, "cm_core_ldfifol1_part2"},
      {0x000324a0, "cm_core_ldfifol1_part3"},
      {0x000324b0, "cm_core_ldfifol1_part4"},
      {0x00094210, "mem_event_status4"},
      {0x000092a4, "shim_dma_bd14_1"},
      {0x000a010c, "mem_dma_bd8_3"},
      {0x000a0108, "mem_dma_bd8_2"},
      {0x000a0104, "mem_dma_bd8_1"},
      {0x000a0100, "mem_dma_bd8_0"},
      {0x000a011c, "mem_dma_bd8_7"},
      {0x000a0118, "mem_dma_bd8_6"},
      {0x000a0114, "mem_dma_bd8_5"},
      {0x000a0110, "mem_dma_bd8_4"},
      {0x000092b4, "shim_dma_bd14_5"},
      {0x000a00c4, "mem_dma_bd6_1"},
      {0x000a00c0, "mem_dma_bd6_0"},
      {0x000a00cc, "mem_dma_bd6_3"},
      {0x000a00c8, "mem_dma_bd6_2"},
      {0x000a00d4, "mem_dma_bd6_5"},
      {0x000a00d0, "mem_dma_bd6_4"},
      {0x000a00dc, "mem_dma_bd6_7"},
      {0x000a00d8, "mem_dma_bd6_6"},
      {0x000940b0, "mem_event_broadcast_b_block_north_set"},
      {0x000c0118, "uc_dma_mm2dm_axi_control"},
      {0x0001d0f0, "mm_dma_bd7_4"},
      {0x0001d0f4, "mm_dma_bd7_5"},
      {0x0001d0e0, "mm_dma_bd7_0"},
      {0x0001d0e4, "mm_dma_bd7_1"},
      {0x0001d0e8, "mm_dma_bd7_2"},
      {0x0001d0ec, "mm_dma_bd7_3"},
      {0x000b02f8, "mem_stream_switch_slave_north_2_slot2"},
      {0x000b02fc, "mem_stream_switch_slave_north_2_slot3"},
      {0x000b02f0, "mem_stream_switch_slave_north_2_slot0"},
      {0x000b02f4, "mem_stream_switch_slave_north_2_slot1"},
      {0x0001d0a8, "mm_dma_bd5_2"},
      {0x0001d0ac, "mm_dma_bd5_3"},
      {0x0001d0a0, "mm_dma_bd5_0"},
      {0x0001d0a4, "mm_dma_bd5_1"},
      {0x0001d0b0, "mm_dma_bd5_4"},
      {0x0001d0b4, "mm_dma_bd5_5"},
      {0x00094518, "mem_event_group_error_enable"},
      {0x0001f020, "mm_lock2_value"},
      {0x00030080, "cm_core_bmhl0_part1"},
      {0x000300a0, "cm_core_bmhl0_part3"},
      {0x00030090, "cm_core_bmhl0_part2"},
      {0x000302b0, "cm_core_bmhl2_part4"},
      {0x000302a0, "cm_core_bmhl2_part3"},
      {0x00030290, "cm_core_bmhl2_part2"},
      {0x00030280, "cm_core_bmhl2_part1"},
      {0x00034060, "shim_event_broadcast_a_block_west_set"},
      {0x000a0028, "mem_dma_bd1_2"},
      {0x000a002c, "mem_dma_bd1_3"},
      {0x000a0020, "mem_dma_bd1_0"},
      {0x000a0024, "mem_dma_bd1_1"},
      {0x000a0038, "mem_dma_bd1_6"},
      {0x000a003c, "mem_dma_bd1_7"},
      {0x000a0030, "mem_dma_bd1_4"},
      {0x000a0034, "mem_dma_bd1_5"},
      {0x000340d0, "shim_trace_control0"},
      {0x000340d4, "shim_trace_control1"},
      {0x000a0610, "mem_dma_s2mm_2_ctrl"},
      {0x0001d064, "mm_dma_bd3_1"},
      {0x0001d068, "mm_dma_bd3_2"},
      {0x0001d06c, "mm_dma_bd3_3"},
      {0x0001d070, "mm_dma_bd3_4"},
      {0x0001d074, "mm_dma_bd3_5"},
      {0x00034074, "cm_event_broadcast_block_north_clr"},
      {0x0001d030, "mm_dma_bd1_4"},
      {0x0001d034, "mm_dma_bd1_5"},
      {0x0001d028, "mm_dma_bd1_2"},
      {0x0001d02c, "mm_dma_bd1_3"},
      {0x00038008, "cm_enable_events"},
      {0x0001d024, "mm_dma_bd1_1"},
      {0x000c0250, "mem_lock37_value"},
      {0x000b0110, "mem_stream_switch_slave_config_dma_4"},
      {0x000b0114, "mem_stream_switch_slave_config_dma_5"},
      {0x000b0100, "mem_stream_switch_slave_config_dma_0"},
      {0x000b0104, "mem_stream_switch_slave_config_dma_1"},
      {0x000b0108, "mem_stream_switch_slave_config_dma_2"},
      {0x00094508, "mem_event_group_dma_enable"},
      {0x00094028, "mem_event_broadcast6"},
      {0x0009402c, "mem_event_broadcast7"},
      {0x000326a0, "cm_core_eg10"},
      {0x00094024, "mem_event_broadcast5"},
      {0x00094018, "mem_event_broadcast2"},
      {0x0009401c, "mem_event_broadcast3"},
      {0x00094010, "mem_event_broadcast0"},
      {0x00094014, "mem_event_broadcast1"},
      {0x000b0020, "mem_stream_switch_master_config_south1"},
      {0x000b001c, "mem_stream_switch_master_config_south0"},
      {0x000b0028, "mem_stream_switch_master_config_south3"},
      {0x000b0024, "mem_stream_switch_master_config_south2"},
      {0x000a0060, "mem_dma_bd3_0"},
      {0x000a0064, "mem_dma_bd3_1"},
      {0x000a0068, "mem_dma_bd3_2"},
      {0x000a006c, "mem_dma_bd3_3"},
      {0x000c003c, "uc_memory_dm_ecc_error_generation"},
      {0x00038004, "cm_core_status"},
      {0x0001f120, "mm_locks_overflow"},
      {0x0001de08, "mm_dma_s2mm_1_ctrl"},
      {0x00030440, "cm_core_bmlh4_part1"},
      {0x00030460, "cm_core_bmlh4_part3"},
      {0x00030450, "cm_core_bmlh4_part2"},
      {0x00030470, "cm_core_bmlh4_part4"},
      {0x00003000, "shim_smid"},
      {0x000a0624, "mem_dma_s2mm_4_start_queue"},
      {0x000a06f4, "mem_dma_mm2s_5_constant_pad_value"},
      {0x0001f0b0, "mm_lock11_value"},
      {0x00034080, "cm_event_broadcast_block_east_set"},
      {0x0001df14, "mm_dma_mm2s_status_1"},
      {0x0001df10, "mm_dma_mm2s_status_0"},
      {0x000c0020, "mem_lock2_value"},
      {0x00031a30, "cm_core_x8_part4"},
      {0x00031a00, "cm_core_x8_part1"},
      {0x00031018, "shim_performance_control5"},
      {0x00031a20, "cm_core_x8_part3"},
      {0x00031a10, "cm_core_x8_part2"},
      {0x00031014, "shim_performance_control4"},
      {0x0001f080, "mm_lock8_value"},
      {0x00034518, "cm_event_group_stream_switch_enable"},
      {0x000a0158, "mem_dma_bd10_6"},
      {0x000a0144, "mem_dma_bd10_1"},
      {0x000a0140, "mem_dma_bd10_0"},
      {0x0001f000, "mm_lock0_value"},
      {0x000a014c, "mem_dma_bd10_3"},
      {0x00032790, "cm_core_f9"},
      {0x00032780, "cm_core_f8"},
      {0x00000080, "shim_lock8_value"},
      {0x00032710, "cm_core_f1"},
      {0x00032700, "cm_core_f0"},
      {0x00032730, "cm_core_f3"},
      {0x00032720, "cm_core_f2"},
      {0x00032750, "cm_core_f5"},
      {0x00032740, "cm_core_f4"},
      {0x00032770, "cm_core_f7"},
      {0x00032760, "cm_core_f6"},
      {0x000a0404, "mem_dma_bd32_1"},
      {0x000a0400, "mem_dma_bd32_0"},
      {0x000140d8, "mm_trace_status"},
      {0x000a0408, "mem_dma_bd32_2"},
      {0x000a0414, "mem_dma_bd32_5"},
      {0x000a0410, "mem_dma_bd32_4"},
      {0x000a041c, "mem_dma_bd32_7"},
      {0x000a0418, "mem_dma_bd32_6"},
      {0x000c0040, "mem_lock4_value"},
      {0x0003f204, "cm_stream_switch_slave_aie_core0_slot1"},
      {0x0003f200, "cm_stream_switch_slave_aie_core0_slot0"},
      {0x0003f20c, "cm_stream_switch_slave_aie_core0_slot3"},
      {0x0003f208, "cm_stream_switch_slave_aie_core0_slot2"},
      {0x0001451c, "mm_event_group_user_event_enable"},
      {0x00014000, "mm_timer_control"},
      {0x00009284, "shim_dma_bd13_5"},
      {0x000a04dc, "mem_dma_bd38_7"},
      {0x000a04d8, "mem_dma_bd38_6"},
      {0x00032d10, "cm_core_fc"},
      {0x000a04d0, "mem_dma_bd38_4"},
      {0x000a04cc, "mem_dma_bd38_3"},
      {0x000a04c8, "mem_dma_bd38_2"},
      {0x000a04c4, "mem_dma_bd38_1"},
      {0x000a04c0, "mem_dma_bd38_0"},
      {0x0003f2e8, "cm_stream_switch_slave_west_3_slot2"},
      {0x00031080, "shim_performance_counter0_event_value"},
      {0x0003f2ec, "cm_stream_switch_slave_west_3_slot3"},
      {0x00001008, "shim_interrupt_controller_2nd_level_disable"},
      {0x000c0108, "uc_dma_dm2mm_axi_control"},
      {0x0003f310, "cm_stream_switch_slave_north_2_slot0"},
      {0x0003f314, "cm_stream_switch_slave_north_2_slot1"},
      {0x0003f318, "cm_stream_switch_slave_north_2_slot2"},
      {0x0003f31c, "cm_stream_switch_slave_north_2_slot3"},
      {0x00031030, "shim_performance_counter4"},
      {0x00031034, "shim_performance_counter5"},
      {0x00031028, "shim_performance_counter2"},
      {0x0003102C, "shim_performance_counter3"},
      {0x00031020, "shim_performance_counter0"},
      {0x00031024, "shim_performance_counter1"},
      {0x00014508, "mm_event_group_dma_enable"},
      {0x00000030, "shim_lock3_value"},
      {0x000340f4, "cm_timer_trig_event_high_value"},
      {0x0001d134, "mm_dma_bd9_5"},
      {0x000c0190, "mem_lock25_value"},
      {0x000d0000, "mem_lock_request"},
      {0x0001f128, "mm_locks_underflow"},
      {0x00034088, "cm_event_broadcast_block_east_value"},
      {0x00011084, "mm_performance_counter1_event_value"},
      {0x00034058, "cm_event_broadcast_block_south_value"},
      {0x0003f340, "cm_stream_switch_slave_east_1_slot0"},
      {0x0003f344, "cm_stream_switch_slave_east_1_slot1"},
      {0x0003f348, "cm_stream_switch_slave_east_1_slot2"},
      {0x0003f34c, "cm_stream_switch_slave_east_1_slot3"},
      {0x000c02a0, "mem_lock42_value"},
      {0x000a0468, "mem_dma_bd35_2"},
      {0x000a046c, "mem_dma_bd35_3"},
      {0x000a0460, "mem_dma_bd35_0"},
      {0x000a0464, "mem_dma_bd35_1"},
      {0x000a0478, "mem_dma_bd35_6"},
      {0x0003ff20, "cm_stream_switch_parity_injection"},
      {0x000a0470, "mem_dma_bd35_4"},
      {0x000a0474, "mem_dma_bd35_5"},
      {0x000a04b0, "mem_dma_bd37_4"},
      {0x00032510, "cm_core_stfifol_part2"},
      {0x000a04b8, "mem_dma_bd37_6"},
      {0x000a04bc, "mem_dma_bd37_7"},
      {0x000a04a0, "mem_dma_bd37_0"},
      {0x000a04a4, "mem_dma_bd37_1"},
      {0x000a04a8, "mem_dma_bd37_2"},
      {0x00032530, "cm_core_stfifol_part4"},
      {0x000a03f8, "mem_dma_bd31_6"},
      {0x000a03fc, "mem_dma_bd31_7"},
      {0x000a03f0, "mem_dma_bd31_4"},
      {0x000a03f4, "mem_dma_bd31_5"},
      {0x000a03e8, "mem_dma_bd31_2"},
      {0x000a03ec, "mem_dma_bd31_3"},
      {0x000a03e0, "mem_dma_bd31_0"},
      {0x000a03e4, "mem_dma_bd31_1"},
      {0x000a0608, "mem_dma_s2mm_1_ctrl"},
      {0x0003f038, "cm_stream_switch_master_config_north1"},
      {0x0003f034, "cm_stream_switch_master_config_north0"},
      {0x0003f040, "cm_stream_switch_master_config_north3"},
      {0x0003f03c, "cm_stream_switch_master_config_north2"},
      {0x0003f048, "cm_stream_switch_master_config_north5"},
      {0x0003f044, "cm_stream_switch_master_config_north4"},
      {0x00009064, "shim_dma_bd2_1"},
      {0x000fff00, "mem_module_clock_control"},
      {0x0001de10, "mm_dma_mm2s_0_ctrl"},
      {0x000a0674, "mem_dma_s2mm_status_5"},
      {0x000a0670, "mem_dma_s2mm_status_4"},
      {0x000a066c, "mem_dma_s2mm_status_3"},
      {0x000a0668, "mem_dma_s2mm_status_2"},
      {0x000a0664, "mem_dma_s2mm_status_1"},
      {0x000a0660, "mem_dma_s2mm_status_0"},
      {0x00000120, "shim_locks_overflow"},
      {0x00032570, "cm_core_stfifoh_part4"},
      {0x00032560, "cm_core_stfifoh_part3"},
      {0x00032550, "cm_core_stfifoh_part2"},
      {0x00032540, "cm_core_stfifoh_part1"},
      {0x00014060, "mm_event_broadcast_block_west_set"},
      {0x00060000, "cm_module_clock_control"},
      {0x000b0f10, "mem_stream_switch_parity_status"},
      {0x000300b0, "cm_core_bmhl0_part4"},
      {0x00094408, "mem_edge_detection_event_control"},
      {0x0003f128, "cm_stream_switch_slave_config_south_5"},
      {0x0003f124, "cm_stream_switch_slave_config_south_4"},
      {0x0003f2a4, "cm_stream_switch_slave_south_5_slot1"},
      {0x0003f2a0, "cm_stream_switch_slave_south_5_slot0"},
      {0x0003f2ac, "cm_stream_switch_slave_south_5_slot3"},
      {0x0003f2a8, "cm_stream_switch_slave_south_5_slot2"},
      {0x000340d4, "cm_trace_control1"},
      {0x000340d0, "cm_trace_control0"},
      {0x00032ca0, "cm_core_s2"},
      {0x00032cb0, "cm_core_s3"},
      {0x00032c80, "cm_core_s0"},
      {0x00032c90, "cm_core_s1"},
      {0x0003f118, "cm_stream_switch_slave_config_south_1"},
      {0x00032b20, "cm_core_dj2"},
      {0x00032b30, "cm_core_dj3"},
      {0x00032b00, "cm_core_dj0"},
      {0x00032b10, "cm_core_dj1"},
      {0x00032b60, "cm_core_dj6"},
      {0x00032b70, "cm_core_dj7"},
      {0x00032b40, "cm_core_dj4"},
      {0x00032b50, "cm_core_dj5"},
      {0x000c02f0, "mem_lock47_value"},
      {0x00009248, "shim_dma_bd12_2"},
      {0x0000924c, "shim_dma_bd12_3"},
      {0x00009240, "shim_dma_bd12_0"},
      {0x00009244, "shim_dma_bd12_1"},
      {0x00009258, "shim_dma_bd12_6"},
      {0x0000925c, "shim_dma_bd12_7"},
      {0x00032d20, "cm_core_sp"},
      {0x00009254, "shim_dma_bd12_5"},
      {0x00009260, "shim_dma_bd12_8"},
      {0x000b0268, "mem_stream_switch_slave_tile_ctrl_slot2"},
      {0x000b026c, "mem_stream_switch_slave_tile_ctrl_slot3"},
      {0x000b0260, "mem_stream_switch_slave_tile_ctrl_slot0"},
      {0x000b0264, "mem_stream_switch_slave_tile_ctrl_slot1"},
      {0x000091e0, "shim_dma_bd10_0"},
      {0x000091e4, "shim_dma_bd10_1"},
      {0x000091e8, "shim_dma_bd10_2"},
      {0x000091ec, "shim_dma_bd10_3"},
      {0x00009200, "shim_dma_bd10_8"},
      {0x000b0f00, "mem_stream_switch_event_port_selection_0"},
      {0x000c00d0, "mem_lock13_value"},
      {0x0003f2cc, "cm_stream_switch_slave_west_1_slot3"},
      {0x00014088, "mm_event_broadcast_block_east_value"},
      {0x00009318, "shim_dma_mm2s_1_ctrl"},
      {0x00014054, "mm_event_broadcast_block_south_clr"},
      {0x000b02e4, "mem_stream_switch_slave_north_1_slot1"},
      {0x000b02e0, "mem_stream_switch_slave_north_1_slot0"},
      {0x000b02ec, "mem_stream_switch_slave_north_1_slot3"},
      {0x000b02e8, "mem_stream_switch_slave_north_1_slot2"},
      {0x000940a4, "mem_event_broadcast_b_block_west_clr"},
      {0x000b0f20, "mem_stream_switch_parity_injection"},
      {0x0001d060, "mm_dma_bd3_0"},
      {0x000b0f04, "mem_stream_switch_event_port_selection_1"},
      {0x000c01e0, "mem_lock30_value"},
      {0x00034504, "cm_event_group_pc_enable"},
      {0x00009128, "shim_dma_bd6_2"},
      {0x0000912c, "shim_dma_bd6_3"},
      {0x00009120, "shim_dma_bd6_0"},
      {0x00009124, "shim_dma_bd6_1"},
      {0x000c03c0, "mem_lock60_value"},
      {0x00009138, "shim_dma_bd6_6"},
      {0x0001df04, "mm_dma_s2mm_status_1"},
      {0x0001df00, "mm_dma_s2mm_status_0"},
      {0x0000913c, "shim_dma_bd6_7"},
      {0x00009130, "shim_dma_bd6_4"},
      {0x00009134, "shim_dma_bd6_5"},
      {0x00001010, "shim_interrupt_controller_2nd_level_interrupt"},
      {0x20, "shim_dma_bd_step_size"},
      {0x000c00a0, "mem_lock10_value"},
      {0x000a062c, "mem_dma_s2mm_5_start_queue"},
      {0x00009308, "shim_dma_s2mm_1_ctrl"},
      {0x0001d020, "mm_dma_bd1_0"},
      {0x000a0380, "mem_dma_bd28_0"},
      {0x000a0384, "mem_dma_bd28_1"},
      {0x000a0388, "mem_dma_bd28_2"},
      {0x000a038c, "mem_dma_bd28_3"},
      {0x000a0390, "mem_dma_bd28_4"},
      {0x000a0394, "mem_dma_bd28_5"},
      {0x000a0398, "mem_dma_bd28_6"},
      {0x000a039c, "mem_dma_bd28_7"},
      {0x00034514, "cm_event_group_errors1_enable"},
      {0x00009290, "shim_dma_bd13_8"},
      {0x000a0280, "mem_dma_bd20_0"},
      {0x000a0284, "mem_dma_bd20_1"},
      {0x000a0288, "mem_dma_bd20_2"},
      {0x000a028c, "mem_dma_bd20_3"},
      {0x000a0290, "mem_dma_bd20_4"},
      {0x000a0294, "mem_dma_bd20_5"},
      {0x000a0298, "mem_dma_bd20_6"},
      {0x000a029c, "mem_dma_bd20_7"},
      {0x00040000, "mm_lock_request"},
      {0x00094084, "mem_event_broadcast_a_block_east_clr"},
      {0x000c0270, "mem_lock39_value"},
      {0x00032850, "cm_core_r5"},
      {0x00032840, "cm_core_r4"},
      {0x00032870, "cm_core_r7"},
      {0x00032860, "cm_core_r6"},
      {0x00032810, "cm_core_r1"},
      {0x00032800, "cm_core_r0"},
      {0x00014084, "mm_event_broadcast_block_east_clr"},
      {0x00032820, "cm_core_r2"},
      {0x000092f0, "shim_dma_bd15_8"},
      {0x00032890, "cm_core_r9"},
      {0x00032880, "cm_core_r8"},
      {0x00009230, "shim_dma_bd11_8"},
      {0x00009220, "shim_dma_bd11_4"},
      {0x000940a8, "mem_event_broadcast_b_block_west_value"},
      {0x00094030, "mem_event_broadcast8"},
      {0x00094034, "mem_event_broadcast9"},
      {0x00014510, "mm_event_group_memory_conflict_enable"},
      {0x000c0260, "mem_lock38_value"},
      {0x00034034, "shim_event_broadcast_a_9"},
      {0x00034030, "shim_event_broadcast_a_8"},
      {0x0003402c, "shim_event_broadcast_a_7"},
      {0x00034028, "shim_event_broadcast_a_6"},
      {0x00034024, "shim_event_broadcast_a_5"},
      {0x00034020, "shim_event_broadcast_a_4"},
      {0x0003401c, "shim_event_broadcast_a_3"},
      {0x00094020, "mem_event_broadcast4"},
      {0x00034014, "shim_event_broadcast_a_1"},
      {0x00034010, "shim_event_broadcast_a_0"},
      {0x000326b0, "cm_core_eg11"},
      {0x00009348, "shim_dma_pause"},
      {0x00002120, "shim_axi_mm_outstanding_transactions"},
      {0x000a0620, "mem_dma_s2mm_4_ctrl"},
      {0x00009104, "shim_dma_bd5_5"},
      {0x00009100, "shim_dma_bd5_4"},
      {0x0000910c, "shim_dma_bd5_7"},
      {0x00009108, "shim_dma_bd5_6"},
      {0x0003f240, "cm_stream_switch_slave_fifo_0_slot0"},
      {0x0003f244, "cm_stream_switch_slave_fifo_0_slot1"},
      {0x0003f248, "cm_stream_switch_slave_fifo_0_slot2"},
      {0x0003f24c, "cm_stream_switch_slave_fifo_0_slot3"},
      {0x000091c8, "shim_dma_bd9_6"},
      {0x00009110, "shim_dma_bd5_8"},
      {0x000000f0, "shim_lock15_value"},
      {0x000090ac, "shim_dma_bd3_7"},
      {0x000090a8, "shim_dma_bd3_6"},
      {0x000090a4, "shim_dma_bd3_5"},
      {0x000090a0, "shim_dma_bd3_4"},
      {0x0000909c, "shim_dma_bd3_3"},
      {0x00009098, "shim_dma_bd3_2"},
      {0x00009094, "shim_dma_bd3_1"},
      {0x00009090, "shim_dma_bd3_0"},
      {0x000090b0, "shim_dma_bd3_8"},
      {0x000c0180, "mem_lock24_value"},
      {0x000a0070, "mem_dma_bd3_4"},
      {0x000a0074, "mem_dma_bd3_5"},
      {0x000a0078, "mem_dma_bd3_6"},
      {0x000a007c, "mem_dma_bd3_7"},
      {0x000b0000, "mem_stream_switch_master_config_dma0"},
      {0x000b0004, "mem_stream_switch_master_config_dma1"},
      {0x000b0008, "mem_stream_switch_master_config_dma2"},
      {0x000b000c, "mem_stream_switch_master_config_dma3"},
      {0x000b0010, "mem_stream_switch_master_config_dma4"},
      {0x000b0014, "mem_stream_switch_master_config_dma5"},
      {0x000a036c, "mem_dma_bd27_3"},
      {0x000a0368, "mem_dma_bd27_2"},
      {0x000a0364, "mem_dma_bd27_1"},
      {0x000a0360, "mem_dma_bd27_0"},
      {0x000a037c, "mem_dma_bd27_7"},
      {0x000a0378, "mem_dma_bd27_6"},
      {0x000a0374, "mem_dma_bd27_5"},
      {0x000a0370, "mem_dma_bd27_4"},
      {0x000a0334, "mem_dma_bd25_5"},
      {0x000a0330, "mem_dma_bd25_4"},
      {0x000a033c, "mem_dma_bd25_7"},
      {0x000a0338, "mem_dma_bd25_6"},
      {0x000a0324, "mem_dma_bd25_1"},
      {0x000a0320, "mem_dma_bd25_0"},
      {0x000a032c, "mem_dma_bd25_3"},
      {0x000a0328, "mem_dma_bd25_2"},
      {0x000a02fc, "mem_dma_bd23_7"},
      {0x000a02f8, "mem_dma_bd23_6"},
      {0x000a02f4, "mem_dma_bd23_5"},
      {0x000a02f0, "mem_dma_bd23_4"},
      {0x000a02ec, "mem_dma_bd23_3"},
      {0x000a02e8, "mem_dma_bd23_2"},
      {0x000a02e4, "mem_dma_bd23_1"},
      {0x000a02e0, "mem_dma_bd23_0"},
      {0x00009170, "shim_dma_bd7_8"},
      {0x000b0204, "mem_stream_switch_slave_dma_0_slot1"},
      {0x000b0200, "mem_stream_switch_slave_dma_0_slot0"},
      {0x000b020c, "mem_stream_switch_slave_dma_0_slot3"},
      {0x000b0208, "mem_stream_switch_slave_dma_0_slot2"},
      {0x000092e4, "shim_dma_bd15_5"},
      {0x000a0638, "mem_dma_mm2s_1_ctrl"},
      {0x00034404, "cm_combo_event_control"},
      {0x000b0144, "mem_stream_switch_slave_config_trace"},
      {0x000940e0, "mem_trace_event0"},
      {0x000940e4, "mem_trace_event1"},
      {0x00032600, "cm_core_eg0"},
      {0x00032610, "cm_core_eg1"},
      {0x00038014, "cm_debug_control1"},
      {0x00038010, "cm_debug_control0"},
      {0x00032640, "cm_core_eg4"},
      {0x00032650, "cm_core_eg5"},
      {0x00032660, "cm_core_eg6"},
      {0x00032670, "cm_core_eg7"},
      {0x000091a0, "shim_dma_bd8_8"},
      {0x00009180, "shim_dma_bd8_0"},
      {0x00000128, "shim_locks_underflow"},
      {0x00009188, "shim_dma_bd8_2"},
      {0x0000918c, "shim_dma_bd8_3"},
      {0x00009190, "shim_dma_bd8_4"},
      {0x00009194, "shim_dma_bd8_5"},
      {0x00009198, "shim_dma_bd8_6"},
      {0x0000919c, "shim_dma_bd8_7"},
      {0x000092ec, "shim_dma_bd15_7"},
      {0x000c02b0, "mem_lock43_value"},
      {0x000340f8, "shim_timer_low"},
      {0x000c0390, "mem_lock57_value"},
      {0x0009451c, "mem_event_group_broadcast_enable"},
      {0x0003f308, "cm_stream_switch_slave_north_1_slot2"},
      {0x0003f304, "cm_stream_switch_slave_north_1_slot1"},
      {0x0003f300, "cm_stream_switch_slave_north_1_slot0"},
      {0x00009000, "shim_dma_bd0_0"},
      {0x00009004, "shim_dma_bd0_1"},
      {0x00009008, "shim_dma_bd0_2"},
      {0x0000900c, "shim_dma_bd0_3"},
      {0x00009010, "shim_dma_bd0_4"},
      {0x00009014, "shim_dma_bd0_5"},
      {0x00009018, "shim_dma_bd0_6"},
      {0x0000901c, "shim_dma_bd0_7"},
      {0x00009020, "shim_dma_bd0_8"},
      {0x000940b4, "mem_event_broadcast_b_block_north_clr"},
      {0x000b0210, "mem_stream_switch_slave_dma_1_slot0"},
      {0x000b0214, "mem_stream_switch_slave_dma_1_slot1"},
      {0x000b0218, "mem_stream_switch_slave_dma_1_slot2"},
      {0x000b021c, "mem_stream_switch_slave_dma_1_slot3"},
      {0x00034408, "cm_edge_detection_event_control"},
      {0x00034050, "shim_event_broadcast_a_block_south_set"},
      {0x000c0070, "mem_lock7_value"},
      {0x0003801c, "cm_debug_status"},
      {0x000b030c, "mem_stream_switch_slave_north_3_slot3"},
      {0x000b0308, "mem_stream_switch_slave_north_3_slot2"},
      {0x000b0304, "mem_stream_switch_slave_north_3_slot1"},
      {0x000b0300, "mem_stream_switch_slave_north_3_slot0"},
      {0x0003802c, "cm_pc_event3"},
      {0x00038028, "cm_pc_event2"},
      {0x00038024, "cm_pc_event1"},
      {0x00038020, "cm_pc_event0"},
      {0x0003f000, "cm_stream_switch_master_config_aie_core0"},
      {0x000a0238, "mem_dma_bd17_6"},
      {0x000a023c, "mem_dma_bd17_7"},
      {0x0001f118, "mm_locks_event_selection_6"},
      {0x0001f11c, "mm_locks_event_selection_7"},
      {0x0001f110, "mm_locks_event_selection_4"},
      {0x0001f114, "mm_locks_event_selection_5"},
      {0x0001f108, "mm_locks_event_selection_2"},
      {0x000a06b8, "mem_dma_s2mm_current_write_count_2"},
      {0x0001f100, "mm_locks_event_selection_0"},
      {0x0001f104, "mm_locks_event_selection_1"},
      {0x000940f0, "mem_timer_trig_event_low_value"},
      {0x0009404c, "mem_event_broadcast15"},
      {0x00094038, "mem_event_broadcast10"},
      {0x0009403c, "mem_event_broadcast11"},
      {0x00094040, "mem_event_broadcast12"},
      {0x00094044, "mem_event_broadcast13"},
      {0x0003f324, "cm_stream_switch_slave_north_3_slot1"},
      {0x0003f320, "cm_stream_switch_slave_north_3_slot0"},
      {0x0003f32c, "cm_stream_switch_slave_north_3_slot3"},
      {0x0003f328, "cm_stream_switch_slave_north_3_slot2"},
      {0x00030130, "cm_core_bmll1_part4"},
      {0x00030100, "cm_core_bmll1_part1"},
      {0x00030110, "cm_core_bmll1_part2"},
      {0x00030120, "cm_core_bmll1_part3"},
      {0x00031830, "cm_core_x0_part4"},
      {0x00031800, "cm_core_x0_part1"},
      {0x0003ff10, "cm_stream_switch_parity_status"},
      {0x00060020, "cm_tile_control"},
      {0x00031810, "cm_core_x0_part2"},
      {0x00030350, "cm_core_bmlh3_part2"},
      {0x00030360, "cm_core_bmlh3_part3"},
      {0x00037580, "cm_performance_counter0_event_value"},
      {0x00030340, "cm_core_bmlh3_part1"},
      {0x00030370, "cm_core_bmlh3_part4"},
      {0x000b02a4, "mem_stream_switch_slave_south_3_slot1"},
      {0x000b02a0, "mem_stream_switch_slave_south_3_slot0"},
      {0x000b02ac, "mem_stream_switch_slave_south_3_slot3"},
      {0x000b02a8, "mem_stream_switch_slave_south_3_slot2"},
      {0x000c0160, "mem_lock22_value"},
      {0x000340f8, "cm_timer_low"},
      {0x00094080, "mem_event_broadcast_a_block_east_set"},
      {0x00014504, "mm_event_group_watchpoint_enable"},
      {0x000c0080, "mem_lock8_value"},
      {0x000c0210, "mem_lock33_value"},
      {0x000c0024, "uc_module_axi_mm_outstanding_transactions"},
      {0x000a064c, "mem_dma_mm2s_3_start_queue"},
      {0x000092d4, "shim_dma_bd15_1"},
      {0x00000208, "npi_me_secure_reg"},
      {0x000092d0, "shim_dma_bd15_0"},
      {0x000092dc, "shim_dma_bd15_3"},
      {0x000306a0, "cm_core_bmhl6_part3"},
      {0x00030690, "cm_core_bmhl6_part2"},
      {0x00030680, "cm_core_bmhl6_part1"},
      {0x000092d8, "shim_dma_bd15_2"},
      {0x00094098, "mem_event_broadcast_b_block_south_value"},
      {0x000306b0, "cm_core_bmhl6_part4"},
      {0x000000d0, "shim_lock13_value"},
      {0x000940b8, "mem_event_broadcast_b_block_north_value"},
      {0x000092e0, "shim_dma_bd15_4"},
      {0x0003f04c, "cm_stream_switch_master_config_east0"},
      {0x0003f050, "cm_stream_switch_master_config_east1"},
      {0x0003f054, "cm_stream_switch_master_config_east2"},
      {0x0003f058, "cm_stream_switch_master_config_east3"},
      {0x00032830, "cm_core_r3"},
      {0x000092e8, "shim_dma_bd15_6"},
      {0x000a06c8, "mem_dma_s2mm_fot_count_fifo_pop_0"},
      {0x000305b0, "cm_core_bmhl5_part4"},
      {0x00030580, "cm_core_bmhl5_part1"},
      {0x00030590, "cm_core_bmhl5_part2"},
      {0x000305a0, "cm_core_bmhl5_part3"},
      {0x000b0800, "mem_stream_switch_deterministic_merge_arb0_slave0_1"},
      {0x000c0320, "mem_lock50_value"},
      {0x00034504, "shim_event_group_dma_enable"},
      {0x0003450c, "cm_event_group_core_program_flow_enable"},
      {0x00014050, "mm_event_broadcast_block_south_set"},
      {0x000303f0, "cm_core_bmhh3_part4"},
      {0x000303d0, "cm_core_bmhh3_part2"},
      {0x000303e0, "cm_core_bmhh3_part3"},
      {0x000303c0, "cm_core_bmhh3_part1"},
      {0x00031890, "cm_core_x2_part2"},
      {0x0001d1d4, "mm_dma_bd14_5"},
      {0x0001d1d0, "mm_dma_bd14_4"},
      {0x0001d1c4, "mm_dma_bd14_1"},
      {0x0001d1c0, "mm_dma_bd14_0"},
      {0x0001d1cc, "mm_dma_bd14_3"},
      {0x0001d1c8, "mm_dma_bd14_2"},
      {0x00034500, "cm_event_group_0_enable"},
      {0x0003f148, "cm_stream_switch_slave_config_north_3"},
      {0x0003f00c, "cm_stream_switch_master_config_tile_ctrl"},
      {0x0003f140, "cm_stream_switch_slave_config_north_1"},
      {0x0003f13c, "cm_stream_switch_slave_config_north_0"},
      {0x00094504, "mem_event_group_watchpoint_enable"},
      {0x000fff20, "mem_tile_control"},
      {0x000c0060, "mem_lock6_value"},
      {0x0001d144, "mm_dma_bd10_1"},
      {0x0001d140, "mm_dma_bd10_0"},
      {0x0001d14c, "mm_dma_bd10_3"},
      {0x0001d148, "mm_dma_bd10_2"},
      {0x0001d154, "mm_dma_bd10_5"},
      {0x0001d150, "mm_dma_bd10_4"},
      {0x0001d194, "mm_dma_bd12_5"},
      {0x0001d190, "mm_dma_bd12_4"},
      {0x0001d18c, "mm_dma_bd12_3"},
      {0x0001d188, "mm_dma_bd12_2"},
      {0x0001d184, "mm_dma_bd12_1"},
      {0x0001d180, "mm_dma_bd12_0"},
      {0x000b0290, "mem_stream_switch_slave_south_2_slot0"},
      {0x000a01c4, "mem_dma_bd14_1"},
      {0x000a01c0, "mem_dma_bd14_0"},
      {0x000a01cc, "mem_dma_bd14_3"},
      {0x000a01c8, "mem_dma_bd14_2"},
      {0x000a01d4, "mem_dma_bd14_5"},
      {0x000a01d0, "mem_dma_bd14_4"},
      {0x000a01dc, "mem_dma_bd14_7"},
      {0x000a01d8, "mem_dma_bd14_6"},
      {0x000b0294, "mem_stream_switch_slave_south_2_slot1"},
      {0x00034058, "shim_event_broadcast_a_block_south_value"},
      {0x00038030, "cm_error_halt_control"},
      {0x000090cc, "shim_dma_bd4_3"},
      {0x0003f2e4, "cm_stream_switch_slave_west_3_slot1"},
      {0x000b0298, "mem_stream_switch_slave_south_2_slot2"},
      {0x000a01a8, "mem_dma_bd13_2"},
      {0x000c0120, "uc_dma_pause"},
      {0x000a018c, "mem_dma_bd12_3"},
      {0x000a0188, "mem_dma_bd12_2"},
      {0x000a0184, "mem_dma_bd12_1"},
      {0x000a0180, "mem_dma_bd12_0"},
      {0x000a019c, "mem_dma_bd12_7"},
      {0x000a0198, "mem_dma_bd12_6"},
      {0x000a0194, "mem_dma_bd12_5"},
      {0x000a0190, "mem_dma_bd12_4"},
      {0x00031010, "shim_performance_control3"},
      {0x0003100C, "shim_performance_control2"},
      {0x00031008, "shim_performance_control1"},
      {0x00031000, "shim_performance_control0"},
      {0x0003ff00, "cm_stream_switch_event_port_selection_0"},
      {0x0003ff04, "cm_stream_switch_event_port_selection_1"},
      {0x000a0154, "mem_dma_bd10_5"},
      {0x000a0150, "mem_dma_bd10_4"},
      {0x000a015c, "mem_dma_bd10_7"},
      {0x00030630, "cm_core_bmll6_part4"},
      {0x00030620, "cm_core_bmll6_part3"},
      {0x00030610, "cm_core_bmll6_part2"},
      {0x00030600, "cm_core_bmll6_part1"},
      {0x000a0148, "mem_dma_bd10_2"},
      {0x000a021c, "mem_dma_bd16_7"},
      {0x000a0218, "mem_dma_bd16_6"},
      {0x000a0214, "mem_dma_bd16_5"},
      {0x000a0210, "mem_dma_bd16_4"},
      {0x000a020c, "mem_dma_bd16_3"},
      {0x000a0208, "mem_dma_bd16_2"},
      {0x000a0204, "mem_dma_bd16_1"},
      {0x000a0200, "mem_dma_bd16_0"},
      {0x0003f334, "cm_stream_switch_slave_east_0_slot1"},
      {0x0003f330, "cm_stream_switch_slave_east_0_slot0"},
      {0x0003f33c, "cm_stream_switch_slave_east_0_slot3"},
      {0x0003f338, "cm_stream_switch_slave_east_0_slot2"},
      {0x00034080, "shim_event_broadcast_a_block_east_set"},
      {0x00014200, "mm_event_status0"},
      {0x00014204, "mm_event_status1"},
      {0x00014208, "mm_event_status2"},
      {0x0001420c, "mm_event_status3"},
      {0x00032d00, "cm_core_pc"},
      {0x00034074, "shim_event_broadcast_a_block_north_clr"},
      {0x0001f040, "mm_lock4_value"},
      {0x000a0178, "mem_dma_bd11_6"},
      {0x00034018, "shim_event_broadcast_a_2"},
      {0x000a0630, "mem_dma_mm2s_0_ctrl"},
      {0x00000050, "npi_me_imr2"},
      {0x00032c70, "cm_core_p7"},
      {0x00032c60, "cm_core_p6"},
      {0x00032c50, "cm_core_p5"},
      {0x00032c40, "cm_core_p4"},
      {0x00032c30, "cm_core_p3"},
      {0x00032c20, "cm_core_p2"},
      {0x00032c10, "cm_core_p1"},
      {0x00032c00, "cm_core_p0"},
      {0x00037588, "cm_performance_counter2_event_value"},
      {0x00009314, "shim_dma_mm2s_0_task_queue"},
      {0x0003ff34, "cm_stream_switch_adaptive_clock_gate_status"},
      {0x000302e0, "cm_core_bmhh2_part3"},
      {0x000302d0, "cm_core_bmhh2_part2"},
      {0x000302c0, "cm_core_bmhh2_part1"},
      {0x000302f0, "cm_core_bmhh2_part4"},
      {0x000b0804, "mem_stream_switch_deterministic_merge_arb0_slave2_3"},
      {0x000a00b0, "mem_dma_bd5_4"},
      {0x00094000, "mem_timer_control"},
      {0x000c0420, "mem_locks_overflow_0"},
      {0x000c0424, "mem_locks_overflow_1"},
      {0x000a0270, "mem_dma_bd19_4"},
      {0x000a0274, "mem_dma_bd19_5"},
      {0x000a0278, "mem_dma_bd19_6"},
      {0x000a027c, "mem_dma_bd19_7"},
      {0x000a0260, "mem_dma_bd19_0"},
      {0x000a0264, "mem_dma_bd19_1"},
      {0x000a0268, "mem_dma_bd19_2"},
      {0x000a026c, "mem_dma_bd19_3"},
      {0x000b0128, "mem_stream_switch_slave_config_south_3"},
      {0x000b0124, "mem_stream_switch_slave_config_south_2"},
      {0x000b0120, "mem_stream_switch_slave_config_south_1"},
      {0x000b011c, "mem_stream_switch_slave_config_south_0"},
      {0x000b0130, "mem_stream_switch_slave_config_south_5"},
      {0x000b012c, "mem_stream_switch_slave_config_south_4"},
      {0x000090fc, "shim_dma_bd5_3"},
      {0x0001450c, "mm_event_group_lock_enable"},
      {0x000090f8, "shim_dma_bd5_2"},
      {0x000c0030, "mem_lock3_value"},
      {0x000c0330, "mem_lock51_value"},
      {0x000a0654, "mem_dma_mm2s_4_start_queue"},
      {0x000a0600, "mem_dma_s2mm_0_ctrl"},
      {0x00094510, "mem_event_group_stream_switch_enable"},
      {0x00030050, "cm_core_bmlh0_part2"},
      {0x000c0370, "mem_lock55_value"},
      {0x000a0694, "mem_dma_mm2s_status_5"},
      {0x000a0690, "mem_dma_mm2s_status_4"},
      {0x000a0684, "mem_dma_mm2s_status_1"},
      {0x000a0680, "mem_dma_mm2s_status_0"},
      {0x000a068c, "mem_dma_mm2s_status_3"},
      {0x000a0688, "mem_dma_mm2s_status_2"},
      {0x00094058, "mem_event_broadcast_a_block_south_value"},
      {0x000940c0, "mem_event_broadcast_b_block_east_set"},
      {0x000a0648, "mem_dma_mm2s_3_ctrl"},
      {0x00034060, "cm_event_broadcast_block_west_set"},
      {0x00009344, "shim_dma_mm2s_1_response_fifo_parity_error_injection"},
      {0x000b0f38, "mem_stream_switch_adaptive_clock_gate_abort_period"},
      {0x00034084, "cm_event_broadcast_block_east_clr"},
      {0x0009108c, "mem_performance_counter3_event_value"},
      {0x00000020, "shim_lock2_value"},
      {0x0001102c, "mm_performance_counter3"},
      {0x00011028, "mm_performance_counter2"},
      {0x00011024, "mm_performance_counter1"},
      {0x00011020, "mm_performance_counter0"},
      {0x000a065c, "mem_dma_mm2s_5_start_queue"},
      {0x00014068, "mm_event_broadcast_block_west_value"},
      {0x000b027c, "mem_stream_switch_slave_south_0_slot3"},
      {0x000b0270, "mem_stream_switch_slave_south_0_slot0"},
      {0x000b0274, "mem_stream_switch_slave_south_0_slot1"},
      {0x00001000, "shim_interrupt_controller_2nd_level_mask"},
      {0x00038000, "cm_core_control"},
      {0x000c0010, "mem_lock1_value"},
      {0x000a0508, "mem_dma_bd40_2"},
      {0x000a050c, "mem_dma_bd40_3"},
      {0x000a0500, "mem_dma_bd40_0"},
      {0x00032ad0, "cm_core_dn5"},
      {0x000a0518, "mem_dma_bd40_6"},
      {0x000a051c, "mem_dma_bd40_7"},
      {0x00032a80, "cm_core_dn0"},
      {0x000a0514, "mem_dma_bd40_5"},
      {0x000a0550, "mem_dma_bd42_4"},
      {0x000a0554, "mem_dma_bd42_5"},
      {0x000a0558, "mem_dma_bd42_6"},
      {0x000a055c, "mem_dma_bd42_7"},
      {0x000a0540, "mem_dma_bd42_0"},
      {0x000a0544, "mem_dma_bd42_1"},
      {0x000a0548, "mem_dma_bd42_2"},
      {0x000a054c, "mem_dma_bd42_3"},
      {0x000a0598, "mem_dma_bd44_6"},
      {0x000a059c, "mem_dma_bd44_7"},
      {0x000a0590, "mem_dma_bd44_4"},
      {0x000a0594, "mem_dma_bd44_5"},
      {0x000a0588, "mem_dma_bd44_2"},
      {0x000a058c, "mem_dma_bd44_3"},
      {0x000a0580, "mem_dma_bd44_0"},
      {0x000a0584, "mem_dma_bd44_1"},
      {0x0003f810, "cm_stream_switch_deterministic_merge_arb1_slave0_1"},
      {0x000c0100, "uc_dma_dm2mm_status"},
      {0x00094088, "mem_event_broadcast_a_block_east_value"},
      {0x00034054, "cm_event_broadcast_block_south_clr"},
      {0x0003f814, "cm_stream_switch_deterministic_merge_arb1_slave2_3"},
      {0x00000070, "shim_lock7_value"},
      {0x000b02b8, "mem_stream_switch_slave_south_4_slot2"},
      {0x000b02bc, "mem_stream_switch_slave_south_4_slot3"},
      {0x000b02b0, "mem_stream_switch_slave_south_4_slot0"},
      {0x000a0124, "mem_dma_bd9_1"},
      {0x0003420c, "shim_event_status3"},
      {0x00034208, "shim_event_status2"},
      {0x00034204, "shim_event_status1"},
      {0x00034200, "shim_event_status0"},
      {0x000304f0, "cm_core_bmhh4_part4"},
      {0x000304c0, "cm_core_bmhh4_part1"},
      {0x000304e0, "cm_core_bmhh4_part3"},
      {0x000304d0, "cm_core_bmhh4_part2"},
      {0x0003f228, "cm_stream_switch_slave_dma_1_slot2"},
      {0x000a06e8, "mem_dma_mm2s_2_constant_pad_value"},
      {0x000a0604, "mem_dma_s2mm_0_start_queue"},
      {0x0003f22c, "cm_stream_switch_slave_dma_1_slot3"},
      {0x0003f220, "cm_stream_switch_slave_dma_1_slot0"},
      {0x0001df24, "mm_dma_s2mm_fot_count_fifo_pop_1"},
      {0x0001df20, "mm_dma_s2mm_fot_count_fifo_pop_0"},
      {0x0003f224, "cm_stream_switch_slave_dma_1_slot1"},
      {0x000940d8, "mem_trace_status"},
      {0x0003f020, "cm_stream_switch_master_config_south3"},
      {0x0003f01c, "cm_stream_switch_master_config_south2"},
      {0x0003f018, "cm_stream_switch_master_config_south1"},
      {0x0003f014, "cm_stream_switch_master_config_south0"},
      {0x00031084, "shim_performance_counter1_event_value"},
      {0x00000010, "shim_lock1_value"},
      {0x000a06ec, "mem_dma_mm2s_3_constant_pad_value"},
      {0x000340e4, "shim_trace_event1"},
      {0x000340e0, "shim_trace_event0"},
      {0x0000010c, "npi_me_pll_status"},
      {0x000a040c, "mem_dma_bd32_3"},
      {0x000c0008, "uc_core_interrupt_status"},
      {0x000c042c, "mem_locks_underflow_1"},
      {0x000c0428, "mem_locks_underflow_0"},
      {0x000b028c, "mem_stream_switch_slave_south_1_slot3"},
      {0x000c0140, "mem_lock20_value"},
      {0x000b0288, "mem_stream_switch_slave_south_1_slot2"},
      {0x000b0284, "mem_stream_switch_slave_south_1_slot1"},
      {0x00034078, "cm_event_broadcast_block_north_value"},
      {0x000b0280, "mem_stream_switch_slave_south_1_slot0"},
      {0x00014038, "mm_event_broadcast10"},
      {0x0001403c, "mm_event_broadcast11"},
      {0x00014040, "mm_event_broadcast12"},
      {0x00014044, "mm_event_broadcast13"},
      {0x00014048, "mm_event_broadcast14"},
      {0x0001404c, "mm_event_broadcast15"},
      {0x0003f160, "cm_stream_switch_slave_config_mem_trace"},
      {0x000c0418, "mem_locks_event_selection_6"},
      {0x000c041c, "mem_locks_event_selection_7"},
      {0x000c0410, "mem_locks_event_selection_4"},
      {0x000c0414, "mem_locks_event_selection_5"},
      {0x000c0408, "mem_locks_event_selection_2"},
      {0x000c040c, "mem_locks_event_selection_3"},
      {0x000c0400, "mem_locks_event_selection_0"},
      {0x000c0404, "mem_locks_event_selection_1"},
      {0x00094060, "mem_event_broadcast_a_block_west_set"},
      {0x000318d0, "cm_core_x3_part2"},
      {0x000318e0, "cm_core_x3_part3"},
      {0x000318c0, "cm_core_x3_part1"},
      {0x0003f2e0, "cm_stream_switch_slave_west_3_slot0"},
      {0x000318f0, "cm_core_x3_part4"},
      {0x000c0340, "mem_lock52_value"},
      {0x000c01d0, "mem_lock29_value"},
      {0x0000915c, "shim_dma_bd7_3"},
      {0x000c03e0, "mem_lock62_value"},
      {0x00009154, "shim_dma_bd7_1"},
      {0x00032d80, "cm_core_cr1"},
      {0x000c01b0, "mem_lock27_value"},
      {0x00032da0, "cm_core_cr3"},
      {0x00032d90, "cm_core_cr2"},
      {0x0000916c, "shim_dma_bd7_7"},
      {0x00009168, "shim_dma_bd7_6"},
      {0x0000931c, "shim_dma_mm2s_1_task_queue"},
      {0x00009164, "shim_dma_bd7_5"},
      {0x00009160, "shim_dma_bd7_4"},
      {0x0003451c, "cm_event_group_broadcast_enable"},
      {0x000a0650, "mem_dma_mm2s_4_ctrl"},
      {0x00094050, "mem_event_broadcast_a_block_south_set"},
      {0x00011080, "mm_performance_counter0_event_value"},
      {0x000c0020, "uc_module_aximm_offset"},
      {0x00038038, "cm_core_processor_bus"},
      {0x0001d08c, "mm_dma_bd4_3"},
      {0x0001d088, "mm_dma_bd4_2"},
      {0x0001d084, "mm_dma_bd4_1"},
      {0x0001d080, "mm_dma_bd4_0"},
      {0x0001d094, "mm_dma_bd4_5"},
      {0x0001d090, "mm_dma_bd4_4"},
      {0x00014408, "mm_edge_detection_event_control"},
      {0x00032dc0, "cm_core_sr1"},
      {0x00032dd0, "cm_core_sr2"},
      {0x0001d044, "mm_dma_bd2_1"},
      {0x0001d040, "mm_dma_bd2_0"},
      {0x0001d04c, "mm_dma_bd2_3"},
      {0x0001d048, "mm_dma_bd2_2"},
      {0x0001d054, "mm_dma_bd2_5"},
      {0x000a04d4, "mem_dma_bd38_5"},
      {0x00009184, "shim_dma_bd8_1"},
      {0x000b0010, "uc_mdm_dbg_ctrl_status"},
      {0x0001d014, "mm_dma_bd0_5"},
      {0x0001d010, "mm_dma_bd0_4"},
      {0x0001d00c, "mm_dma_bd0_3"},
      {0x0001d008, "mm_dma_bd0_2"},
      {0x0001d004, "mm_dma_bd0_1"},
      {0x0001d000, "mm_dma_bd0_0"},
      {0x00014030, "mm_event_broadcast8"},
      {0x00014034, "mm_event_broadcast9"},
      {0x0001d0d4, "mm_dma_bd6_5"},
      {0x0001d0d0, "mm_dma_bd6_4"},
      {0x00014020, "mm_event_broadcast4"},
      {0x00014024, "mm_event_broadcast5"},
      {0x0001d0c4, "mm_dma_bd6_1"},
      {0x0001d0c0, "mm_dma_bd6_0"},
      {0x0001d0cc, "mm_dma_bd6_3"},
      {0x0001d0c8, "mm_dma_bd6_2"},
      {0x000a05e4, "mem_dma_bd47_1"},
      {0x000a05e0, "mem_dma_bd47_0"},
      {0x000a05ec, "mem_dma_bd47_3"},
      {0x000a05e8, "mem_dma_bd47_2"},
      {0x000a05f4, "mem_dma_bd47_5"},
      {0x000a05f0, "mem_dma_bd47_4"},
      {0x000a05fc, "mem_dma_bd47_7"},
      {0x000a05f8, "mem_dma_bd47_6"},
      {0x00032d70, "cm_core_lci"},
      {0x0001f010, "mm_lock1_value"},
      {0x000b0814, "mem_stream_switch_deterministic_merge_arb1_slave2_3"},
      {0x00037500, "cm_performance_control0"},
      {0x00037504, "cm_performance_control1"},
      {0x00037508, "cm_performance_control2"},
      {0x000b0810, "mem_stream_switch_deterministic_merge_arb1_slave0_1"},
      {0x00034044, "shim_event_broadcast_a_13"},
      {0x00034040, "shim_event_broadcast_a_12"},
      {0x0003403c, "shim_event_broadcast_a_11"},
      {0x00034038, "shim_event_broadcast_a_10"},
      {0x0001de00, "mm_dma_s2mm_0_ctrl"},
      {0x00034048, "shim_event_broadcast_a_14"},
      {0x00032430, "cm_core_ldfifol0_part4"},
      {0x00032400, "cm_core_ldfifol0_part1"},
      {0x00032420, "cm_core_ldfifol0_part3"},
      {0x00032410, "cm_core_ldfifol0_part2"},
      {0x000a0618, "mem_dma_s2mm_3_ctrl"},
      {0x00094054, "mem_event_broadcast_a_block_south_clr"},
      {0x00030430, "cm_core_bmll4_part4"},
      {0x00030400, "cm_core_bmll4_part1"},
      {0x00030420, "cm_core_bmll4_part3"},
      {0x00030410, "cm_core_bmll4_part2"},
      {0x000300c0, "cm_core_bmhh0_part1"},
      {0x000300e0, "cm_core_bmhh0_part3"},
      {0x000300d0, "cm_core_bmhh0_part2"},
      {0x000300f0, "cm_core_bmhh0_part4"},
      {0x00038018, "cm_debug_control2"},
      {0x00032620, "cm_core_eg2"},
      {0x00032630, "cm_core_eg3"},
      {0x00030000, "cm_core_bmll0_part1"},
      {0x00030020, "cm_core_bmll0_part3"},
      {0x00030010, "cm_core_bmll0_part2"},
      {0x00030030, "cm_core_bmll0_part4"},
      {0x00014078, "mm_event_broadcast_block_north_value"},
      {0x00032680, "cm_core_eg8"},
      {0x00014080, "mm_event_broadcast_block_east_set"},
      {0x000b0278, "mem_stream_switch_slave_south_0_slot2"},
      {0x000a0094, "mem_dma_bd4_5"},
      {0x000c0380, "mem_lock56_value"},
      {0x00032690, "cm_core_eg9"},
      {0x0001f060, "mm_lock6_value"},
      {0x0001d130, "mm_dma_bd9_4"},
      {0x000a0080, "mem_dma_bd4_0"},
      {0x0001d128, "mm_dma_bd9_2"},
      {0x0001d12c, "mm_dma_bd9_3"},
      {0x0001d120, "mm_dma_bd9_0"},
      {0x0001d124, "mm_dma_bd9_1"},
      {0x000b010c, "mem_stream_switch_slave_config_dma_3"},
      {0x000c02d0, "mem_lock45_value"},
      {0x000329f0, "cm_core_r31"},
      {0x000329e0, "cm_core_r30"},
      {0x000a02a4, "mem_dma_bd21_1"},
      {0x0003ff00, "shim_stream_switch_event_port_selection_0"},
      {0x00030180, "cm_core_bmhl1_part1"},
      {0x0003f2c8, "cm_stream_switch_slave_west_1_slot2"},
      {0x0009450c, "mem_event_group_lock_enable"},
      {0x0003f2c0, "cm_stream_switch_slave_west_1_slot0"},
      {0x0003f2c4, "cm_stream_switch_slave_west_1_slot1"},
      {0x00030190, "cm_core_bmhl1_part2"},
      {0x0003f110, "cm_stream_switch_slave_config_fifo_0"},
      {0x000c0290, "mem_lock41_value"},
      {0x00034000, "cm_timer_control"},
      {0x0001f090, "mm_lock9_value"},
      {0x00030710, "cm_core_bmll7_part2"},
      {0x00030720, "cm_core_bmll7_part3"},
      {0x00030700, "cm_core_bmll7_part1"},
      {0x00030730, "cm_core_bmll7_part4"},
      {0x00036060, "cm_accumulator_control"},
      {0x000a06e0, "mem_dma_mm2s_0_constant_pad_value"},
      {0x000319d0, "cm_core_x7_part2"},
      {0x0003f250, "cm_stream_switch_slave_south_0_slot0"},
      {0x0003f254, "cm_stream_switch_slave_south_0_slot1"},
      {0x0003f258, "cm_stream_switch_slave_south_0_slot2"},
      {0x0003f25c, "cm_stream_switch_slave_south_0_slot3"},
      {0x000000c0, "shim_lock12_value"},
      {0x000c03b0, "mem_lock59_value"},
      {0x00014404, "mm_combo_event_control"},
      {0x0001f0a0, "mm_lock10_value"},
      {0x000b54c0, "uc_mdm_pcsr"},
      {0x0003f21c, "cm_stream_switch_slave_dma_0_slot3"},
      {0x0003f218, "cm_stream_switch_slave_dma_0_slot2"},
      {0x0003f214, "cm_stream_switch_slave_dma_0_slot1"},
      {0x0003f210, "cm_stream_switch_slave_dma_0_slot0"},
      {0x0003f30c, "cm_stream_switch_slave_north_1_slot3"},
      {0x00034208, "cm_event_status2"},
      {0x0003420c, "cm_event_status3"},
      {0x00034200, "cm_event_status0"},
      {0x00034204, "cm_event_status1"},
      {0x00009330, "shim_dma_s2mm_current_write_count_0"},
      {0x00009334, "shim_dma_s2mm_current_write_count_1"},
      {0x00034048, "cm_event_broadcast14"},
      {0x0003404c, "cm_event_broadcast15"},
      {0x00034040, "cm_event_broadcast12"},
      {0x00034044, "cm_event_broadcast13"},
      {0x00034038, "cm_event_broadcast10"},
      {0x0003403c, "cm_event_broadcast11"},
      {0x000301f0, "cm_core_bmhh1_part4"},
      {0x000301c0, "cm_core_bmhh1_part1"},
      {0x000301d0, "cm_core_bmhh1_part2"},
      {0x000301e0, "cm_core_bmhh1_part3"},
      {0x00034088, "shim_event_broadcast_a_block_east_value"},
      {0x0001de1c, "mm_dma_mm2s_1_start_queue"},
      {0x000a00e0, "mem_dma_bd7_0"},
      {0x000a00e4, "mem_dma_bd7_1"},
      {0x000a00e8, "mem_dma_bd7_2"},
      {0x000a00ec, "mem_dma_bd7_3"},
      {0x000a00f0, "mem_dma_bd7_4"},
      {0x000a00f4, "mem_dma_bd7_5"},
      {0x000a00f8, "mem_dma_bd7_6"},
      {0x000a00fc, "mem_dma_bd7_7"},
      {0x000a044c, "mem_dma_bd34_3"},
      {0x000a0448, "mem_dma_bd34_2"},
      {0x000a0444, "mem_dma_bd34_1"},
      {0x000a0440, "mem_dma_bd34_0"},
      {0x000a045c, "mem_dma_bd34_7"},
      {0x000a0458, "mem_dma_bd34_6"},
      {0x000a0454, "mem_dma_bd34_5"},
      {0x000a0450, "mem_dma_bd34_4"},
      {0x000a00b8, "mem_dma_bd5_6"},
      {0x000a00bc, "mem_dma_bd5_7"},
      {0x00094064, "mem_event_broadcast_a_block_west_clr"},
      {0x000a00b4, "mem_dma_bd5_5"},
      {0x000a00a8, "mem_dma_bd5_2"},
      {0x000a00ac, "mem_dma_bd5_3"},
      {0x000a00a0, "mem_dma_bd5_0"},
      {0x000a00a4, "mem_dma_bd5_1"},
      {0x000a0494, "mem_dma_bd36_5"},
      {0x000a0490, "mem_dma_bd36_4"},
      {0x000a049c, "mem_dma_bd36_7"},
      {0x000a0498, "mem_dma_bd36_6"},
      {0x000a0484, "mem_dma_bd36_1"},
      {0x000a0480, "mem_dma_bd36_0"},
      {0x000a048c, "mem_dma_bd36_3"},
      {0x000a0488, "mem_dma_bd36_2"},
      {0x000a03dc, "mem_dma_bd30_7"},
      {0x000a03d8, "mem_dma_bd30_6"},
      {0x000a03d4, "mem_dma_bd30_5"},
      {0x000a03d0, "mem_dma_bd30_4"},
      {0x000a03cc, "mem_dma_bd30_3"},
      {0x000a03c8, "mem_dma_bd30_2"},
      {0x000a03c4, "mem_dma_bd30_1"},
      {0x000a03c0, "mem_dma_bd30_0"},
      {0x000329a0, "cm_core_r26"},
      {0x000b029c, "mem_stream_switch_slave_south_2_slot3"},
      {0x00009158, "shim_dma_bd7_2"},
      {0x00032ae0, "cm_core_dn6"},
      {0x00032af0, "cm_core_dn7"},
      {0x00032ac0, "cm_core_dn4"},
      {0x00014008, "mm_event_generate"},
      {0x00037584, "cm_performance_counter1_event_value"},
      {0x000c00f0, "mem_lock15_value"},
      {0x00032aa0, "cm_core_dn2"},
      {0x00094100, "mem_watchpoint0"},
      {0x00032ab0, "cm_core_dn3"},
      {0x000a0510, "mem_dma_bd40_4"},
      {0x000a0128, "mem_dma_bd9_2"},
      {0x000a012c, "mem_dma_bd9_3"},
      {0x000a0120, "mem_dma_bd9_0"},
      {0x00032a90, "cm_core_dn1"},
      {0x000a0138, "mem_dma_bd9_6"},
      {0x000a013c, "mem_dma_bd9_7"},
      {0x000a0130, "mem_dma_bd9_4"},
      {0x000a0134, "mem_dma_bd9_5"},
      {0x000c0170, "mem_lock23_value"},
      {0x000a000c, "mem_dma_bd0_3"},
      {0x000a0008, "mem_dma_bd0_2"},
      {0x000a0004, "mem_dma_bd0_1"},
      {0x000a0000, "mem_dma_bd0_0"},
      {0x000a001c, "mem_dma_bd0_7"},
      {0x000a0018, "mem_dma_bd0_6"},
      {0x000a0014, "mem_dma_bd0_5"},
      {0x000a0010, "mem_dma_bd0_4"},
      {0x000b0f34, "mem_stream_switch_adaptive_clock_gate_status"},
      {0x00034020, "cm_event_broadcast4"},
      {0x00034024, "cm_event_broadcast5"},
      {0x00034028, "cm_event_broadcast6"},
      {0x0003402c, "cm_event_broadcast7"},
      {0x0003ff04, "shim_stream_switch_event_port_selection_1"},
      {0x000000a0, "shim_lock10_value"},
      {0x0000003c, "npi_me_ier0"},
      {0x00000048, "npi_me_ier1"},
      {0x00000054, "npi_me_ier2"},
      {0x00000060, "npi_me_ier3"},
      {0x000a009c, "mem_dma_bd4_7"},
      {0x000a0098, "mem_dma_bd4_6"},
      {0x00034510, "cm_event_group_errors0_enable"},
      {0x000a0090, "mem_dma_bd4_4"},
      {0x000a008c, "mem_dma_bd4_3"},
      {0x000a0088, "mem_dma_bd4_2"},
      {0x000a0084, "mem_dma_bd4_1"},
      {0x000305c0, "cm_core_bmhh5_part1"},
      {0x000a0054, "mem_dma_bd2_5"},
      {0x000a0050, "mem_dma_bd2_4"},
      {0x000a005c, "mem_dma_bd2_7"},
      {0x000a0058, "mem_dma_bd2_6"},
      {0x000a0044, "mem_dma_bd2_1"},
      {0x000a0040, "mem_dma_bd2_0"},
      {0x000a004c, "mem_dma_bd2_3"},
      {0x000a0048, "mem_dma_bd2_2"},
      {0x000a06e4, "mem_dma_mm2s_1_constant_pad_value"},
      {0x0003f10c, "cm_stream_switch_slave_config_tile_ctrl"},
      {0x000329c0, "cm_core_r28"},
      {0x000329d0, "cm_core_r29"},
      {0x00032980, "cm_core_r24"},
      {0x00032990, "cm_core_r25"},
      {0x0003f804, "cm_stream_switch_deterministic_merge_arb0_slave2_3"},
      {0x000329b0, "cm_core_r27"},
      {0x00032940, "cm_core_r20"},
      {0x00032950, "cm_core_r21"},
      {0x00032960, "cm_core_r22"},
      {0x00032970, "cm_core_r23"},
      {0x00094108, "mem_watchpoint2"},
      {0x0009410c, "mem_watchpoint3"},
      {0x00034050, "cm_event_broadcast_block_south_set"},
      {0x00094104, "mem_watchpoint1"},
      {0x0001de14, "mm_dma_mm2s_0_start_queue"},
      {0x0003f800, "cm_stream_switch_deterministic_merge_arb0_slave0_1"},
      {0x00014058, "mm_event_broadcast_block_south_value"},
      {0x00000000, "uc_base_address"},
      {0x000c0220, "mem_lock34_value"},
      {0x00094078, "mem_event_broadcast_a_block_north_value"},
      {0x0003f010, "cm_stream_switch_master_config_fifo0"},
      {0x00094094, "mem_event_broadcast_b_block_south_clr"},
      {0x000340fc, "cm_timer_high"},
      {0x000c0240, "mem_lock36_value"},
      {0x00009150, "shim_dma_bd7_0"},
      {0x00031970, "cm_core_x5_part4"},
      {0x00031940, "cm_core_x5_part1"},
      {0x00031950, "cm_core_x5_part2"},
      {0x00031960, "cm_core_x5_part3"},
      {0x000c0004, "uc_core_control"},
      {0x00016010, "mm_memory_control"},
      {0x00038034, "cm_error_halt_event"},
      {0x000a04f8, "mem_dma_bd39_6"},
      {0x00014400, "mm_combo_event_inputs"},
      {0x000a04f0, "mem_dma_bd39_4"},
      {0x000a04f4, "mem_dma_bd39_5"},
      {0x000a04e8, "mem_dma_bd39_2"},
      {0x000a04ec, "mem_dma_bd39_3"},
      {0x000a04e0, "mem_dma_bd39_0"},
      {0x000a04e4, "mem_dma_bd39_1"},
      {0x00091034, "mem_performance_counter5"},
      {0x00091030, "mem_performance_counter4"},
      {0x0009102c, "mem_performance_counter3"},
      {0x00091028, "mem_performance_counter2"},
      {0x00091024, "mem_performance_counter1"},
      {0x00091020, "mem_performance_counter0"},
      {0x000c01a0, "mem_lock26_value"},
      {0x000c01f0, "mem_lock31_value"},
      {0x00030390, "cm_core_bmhl3_part2"},
      {0x000303a0, "cm_core_bmhl3_part3"},
      {0x000c0150, "mem_lock21_value"},
      {0x00030380, "cm_core_bmhl3_part1"},
      {0x000303b0, "cm_core_bmhl3_part4"},
      {0x0003f35c, "cm_stream_switch_slave_east_2_slot3"},
      {0x0003f358, "cm_stream_switch_slave_east_2_slot2"},
      {0x0003f354, "cm_stream_switch_slave_east_2_slot1"},
      {0x0003f350, "cm_stream_switch_slave_east_2_slot0"},
      {0x000a0420, "mem_dma_bd33_0"},
      {0x000a0424, "mem_dma_bd33_1"},
      {0x000a0428, "mem_dma_bd33_2"},
      {0x000a042c, "mem_dma_bd33_3"},
      {0x000a0430, "mem_dma_bd33_4"},
      {0x000a0434, "mem_dma_bd33_5"},
      {0x000a0438, "mem_dma_bd33_6"},
      {0x000a043c, "mem_dma_bd33_7"},
      {0x000b0f30, "mem_tile_control_packet_handler_status"},
      {0x0003f264, "cm_stream_switch_slave_south_1_slot1"},
      {0x0003f260, "cm_stream_switch_slave_south_1_slot0"},
      {0x0003f26c, "cm_stream_switch_slave_south_1_slot3"},
      {0x0003f268, "cm_stream_switch_slave_south_1_slot2"},
      {0x00030140, "cm_core_bmlh1_part1"},
      {0x00030150, "cm_core_bmlh1_part2"},
      {0x00030160, "cm_core_bmlh1_part3"},
      {0x00030170, "cm_core_bmlh1_part4"},
      {0x00034400, "cm_combo_event_inputs"},
      {0x000a06c4, "mem_dma_s2mm_current_write_count_5"},
      {0x000a06c0, "mem_dma_s2mm_current_write_count_4"},
      {0x000c0110, "mem_lock17_value"},
      {0x000a06bc, "mem_dma_s2mm_current_write_count_3"},
      {0x0003f15c, "cm_stream_switch_slave_config_aie_trace"},
      {0x0001f10c, "mm_locks_event_selection_3"},
      {0x10, "shim_lock_step_size"},
      {0x000a06b4, "mem_dma_s2mm_current_write_count_1"},
      {0x000a06b0, "mem_dma_s2mm_current_write_count_0"},
      {0x00034070, "cm_event_broadcast_block_north_set"},
      {0x00034008, "shim_event_generate"},
      {0x000c0030, "uc_memory_zeroization"},
      {0x00032bd0, "cm_core_dc5"},
      {0x00032bc0, "cm_core_dc4"},
      {0x00030770, "cm_core_bmlh7_part4"},
      {0x00032be0, "cm_core_dc6"},
      {0x00032b90, "cm_core_dc1"},
      {0x00030760, "cm_core_bmlh7_part3"},
      {0x00032bb0, "cm_core_dc3"},
      {0x00032ba0, "cm_core_dc2"},
      {0x0000c000, "shim_lock_request"},
      {0x000c0100, "mem_lock16_value"},
      {0x00032440, "cm_core_ldfifoh0_part1"},
      {0x00032460, "cm_core_ldfifoh0_part3"},
      {0x00032450, "cm_core_ldfifoh0_part2"},
      {0x00032470, "cm_core_ldfifoh0_part4"},
      {0x000b0818, "mem_stream_switch_deterministic_merge_arb1_ctrl"},
      {0x000324f0, "cm_core_ldfifoh1_part4"},
      {0x000140fc, "mm_timer_high"},
      {0x000324c0, "cm_core_ldfifoh1_part1"},
      {0x000324d0, "cm_core_ldfifoh1_part2"},
      {0x000324e0, "cm_core_ldfifoh1_part3"},
      {0x000c0360, "mem_lock54_value"},
      {0x8, "shim_dma_s2mm_step_size"},
      {0x000b0014, "uc_mdm_dbg_data"},
      {0x00014500, "mm_event_group_0_enable"},
      {0x0001df1c, "mm_dma_s2mm_current_write_count_1"},
      {0x0001df18, "mm_dma_s2mm_current_write_count_0"},
      {0x000a0628, "mem_dma_s2mm_5_ctrl"},
      {0x000c0310, "mem_lock49_value"},
      {0x00009304, "shim_dma_s2mm_0_task_queue"},
      {0x000a047c, "mem_dma_bd35_7"},
      {0x000940c4, "mem_event_broadcast_b_block_east_clr"},
      {0x000340e0, "cm_trace_event0"},
      {0x000340e4, "cm_trace_event1"},
      {0x00000044, "npi_me_imr1"},
      {0x00032520, "cm_core_stfifol_part3"},
      {0x00031a50, "cm_core_x9_part2"},
      {0x00031a60, "cm_core_x9_part3"},
      {0x00031a70, "cm_core_x9_part4"},
      {0x000a04b4, "mem_dma_bd37_5"},
      {0x000c0050, "mem_lock5_value"},
      {0x00032500, "cm_core_stfifol_part1"},
      {0x00094048, "mem_event_broadcast14"},
      {0x000b0238, "mem_stream_switch_slave_dma_3_slot2"},
      {0x000b023c, "mem_stream_switch_slave_dma_3_slot3"},
      {0x000b0230, "mem_stream_switch_slave_dma_3_slot0"},
      {0x000b0234, "mem_stream_switch_slave_dma_3_slot1"},
      {0x00031900, "cm_core_x4_part1"},
      {0x00031920, "cm_core_x4_part3"},
      {0x00031910, "cm_core_x4_part2"},
      {0x00031930, "cm_core_x4_part4"},
      {0x000a04ac, "mem_dma_bd37_3"},
      {0x000c03d0, "mem_lock61_value"},
      {0x000328b0, "cm_core_r11"},
      {0x000328a0, "cm_core_r10"},
      {0x000328d0, "cm_core_r13"},
      {0x000328c0, "cm_core_r12"},
      {0x000328f0, "cm_core_r15"},
      {0x000328e0, "cm_core_r14"},
      {0x00032910, "cm_core_r17"},
      {0x00032900, "cm_core_r16"},
      {0x00032930, "cm_core_r19"},
      {0x00032920, "cm_core_r18"},
      {0x0003f2f8, "cm_stream_switch_slave_north_0_slot2"},
      {0x0003f2fc, "cm_stream_switch_slave_north_0_slot3"},
      {0x0003f2f0, "cm_stream_switch_slave_north_0_slot0"},
      {0x0003f2f4, "cm_stream_switch_slave_north_0_slot1"},
      {0x000306f0, "cm_core_bmhh6_part4"},
      {0x000306e0, "cm_core_bmhh6_part3"},
      {0x000306d0, "cm_core_bmhh6_part2"},
      {0x000306c0, "cm_core_bmhh6_part1"},
      {0x000092c0, "shim_dma_bd14_8"},
      {0x000092a0, "shim_dma_bd14_0"},
      {0x0003f818, "cm_stream_switch_deterministic_merge_arb1_ctrl"},
      {0x000092a8, "shim_dma_bd14_2"},
      {0x000092ac, "shim_dma_bd14_3"},
      {0x000092b0, "shim_dma_bd14_4"},
      {0x000a06f0, "mem_dma_mm2s_4_constant_pad_value"},
      {0x000092b8, "shim_dma_bd14_6"},
      {0x000092bc, "shim_dma_bd14_7"},
      {0x000b0314, "mem_stream_switch_slave_trace_slot1"},
      {0x000b0310, "mem_stream_switch_slave_trace_slot0"},
      {0x000b031c, "mem_stream_switch_slave_trace_slot3"},
      {0x000b0318, "mem_stream_switch_slave_trace_slot2"},
      {0x000091fc, "shim_dma_bd10_7"},
      {0x000c0120, "mem_lock18_value"},
      {0x000a04fc, "mem_dma_bd39_7"},
      {0x00094008, "mem_event_generate"},
      {0x00000108, "shim_locks_event_selection_2"},
      {0x0001f0f0, "mm_lock15_value"},
      {0x00030270, "cm_core_bmlh2_part4"},
      {0x00030260, "cm_core_bmlh2_part3"},
      {0x00030250, "cm_core_bmlh2_part2"},
      {0x00030240, "cm_core_bmlh2_part1"},
      {0x00031820, "cm_core_x0_part3"},
      {0x0003f004, "cm_stream_switch_master_config_dma0"},
      {0x0003f008, "cm_stream_switch_master_config_dma1"},
      {0x000090e0, "shim_dma_bd4_8"},
      {0x000090d0, "shim_dma_bd4_4"},
      {0x000090d4, "shim_dma_bd4_5"},
      {0x000090d8, "shim_dma_bd4_6"},
      {0x000090dc, "shim_dma_bd4_7"},
      {0x000090c0, "shim_dma_bd4_0"},
      {0x000090c4, "shim_dma_bd4_1"},
      {0x000090c8, "shim_dma_bd4_2"},
      {0x000140f4, "mm_timer_trig_event_high_value"},
      {0x00009080, "shim_dma_bd2_8"},
      {0x00009078, "shim_dma_bd2_6"},
      {0x0000907c, "shim_dma_bd2_7"},
      {0x00009070, "shim_dma_bd2_4"},
      {0x00009074, "shim_dma_bd2_5"},
      {0x00009068, "shim_dma_bd2_2"},
      {0x0000906c, "shim_dma_bd2_3"},
      {0x00009060, "shim_dma_bd2_0"},
      {0x000b0118, "mem_stream_switch_slave_config_tile_ctrl"},
      {0x000c0034, "uc_memory_privileged"},
      {0x00000060, "shim_lock6_value"},
      {0x0003f384, "cm_stream_switch_slave_mem_trace_slot1"},
      {0x0003f380, "cm_stream_switch_slave_mem_trace_slot0"},
      {0x0003f38c, "cm_stream_switch_slave_mem_trace_slot3"},
      {0x0003f388, "cm_stream_switch_slave_mem_trace_slot2"},
      {0x000a0644, "mem_dma_mm2s_2_start_queue"},
      {0x0003f114, "cm_stream_switch_slave_config_south_0"},
      {0x0003f120, "cm_stream_switch_slave_config_south_3"},
      {0x0003f11c, "cm_stream_switch_slave_config_south_2"},
      {0x000a0348, "mem_dma_bd26_2"},
      {0x000a034c, "mem_dma_bd26_3"},
      {0x000a0340, "mem_dma_bd26_0"},
      {0x000a0344, "mem_dma_bd26_1"},
      {0x000a0358, "mem_dma_bd26_6"},
      {0x000a035c, "mem_dma_bd26_7"},
      {0x000a0350, "mem_dma_bd26_4"},
      {0x000a0354, "mem_dma_bd26_5"},
      {0x000a0310, "mem_dma_bd24_4"},
      {0x000a0314, "mem_dma_bd24_5"},
      {0x000a0318, "mem_dma_bd24_6"},
      {0x000a031c, "mem_dma_bd24_7"},
      {0x000a0300, "mem_dma_bd24_0"},
      {0x000a0304, "mem_dma_bd24_1"},
      {0x000a0308, "mem_dma_bd24_2"},
      {0x000a030c, "mem_dma_bd24_3"},
      {0x00009140, "shim_dma_bd6_8"},
      {0x000a02d8, "mem_dma_bd22_6"},
      {0x000a02dc, "mem_dma_bd22_7"},
      {0x000a02d0, "mem_dma_bd22_4"},
      {0x000a02d4, "mem_dma_bd22_5"},
      {0x000a02c8, "mem_dma_bd22_2"},
      {0x000a02cc, "mem_dma_bd22_3"},
      {0x000a02c0, "mem_dma_bd22_0"},
      {0x000a02c4, "mem_dma_bd22_1"},
      {0x0001f0e0, "mm_lock14_value"},
      {0x00032a40, "cm_core_m4"},
      {0x00032a50, "cm_core_m5"},
      {0x00032a60, "cm_core_m6"},
      {0x00032a70, "cm_core_m7"},
      {0x00032a00, "cm_core_m0"},
      {0x00032a10, "cm_core_m1"},
      {0x00032a20, "cm_core_m2"},
      {0x00032a30, "cm_core_m3"},
      {0x0000927c, "shim_dma_bd13_3"},
      {0x00009278, "shim_dma_bd13_2"},
      {0x00009274, "shim_dma_bd13_1"},
      {0x00009270, "shim_dma_bd13_0"},
      {0x0000928c, "shim_dma_bd13_7"},
      {0x00009288, "shim_dma_bd13_6"},
      {0x000c03a0, "mem_lock58_value"},
      {0x00009280, "shim_dma_bd13_4"},
      {0x000a060c, "mem_dma_s2mm_1_start_queue"},
      {0x000307b0, "cm_core_bmhl7_part4"},
      {0x00030790, "cm_core_bmhl7_part2"},
      {0x000307a0, "cm_core_bmhl7_part3"},
      {0x00030780, "cm_core_bmhl7_part1"},
      {0x00009224, "shim_dma_bd11_5"},
      {0x000c0280, "mem_lock40_value"},
      {0x0000922c, "shim_dma_bd11_7"},
      {0x00009228, "shim_dma_bd11_6"},
      {0x00009214, "shim_dma_bd11_1"},
      {0x00009210, "shim_dma_bd11_0"},
      {0x0000921c, "shim_dma_bd11_3"},
      {0x00009218, "shim_dma_bd11_2"},
      {0x000091b4, "shim_dma_bd9_1"},
      {0x000091b0, "shim_dma_bd9_0"},
      {0x000091bc, "shim_dma_bd9_3"},
      {0x000091b8, "shim_dma_bd9_2"},
      {0x000091c4, "shim_dma_bd9_5"},
      {0x000091c0, "shim_dma_bd9_4"},
      {0x000091cc, "shim_dma_bd9_7"},
      {0x0003800c, "cm_reset_event"},
      {0x000091d0, "shim_dma_bd9_8"},
      {0x00009050, "shim_dma_bd1_8"},
      {0x00034064, "cm_event_broadcast_block_west_clr"},
      {0x00009034, "shim_dma_bd1_1"},
      {0x00009030, "shim_dma_bd1_0"},
      {0x0000903c, "shim_dma_bd1_3"},
      {0x00009038, "shim_dma_bd1_2"},
      {0x00009044, "shim_dma_bd1_5"},
      {0x00009040, "shim_dma_bd1_4"},
      {0x0000904c, "shim_dma_bd1_7"},
      {0x00009048, "shim_dma_bd1_6"},
      {0x0000100c, "shim_interrupt_controller_2nd_level_status"},
      {0x0001f070, "mm_lock7_value"},
      {0x0001f0c0, "mm_lock12_value"},
      {0x000090f4, "shim_dma_bd5_1"},
      {0x000b0250, "mem_stream_switch_slave_dma_5_slot0"},
      {0x000b0254, "mem_stream_switch_slave_dma_5_slot1"},
      {0x000b0258, "mem_stream_switch_slave_dma_5_slot2"},
      {0x000b025c, "mem_stream_switch_slave_dma_5_slot3"},
      {0x00002100, "shim_me_aximm_config"},
      {0x000090f0, "shim_dma_bd5_0"},
      {0x0003f808, "cm_stream_switch_deterministic_merge_arb0_ctrl"},
      {0x00091008, "mem_performance_control2"},
      {0x0009100c, "mem_performance_control3"},
      {0x00091000, "mem_performance_control0"},
      {0x00091004, "mem_performance_control1"},
      {0x00000038, "npi_me_imr0"},
      {0x00094090, "mem_event_broadcast_b_block_south_set"},
      {0x00091010, "mem_performance_control4"},
      {0x0000005c, "npi_me_imr3"},
      {0x000c0300, "mem_lock48_value"},
      {0x000340d8, "cm_trace_status"},
      {0x00091080, "mem_performance_counter0_event_value"},
      {0x00030070, "cm_core_bmlh0_part4"},
      {0x00030040, "cm_core_bmlh0_part1"},
      {0x00030060, "cm_core_bmlh0_part3"},
      {0x000140f8, "mm_timer_low"},
      {0x000c00e0, "mem_lock14_value"},
      {0x00031840, "cm_core_x1_part1"},
      {0x00031850, "cm_core_x1_part2"},
      {0x00031860, "cm_core_x1_part3"},
      {0x00031870, "cm_core_x1_part4"},
      {0x000940f4, "mem_timer_trig_event_high_value"},
      {0x000b0038, "mem_stream_switch_master_config_north3"},
      {0x000b0034, "mem_stream_switch_master_config_north2"},
      {0x000318b0, "cm_core_x2_part4"},
      {0x000318a0, "cm_core_x2_part3"},
      {0x0003f284, "cm_stream_switch_slave_south_3_slot1"},
      {0x00031880, "cm_core_x2_part1"},
      {0x00096048, "mem_memory_control"},
      {0x000a0640, "mem_dma_mm2s_2_ctrl"},
      {0x00034008, "cm_event_generate"},
      {0x000c00b0, "mem_lock11_value"},
      {0x000c0090, "mem_lock9_value"},
      {0x00031af0, "cm_core_x11_part4"},
      {0x00031ac0, "cm_core_x11_part1"},
      {0x00031ad0, "cm_core_x11_part2"},
      {0x00031ae0, "cm_core_x11_part3"},
      {0x000a03a4, "mem_dma_bd29_1"},
      {0x000a03a0, "mem_dma_bd29_0"},
      {0x000a03ac, "mem_dma_bd29_3"},
      {0x000a03a8, "mem_dma_bd29_2"},
      {0x000a03b4, "mem_dma_bd29_5"},
      {0x000a03b0, "mem_dma_bd29_4"},
      {0x000a03bc, "mem_dma_bd29_7"},
      {0x000a03b8, "mem_dma_bd29_6"},
      {0x000340f0, "shim_timer_trig_event_low_value"},
      {0x0003f280, "cm_stream_switch_slave_south_3_slot0"},
      {0x000a02a0, "mem_dma_bd21_0"},
      {0x000a02ac, "mem_dma_bd21_3"},
      {0x000a02a8, "mem_dma_bd21_2"},
      {0x000a02b4, "mem_dma_bd21_5"},
      {0x000a02b0, "mem_dma_bd21_4"},
      {0x000a02bc, "mem_dma_bd21_7"},
      {0x000a02b8, "mem_dma_bd21_6"},
      {0x00036070, "cm_memory_control"},
      {0x000b02cc, "mem_stream_switch_slave_south_5_slot3"},
      {0x000b02c8, "mem_stream_switch_slave_south_5_slot2"},
      {0x000b02c4, "mem_stream_switch_slave_south_5_slot1"},
      {0x000b02c0, "mem_stream_switch_slave_south_5_slot0"},
      {0x000319f0, "cm_core_x7_part4"},
      {0x00094404, "mem_combo_event_control"},
      {0x000319e0, "cm_core_x7_part3"},
      {0x000319c0, "cm_core_x7_part1"},
      {0x00094074, "mem_event_broadcast_a_block_north_clr"},
      {0x00032590, "cm_core_fifoxtra_part2"},
      {0x000325a0, "cm_core_fifoxtra_part3"},
      {0x00032580, "cm_core_fifoxtra_part1"},
      {0x000325b0, "cm_core_fifoxtra_part4"},
      {0x0000006c, "npi_me_ior"},
      {0x00038110, "cm_ecc_scrubbing_event"},
      {0x00034064, "shim_event_broadcast_a_block_west_clr"},
      {0x00032d40, "cm_core_ls"},
      {0x00032d30, "cm_core_lr"},
      {0x00032d60, "cm_core_lc"},
      {0x00032d50, "cm_core_le"},
      {0x000a0658, "mem_dma_mm2s_5_ctrl"},
      {0x000b0138, "mem_stream_switch_slave_config_north_1"},
      {0x000b0134, "mem_stream_switch_slave_config_north_0"},
      {0x000b0140, "mem_stream_switch_slave_config_north_3"},
      {0x000b013c, "mem_stream_switch_slave_config_north_2"},
      {0x00032d00, "cm_program_counter"},
      {0x00009328, "shim_dma_mm2s_status_0"},
      {0x0000932c, "shim_dma_mm2s_status_1"},
      {0x000b5480, "uc_mdm_pccmdr"},
      {0x000a0634, "mem_dma_mm2s_0_start_queue"},
      {0x0001de04, "mm_dma_s2mm_0_start_queue"},
      {0x000340fc, "shim_timer_high"},
      {0x000c0000, "uc_core_status"},
      {0x000a06a0, "mem_dma_event_channel_selection"},
      {0x00034520, "cm_event_group_user_event_enable"},
      {0x000c0104, "uc_dma_dm2mm_control"}
    };
  }
};

} // end XDP namespace

#endif
